﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AssetFileDataHolder.h,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This files defines some structures that hold
 *  the data provided by an asset file created by
 *  Tutorial02 as well as a class that holds lists
 *  of each section type that made up an single
 *  asset file.
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <vector>
#include <string>

#include "Model.h"
#include "OutputFileHeaders.h"
#include "ShaderTypeEnum.h"
#include "MemoryPool.h"

class TextureIDManager;

/*
 * NVNProgramData
 * --------------
 * Holds the NVNprogram object created from
 * the compiled shader binary as well as the
 * shader metadata provided by the asset file.
 */
class NVNProgramData
{
    public:
        void Initialize();
        void SetName(const char* pName, int length);
        void Finalize();

        NVNprogram              m_Program;
        int                     m_ShaderStages;
        char*                   m_pProgramName;
        ShaderTypes::ShaderType m_ShaderType;

        std::vector<NVNbuffer*> m_ShaderBuffers;
        MemoryPool              m_ShaderMemoryPool;
};

/*
 * NVNModelData
 * ------------
 * Holds the vertex buffer and index buffer needed
 * to draw the model as well as the additional data
 * from the asset file.
 */
class NVNModelData
{
    public:
        void Initialize();
        void SetName(const char* pName, int length);
        void Finalize();

        void AddVertexAttribute(const VertexAttributeHeader& header);

        void SetupAttributeStatesNVN(int32_t (*GetLocation)(const std::string&));

        Model                 m_Model;
        char*                 m_pModelName;
        NVNbuffer             m_VertexBuffer;
        NVNbuffer             m_IndexBuffer;
        uint32_t              m_VertexBufferSize;
        MemoryPool            m_BufferMemoryPool;
        std::vector<uint32_t> m_VertexAttributeBufferOffsets;

        std::vector<NVNvertexAttribState> m_VertexAttributeStates;
        std::vector<NVNvertexStreamState> m_VertexStreamStates;
};

/*
 * NVNTexureData
 * -------------
 * Holds the NVNtexture created from the packaged
 * texture data as well as the data describing the
 * texture.
 */
class NVNTextureData
{
    public:
        void Initialize();
        void SetName(const char* pName, int length);
        void Finalize();

        NVNtexture       m_Texture;
        NVNsampler       m_Sampler;
        NVNtextureHandle m_TextureHandle;
        uint32_t         m_TextureID;
        uint32_t         m_SamplerID;
        char*            m_pTextureName;
        uint64_t         m_TextureDataSize;

        int              m_GpuVersion;
        uint32_t         m_Alignment;
        uint32_t         m_Width;
        uint32_t         m_Height;
        uint32_t         m_Depth;
        NVNtextureTarget m_NvnTextureTarget;
        NVNformat        m_NvnFormat;
        uint32_t         m_MipLevels;

        MemoryPool       m_TextureMemoryPool;
};

/*
 * AssetFileDataHolder
 * -------------------
 * This class holds a list of program data,
 * model data, and texture data structures from
 * a given asset file.
 */
class AssetFileDataHolder
{
    public:
        AssetFileDataHolder();
        ~AssetFileDataHolder();

        void Finalize();

        void AddProgramData(NVNProgramData* pProgramData);
        void AddModelData(NVNModelData* pModelData);
        void AddTextureData(NVNTextureData* pTextureData);

        void SetupAttributeStatesNVN(int32_t(*GetLocation)(const std::string&));
        void SetupTextureSamplerHandle(NVNdevice* pDevice, TextureIDManager* pTextureIDManager, NVNsamplerBuilder* pSamplerBuilder);


        std::vector<NVNProgramData*>&   GetProgramData();
        std::vector<NVNModelData*>&     GetModelData();
        std::vector<NVNTextureData*>&   GetTextureData();

    private:
        std::vector<NVNProgramData*>    m_ProgramData;
        std::vector<NVNModelData*>      m_ModelData;
        std::vector<NVNTextureData*>    m_TextureData;
};
