﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nnt { namespace ldn
{
    /**
     * @brief       テストの設定です。
     */
    struct TestConfig
    {
        //! アクセスポイントを含むノードの数です。
        int8_t nodeCount;

        //! アクセスポイントを含むノードのインデックスです。
        int8_t nodeIndex;

        //! 無線チャンネルです。既定値は nn::ldn::AutoChannel です。
        int16_t channel;

        //! シーン識別子です。
        uint16_t sceneId;

        //! パケットの送信レート (packets/second) です。規定値は 30 です。
        int16_t packetRate;

        //! 通信の暗号化の有無です。既定値は false です。
        bool isEncrypted;

        //! ローカル通信の動作モードです。
        int8_t operationMode;

        //! エイジング時間を秒単位で指定します。既定値は 1 年 (31,536,000) です。
        int32_t duration;

        //! テストのシード値です。指定しなかった場合はランダムです。
        int32_t seed;

        //! テスト成功・失敗の閾値（パーセント）です。
        float threshold;

        //! ユニキャストの送信レート (packets/second) です。規定値は 8 です。
        int16_t unicastRate;

        //! ブロードキャストの送信レート (packets/seconds) です。既定値は 60 です。
        int16_t broadcastRate;

        //! ペイロードサイズ (Bytes) です。既定値は 256 です。
        uint16_t payloadSize;

        //! 無線コントローラーの制限です。既定値は有効です。
        int8_t wirelessControllerRestriction;

        NN_PADDING1;
    };

    /**
     * @brief       コマンドライン引数で設定可能にするパラメータを指定するフラグです。
     */
    enum CommandLineOptionFlag
    {
        //! アクセスポイントを含むノードの数です。
        CommandLineOptionFlag_NodeCount = 1 << 0,

        //! アクセスポイントを含むノードのインデックスです。
        CommandLineOptionFlag_NodeIndex = 1 << 1,

        //! 無線チャンネルです。
        CommandLineOptionFlag_Channel = 1 << 2,

        //! シーン識別子です。
        CommandLineOptionFlag_SceneId = 1 << 3,

        //! パケットの送信レートです。
        CommandLineOptionFlag_PacketRate = 1 << 4,

        //! 通信の暗号化の有無です。
        CommandLineOptionFlag_Encryption = 1 << 5,

        //! エイジング時間です。
        CommandLineOptionFlag_Duration = 1 << 6,

        //! テストのシード値です。
        CommandLineOptionFlag_Seed = 1 << 7,

        //! テストの成功・失敗の判定に使用する閾値です。
        CommandLineOptionFlag_Threshold = 1 << 8,

        //! ユニキャストの送信レートです。
        CommandLineOptionFlag_UnicastRate = 1 << 9,

        //! ブロードキャストの送信レートです。
        CommandLineOptionFlag_BroadcastRate = 1 << 10,

        //! ペイロードサイズです。
        CommandLineOptionFlag_PayloadSize = 1 << 11,

        //! ローカル通信の動作モードです。
        CommandLineOptionFlag_OperationMode = 1 << 12,

        //! 無線コントローラーの接続制限です。
        CommandLineOptionFlag_WirelessControllerRestriction = 1 << 13,
    };

    /**
     * @brief       CommandLineParser の設定です。
     */
    struct CommandLineParserSetting
    {
        //! コマンド引数で設定可能な項目を CommandLineOptionFlag の論理和で指定します。
        nn::Bit32 flag;

        //! アクセスポイントを含むノードの最小数です。
        int8_t nodeCountMin;

        //! アクセスポイントを含むノードの最大数です。
        int8_t nodeCountMax;

        //! デフォルトのシーン識別子です。
        uint16_t defaultSceneId;
    };

    /**
     * @brief       コマンドライン引数を解析し、テスト設定を生成します。
     * @param[out]  pOutConfig      テスト設定の出力先です。
     * @param[in]   setting         CommandLineParser の設定です。
     * @param[in]   argc            コマンドライン引数の数です。
     * @param[in]   argv            コマンドライン引数です。
     */
    void Parse(TestConfig* pOutConfig, const CommandLineParserSetting& setting,
               int argc, char* argv[]) NN_NOEXCEPT;

}} // namespace nnt::ldn
