﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>

namespace nnt{ namespace graphics{

    // 固定長のバッファでパスを扱うクラスです。
    // C 風のインタフェースと C++ 風のインタフェースの両方を提供します。
    class Path
    {
    public:
        // 格納できるパスの最大の終端文字を含まない長さです。
        static const int MaxLength = 768;
        // data の長さです。 MaxLength の長さのパスと終端文字を記録できる長さです。
        static const int Capacity = MaxLength + 1;

    private:
        // 格納されているパスの文字列です。
        char m_Data[Capacity];

    public:
        // 空パスを作成します。
        Path() NN_NOEXCEPT;
        // 書式指定で作られる文字列からパスを作成します。
        // パスの最大文字数に収まりきらなかった文字は無視されます。
        explicit Path(const char* format, ...) NN_NOEXCEPT;

        // 文字列からパスを設定します。
        // 成功した場合 true を返します。
        // src の長さが MaxLength よりも大きい場合、このインスタンスの値は変更されずに false を返します。
        bool SetString(const char* src) NN_NOEXCEPT;

        // 設定されているパスの文字列を取得します。
        const char* GetString() const NN_NOEXCEPT;

        // パスの長さを返します。
        int GetLength() const NN_NOEXCEPT;
        // パスが空かを返します。
        bool IsEmpty() const NN_NOEXCEPT;

        // パスが等しいかを返します。
        bool IsEqual(const Path* path) const NN_NOEXCEPT;
        // パスが等しいかを返します。
        bool IsEqual(const Path& path) const NN_NOEXCEPT;

        // 絶対パスかを返します。
        // 以下の場合 true を返します。
        // - パスがディレクトリの区切り文字で始まっている
        // - パスがマウント名で始まっている
        bool IsAbsolute() const NN_NOEXCEPT;
        // マウント名を持っているかを返します。
        bool HasMountName() const NN_NOEXCEPT;

        // ルートを指しているかを返します。
        // 以下のすべてを満たしているとき true を返します。
        // - パスが空でない
        // - マウント名を除いたパスが空またはディレクトリの区切り文字 1 つだけで構成されている
        bool IsRoot() const NN_NOEXCEPT;
        // このインスタンスのルートパスを作成します。
        // 絶対パスでない場合、パスを変更せずに false を返します。
        bool MakeRoot() NN_NOEXCEPT;
        // ルートパスを取得します。
        // 相対パスの場合、false を返します。
        bool GetRoot(Path* pOutValue) const NN_NOEXCEPT;
        // ルートパスを取得します。
        // 相対パスの場合、空パスを返します。
        Path GetRoot() const NN_NOEXCEPT;

        // このインスタンスのファイル名を削除して親ディレクトリのパスを作成します。
        // 親ディレクトリが存在しない場合、パスを変更せずに false を返します。
        bool MakeParent() NN_NOEXCEPT;
        // 親ディレクトリのパスを取得します。
        // 親ディレクトリが存在しない場合、false を返し、このインスタンスのパスが取得されます。
        bool GetParent(Path* pOutValue) const NN_NOEXCEPT;
        // 親ディレクトリのパスを取得します。
        // 親ディレクトリが存在しない場合、空パスを返します。
        Path GetParent() const NN_NOEXCEPT;

        // ファイル名を取得します。
        // 成功した場合、true を返します。
        // ファイル名が含まれていない場合 false を返し、空パスが取得されます。
        bool GetFilename(Path* pOutValue) const NN_NOEXCEPT;
        // ファイル名を取得します。
        // ファイル名が含まれていない場合 空パスを返します。
        Path GetFilename() const NN_NOEXCEPT;

        // 拡張子が一致するかを返します。
        bool IsEqualExtension(const char* extension, bool ignoreCase) const NN_NOEXCEPT;
        // このインスタンスから拡張子を削除します。
        void RemoveExtension() NN_NOEXCEPT;

        // パスを正規化します。
        void NormalizeAssign() NN_NOEXCEPT;

        // 正規化したパスを取得します。
        void Normalize(Path* pOutValue) const NN_NOEXCEPT;
        // 正規化したパスを取得します。
        Path Normalize() const NN_NOEXCEPT;


        // このインスタンスに与えられた相対パスを連結します。
        // 成功した場合 true を返します。
        // path が絶対パスであるか、連結した長さが MaxLength を超える場合、インスタンスの値を変更せずに false を返します。
        bool CombineAssign(const Path* relativePath) NN_NOEXCEPT;
        bool CombineAssign(const Path& relativePath) NN_NOEXCEPT;
        bool CombineAssign(const char* relativePath) NN_NOEXCEPT;

        // 与えられた相対パスを連結したパスを取得します。
        // 失敗した場合、false を返します。
        bool Combine(Path* pOutValue, const Path* relativePath) const NN_NOEXCEPT;
        bool Combine(Path* pOutValue, const char* relativePath) const NN_NOEXCEPT;
        // 与えられた相対パスを連結したパスを取得します。
        // 失敗した場合、空パスを返します。
        Path Combine(const Path& relativePath) const NN_NOEXCEPT;
        Path Combine(const char* relativePath) const NN_NOEXCEPT;

        // ディレクトリの区切り文字を取得します。
        // プライマリな区切り文字である '/' を返します。
        static char GetDelimiter() NN_NOEXCEPT;
        // 与えられた文字がディレクトリの区切り文字かを返します。
        static bool IsDelimiter(char c) NN_NOEXCEPT;
        // 与えられた文字がマウント名の区切り文字かを返します。
        static bool IsMountNameDelimiter(char c) NN_NOEXCEPT;

    private:
        // ファイル名の開始位置を返します。
        // GetString() で返される文字列中のファイル名の開始位置を返します。
        // ファイル名が含まれていない場合、終端文字の位置が返されます。
        int GetFilenameStartPosition() const NN_NOEXCEPT;
        // 拡張子の開始位置を返します。
        // GetString() で返される文字列中の拡張子の '.' のインデックスを返します。
        // 拡張子が見つからなかった場合、 -1 を返します。
        int GetExtensionStartPosition() const NN_NOEXCEPT;
        // マウント名の長さを返します。
        // マウント名の区切り文字まで含めた "MoutName:" の長さを返します。
        // マウント名がない場合、 -1 を返します。
        int GetMountNameLength() const NN_NOEXCEPT;
        // ルートパスの長さを返します。
        // マウント名の長さに区切り文字の長さを足した長さを返します。
        // 相対パスの場合、 -1 を返します。
        int GetRootLength() const NN_NOEXCEPT;
    };

}}
