﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <algorithm>
#include <random>
#include <memory>

#include <nn/fs.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/fs/fs_Result.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/os/os_TickTypes.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/oe.h>
#include <nnt/fsUtil/testFs_util.h>

namespace nnt { namespace fs { namespace util {

/**
* @brief 時間計測に用いるクラスです。
*/
class TimeCount
{
protected:
    nn::os::Tick tickBegin;
    nn::os::Tick tickEnd;
    nn::os::Tick tickTotal;
    nn::os::Tick tickMax;
    static const int64_t MaxValue = 0x7fffffffffffffff;

public:
    TimeCount() NN_NOEXCEPT:tickBegin(), tickEnd(), tickTotal(), tickMax(nn::os::Tick(this->MaxValue))
    {
    }

    //!< @brief tickBegin に呼ばれた際の Tick を代入します。
    void StartTime() NN_NOEXCEPT
    {
        this->tickBegin = nn::os::GetSystemTick();
    }

    //!< @brief StartTime から StopTime の差を tickEnd に代入し、tickTotal に累計します。
    void StopTime() NN_NOEXCEPT
    {
        if(this->tickBegin != nn::os::Tick(0))
        {
            nn::os::Tick tickCurrent = nn::os::GetSystemTick();

            if(this->tickBegin > tickCurrent)
            {
                this->tickEnd = this->tickMax - this->tickBegin + tickCurrent;
            }
            else
            {
                this->tickEnd = tickCurrent - this->tickBegin;
            }
            this->tickTotal += this->tickEnd;
        }
    }

    //!< @brief tickBegin, tickEnd, tickTotal の値を 0 で初期化します。
    void ResetTime() NN_NOEXCEPT
    {
        this->tickBegin = nn::os::Tick(0);
        this->tickEnd = nn::os::Tick(0);
        this->tickTotal = nn::os::Tick(0);
    }

    //!< @brief tickEnd の値をナノ秒の単位で返します。
    int64_t GetEndTime() NN_NOEXCEPT
    {
        return nn::os::ConvertToTimeSpan( this->tickEnd ).GetNanoSeconds();
    }

    //!< @brief tickTotal の値をナノ秒の単位で返します。
    int64_t GetTotalTime() NN_NOEXCEPT
    {
        return nn::os::ConvertToTimeSpan( this->tickTotal ).GetNanoSeconds();
    }

    /**
    * @brief 与えられた値を [h]:[m]:[s] [m]:[u]:[n] の形式でログ出力します。
    * @param[in] srcTime 表示する時間
    */
    void ViewTime(const int64_t srcTime) NN_NOEXCEPT
    {
        //                   [h]:[m]:[s]  [m]:[u]:[n]
        NN_LOG("  Time = %02lld:%02lld:%02lld %03lld:%03lld:%03lld[ns] \n"
                            , srcTime / 1000 / 1000 / 1000 / 60 / 60
                            , srcTime / 1000 / 1000 / 1000 / 60 % 60
                            , srcTime / 1000 / 1000 / 1000 % 60
                            , srcTime / 1000 / 1000 % 1000
                            , srcTime / 1000 % 1000
                            , srcTime % 1000);
    }

    /**
    * @brief 与えられた値を 0.000000 の形式でログ出力します。
    * @param[in] srcTime 表示する時間
    */
    void ViewSimpleTime(const int64_t srcTime) NN_NOEXCEPT
    {
        NN_LOG("%4lld.%06lld\n"
               , srcTime / 1000000000
               , (srcTime / 1000) % 1000000);
    }
};

/**
* @brief 与えられた値の倍数でのランダムな値を配列に設定します。
* @param[in/out] array[]    オフセット値格納用の配列
* @param[in] arrayCount     ファイルの分割数
* @param[in]     unitSize   バッファサイズ
*/
void CreateRandomArray(int array[], const int arrayCount, const int unitSize) NN_NOEXCEPT;

/**
* @brief パフォーマンス計測用の構造体です。
*/
struct PerformanceResult
{
    int bufferSize;
    int64_t totalSize;
    int64_t elapsedTimeUsec;
    Vector<PerformanceResult> extensionValue;
public:
    PerformanceResult(int bufferSize, int64_t totalSize, int64_t elapsedTimeUsec)
        : bufferSize(bufferSize),
        totalSize(totalSize),
        elapsedTimeUsec(elapsedTimeUsec)
    {
    }
};

/**
* @brief パフォーマンス計測の構造体オブジェクト vector<PerformanceResult> を push_back します。
* @brief 構造体メンバー bufferSize が一致する要素がある場合は、その中の extensionValue を push_back し、
* @brief なければ、パフォーマンス計測の構造体オブジェクト自身を push_back します。
*/
void AddPerformanceResults(Vector<PerformanceResult>& perfObject, const int bufferSize, const int64_t totalSize, const int64_t elapsedTimeUsec) NN_NOEXCEPT;

/**
* @brief マルチスレッド用
* @brief パフォーマンス計測の構造体オブジェクト vector<PerformanceResult> を push_back 用関数。
* @brief elapsedTimeUsec 値の大きい方のみを保持します。
* @brief
*/
void AddPerformanceResultsMultiThread(Vector<PerformanceResult>& perfObject, const int bufferSize, const int64_t totalSize, const int64_t elapsedTimeUsec, const int threadCount) NN_NOEXCEPT;

/**
* @brief パフォーマンス計測結果を TeamCity が解釈できるフォーマットで書き込みます。
* @brief シーケンシャル、ランダムアクセスの順で実行するテスト用です。
* @param[in] prefix     ログに表示する名前
* @param[in] results    パフォーマンス計測データ
*/
void WriteTeamCityPerformanceResults(const char* prefix, Vector<PerformanceResult>& results) NN_NOEXCEPT;

/**
* @brief 構造体 PerformanceResult のログを出力します。
* @param[in] results  ログ出力する構造体
*/
void DumpPerformanceResults(Vector<PerformanceResult>& results) NN_NOEXCEPT;

/**
* @brief FsTest が想定しているパフォーマンスコンフィグを設定します。
*/
void SetFsTestPerformanceConfiguration();

void ReadSequentialCore(Vector<PerformanceResult>& perfObject, const String filePath, const int bufferSize, const int64_t fileSize, int64_t offsetDifference) NN_NOEXCEPT;
void ReadShuffleCore(Vector<PerformanceResult>& perfObject, const String filePath, const int bufferSize, const int64_t fileSize, int64_t offsetDifference) NN_NOEXCEPT;

/**
* @brief パッチのパフォーマンステストとしてシーケンシャルリード、ランダムリードの速度を計測します。
* @param[in] fileName           計測で読み込むファイル名（マウント名を含まないパス）
* @param[in] bufferSizeMin      最小バッファサイズ
* @param[in] bufferSizeMax      最大バッファサイズ
* @param[in] divisionCountMax   バッファサイズごとに読み込む最大回数
*/
void PerformanceReadTestForPatchedRom(
    const char* fileName, int bufferSizeMin, int bufferSizeMax, int divisionCountMax) NN_NOEXCEPT;


bool CheckPerformanceResult(const char* prefix, const Vector<PerformanceResult>& results, bool isVsSlower) NN_NOEXCEPT;

nn::fs::PriorityRaw ConvertStringToPriorityRaw(const char* pPriority) NN_NOEXCEPT;

const char* ConvertPriorityRawToString(nn::fs::PriorityRaw priority) NN_NOEXCEPT;

enum class BackgroundAccessThreadFileSystemType
{
    BisFs,
    SdCard
};

enum class BackgroundAccessThreadAccessType
{
    Read,
    Write
};

struct BackgroundAccessThreadParameter
{
    bool isEnabled;
    BackgroundAccessThreadFileSystemType fileSystemType;
    int threadPriority;
    nn::fs::PriorityRaw accessPriority;
    BackgroundAccessThreadAccessType accessType;
    size_t accessSize;
    int64_t storageSize;
    int64_t intervalMilliSeconds;

    void Dump() NN_NOEXCEPT
    {
        NN_LOG(
            "background access thread parameters:\n"
            "  - fs type: %s\n"
            "  - thread priority: %d\n"
            "  - access priority: %s\n"
            "  - access type: %s\n"
            "  - access size: %u\n"
            "  - storage size: %lld\n"
            "  - interval: %lld ms\n",
            this->fileSystemType == BackgroundAccessThreadFileSystemType::BisFs ? "BisFs" : "SdCard",
            this->threadPriority,
            ConvertPriorityRawToString(this->accessPriority),
            this->accessType == BackgroundAccessThreadAccessType::Read ? "Read" : "Write",
            static_cast<unsigned int>(this->accessSize),
            this->storageSize,
            this->intervalMilliSeconds
        );
    }

    const String GetTag() const NN_NOEXCEPT
    {
        String tag = "With";
        tag += ConvertPriorityRawToString(this->accessPriority);
        tag += this->accessType == BackgroundAccessThreadAccessType::Read ? "Read" : "Write";
        return tag;
    }
};

void ParseBackgroundAccessThreadParameter(BackgroundAccessThreadParameter* pParameter, int* pArgc, char** pArgv) NN_NOEXCEPT;

void RunBackgroundAccessThread(const BackgroundAccessThreadParameter& parameter) NN_NOEXCEPT;

void StopBackgroundAccessThread() NN_NOEXCEPT;

}}}
