﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <functional>

#include <nn/nn_Result.h>
#include <nn/nn_Assert.h>
#include <nn/account/account_Types.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_IntUtil.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_SaveDataTransferType.h>
#include <nn/fs/fs_SaveDataTransferVersion2.h>

#include <nnt/fsUtil/testFs_util_function_diag.h>
#include <nnt/fsUtil/testFs_util_function_stl.h>
#include <nnt/result/testResult_Assert.h>

namespace nnt { namespace fs { namespace util {

/**
    @brief 不一致時の詳細ダンプ付きの EXPECT_TRUE(memcmp(pExpected, pActual, size)==0) です。
*/
#define NNT_FS_UTIL_EXPECT_MEMCMPEQ(pExpected, pActual, size) \
    NNT_DETAIL_FS_UTIL_CHECK_MEMCMPEQ(pExpected, pActual, size, EXPECT_TRUE)

/**
    @brief 不一致時の詳細ダンプ付きの ASSERT_TRUE(memcmp(pExpected, pActual, size)==0) です。
*/
#define NNT_FS_UTIL_ASSERT_MEMCMPEQ(pExpected, pActual, size) \
    NNT_DETAIL_FS_UTIL_CHECK_MEMCMPEQ(pExpected, pActual, size, ASSERT_TRUE)

/**
    @brief 不一致時の詳細ダンプ付きの NN_RESULT_THROW_UNLESS(memcmp(pExpected, pActual, size)==0, result) です。
*/
#define NNT_FS_UTIL_RESULT_THROW_MEMCMPEQ(pExpected, pActual, size, result) { \
    if( std::memcmp(pExpected, pActual, size) == 0 ) \
    { \
    }else{ \
        nnt::fs::util::DumpBufferDiff(pExpected, pActual, size); \
        NN_RESULT_THROW_UNLESS(std::memcmp(pExpected, pActual, size) == 0, result); \
    } \
    NN_RESULT_SUCCESS; \
}

// 内部使用
void DumpBufferDiff(const void* expectedBuffer, const void* actualBuffer, const size_t size) NN_NOEXCEPT;

// 内部使用
#define NNT_DETAIL_FS_UTIL_CHECK_MEMCMPEQ(pExpected, pActual, size, checkFunc) { \
    if( std::memcmp(pExpected, pActual, size) == 0 ) \
    { \
    }else{ \
        nnt::fs::util::DumpBufferDiff(pExpected, pActual, size); \
        checkFunc(std::memcmp(pExpected, pActual, size) == 0); \
    } \
}

/**
    @brief condition が満たされない場合にテストをスキップします。
*/
#define NNT_FS_UTIL_SKIP_TEST_UNLESS(condition) \
    if(!(condition)) \
    { \
        nnt::fs::util::OutPutSkipMarker(); \
        NN_LOG("Skipped test since the fs doesn't meet a condition \"%s\".\n", #condition); \
        return; \
    }

// 内部使用
void OutPutSkipMarker() NN_NOEXCEPT;

/**
    @brief 渡されたバッファをログにダンプします。
*/
void DumpBuffer(const void* buffer, const size_t size) NN_NOEXCEPT;

/**
    @brief 渡されたバッファを 4 バイト単位のインクリメンタルな値 (0x00000001, 0x00000002, ...) で埋めます。
    @details バイトオーダーは動作環境に従います。
            埋める値の起点は offsetCount / 4 になります。
*/
void FillBufferWith32BitCount(void* pBuffer, const size_t size, const int64_t offsetCount) NN_NOEXCEPT;

/**
    @brief 渡されたバッファが 4 バイト単位のインクリメンタルな値 (0x00000001, 0x00000002, ...) で埋められている場合に true を返します。
    @pre size, offsetCount は 4 の倍数
    @details バイトオーダーは動作環境に従います。
            埋める値の起点は offsetCount / 4 になります。
*/
bool IsFilledWith32BitCount(const void* pBuffer, const size_t size, const int64_t offsetCount) NN_NOEXCEPT;

/**
    @brief 渡されたバッファが value で埋められている場合に true を返します。
*/
bool IsFilledWithValue(const void* pBuffer, const size_t size, unsigned char value) NN_NOEXCEPT;

/**
    @brief 渡されたバッファを 1 バイト単位のインクリメンタルな値 (0x01, 0x02, 0x03...) で埋めます。
    @details 埋める値の起点は offsetCount になります。
*/
void FillBufferWith8BitCount(void* pBuffer, const size_t size, const int64_t offsetCount) NN_NOEXCEPT;

/**
    @brief 渡されたバッファが 1 バイト単位のインクリメンタルな値 (0x01, 0x02, 0x03...) で埋められている場合に true を返します。
    @details 埋める値の起点は offsetCount になります。
*/
bool IsFilledWith8BitCount(const void* pBuffer, const size_t size, const int64_t offsetCount) NN_NOEXCEPT;

/**
    @brief 渡されたバッファを擬似乱数列で埋めます。
*/
void FillBufferWithRandomValue(void* pBuffer, const size_t size) NN_NOEXCEPT;

/**
    @brief 渡されたバッファを nact のランダムデータファイル生成ルールに基づいた値で埋めます。
*/
void FillBufferWithXorShift(char* path, void* pBuffer, const size_t size) NN_NOEXCEPT;

/**
    @brief 現在時刻を取得します。
*/
void GetDateTime(tm* pdate, time_t* pmsec) NN_NOEXCEPT;

/**
    @brief 現在時刻から "YYMMDD" のフォーマットの文字列を生成します。
*/
String GetNowString() NN_NOEXCEPT;

const int MAX_LENGTH = 1024 * 1024;
/**
    @brief [1, MAX_LENGTH) のランダムな値を返します。
*/
int64_t GetRandomLength() NN_NOEXCEPT;

/**
    @brief 指定した長さのランダムな文字列を返します。
           長さを指定しなかった場合は、GetRandomeLength の値を length として使用します。
*/
String GenerateRandomString(const int64_t length) NN_NOEXCEPT;
String GenerateRandomString() NN_NOEXCEPT;

/**
    @brief 指定した長さ以内のランダムな長さのランダムな文字列を返します。
           GenerateRandomString() とは異なり、エントリ名に使用できない一部の文字は除かれています。
*/
String GenerateRandomLengthEntryName(int64_t maxLength) NN_NOEXCEPT;

/**
    @brief 128 文字以内のランダムな長さのランダムな文字列を返します。
           GenerateRandomString() とは異なり、エントリ名に使用できない一部の文字は除かれています。
*/
String GenerateRandomEntryName() NN_NOEXCEPT;

/**
    @brief 指定した長さのランダムな文字列を返します。
           GenerateRandomString() とは異なり、エントリ名に使用できない一部の文字は除かれています。
*/
String GenerateRandomEntryName(const int64_t length) NN_NOEXCEPT;

/**
    @brief 指定したパスの親ディレクトリパスを返します。
    @pre
        - path が '/' を含む
*/
String GetParentPath(const char* path) NN_NOEXCEPT;

/**
    @brief Host PC 上の環境変数 TEMP or TMP のパスを返します。
           環境変数が未設定の場合は C:\Windows\Temp を返します。
*/
const char* GetHostTemporaryPath() NN_NOEXCEPT;

/**
    @brief pBase にランダムな文字列を付加したパスを返します。。
*/
String CreateUniquePath(const char* pBase) NN_NOEXCEPT;

/**
    @brief value を alignment 単位に切り上げます。
    @pre alignment は 2 のべき乗
*/
uint64_t RoundUp(const uint64_t value, const uint64_t alignment) NN_NOEXCEPT;

/**
    @brief value を alignment 単位に切り下げます。
    @pre alignment は 2 のべき乗
*/
uint64_t RoundDown(const uint64_t value, const uint64_t alignment) NN_NOEXCEPT;

struct Hash
{
    char value[nn::crypto::Sha256Generator::HashSize];
};

/**
    @brief ファイルの SHA256 値を計算します。
*/
nn::Result CalculateFileHash(Hash *outValue, const char* path) NN_NOEXCEPT; // スレッドアンセーフ
nn::Result CalculateFileHash(Hash *outValue, const char* path, void* buffer, const size_t bufferSize) NN_NOEXCEPT; // 大きなワークバッファを渡す場合
nn::Result CalculateFileHash(Hash *outValue, const char* path, void* buffer, const size_t bufferSize, int64_t offset, int64_t size) NN_NOEXCEPT; // ファイルの一部のみを対象とする


/**
    @brief buffer を使ってファイルをコピーします。
*/
nn::Result CopyFile(const char* srcPath, const char* dstPath, void* buffer, const size_t bufferSize) NN_NOEXCEPT;

/**
    @brief 4 バイト単位のインクリメンタルな値で埋められたパターンをファイルに書き込みます。
*/
nn::Result WriteFileWith32BitCount(const char* path, int64_t size, int64_t offsetCount) NN_NOEXCEPT;

/**
    @brief 4 バイト単位のインクリメンタルな値で埋められたファイルを作成します。
*/
nn::Result CreateFileWith32BitCount(const char* path, int64_t size, int64_t offsetCount) NN_NOEXCEPT;

/**
    @brief 1 バイト単位のインクリメンタルな値で埋められたファイルを作成します。
*/
nn::Result CreateFileWith8BitCount(const char* path, int64_t size, int64_t offsetCount) NN_NOEXCEPT;

/**
    @brief 0で埋められたファイルを作成します。
*/
nn::Result CreateFileWithZero(const char* path, int64_t size, int64_t offsetCount) NN_NOEXCEPT;

/**
    @brief 対象ファイルが 4 バイト単位のインクリメンタルな値で埋められている場合に true を返します。
    @pre 対象ファイルが存在する
*/
bool IsFilledWith32BitCount(const char* path, const int64_t offsetCount) NN_NOEXCEPT;


/**
    @brief ディレクトリを深さ優先探索し処理します。
    @detais 各サブディレクトリに入る際に enterDirectory(), 出る際に exitDirectory, 各ファイルに対し findFile() を実行します。
            ResultSuccess() 以外が返された場合は探索を中断します。
            nullptr を指定された場合は何もしません。
*/
nn::Result IterateDirectoryRecursive(const char* path,
                                 std::function<nn::Result(const char* path, const nn::fs::DirectoryEntry& entry)> enterDirectory,
                                 std::function<nn::Result(const char* path, const nn::fs::DirectoryEntry& entry)> exitDirectory,
                                 std::function<nn::Result(const char* path, const nn::fs::DirectoryEntry& entry)> findFile
                                 ) NN_NOEXCEPT;

/**
    @brief ディレクトリを深さ優先探索し、エントリ名、ファイルサイズ、ファイルのハッシュを列挙します。
*/
nn::Result DumpDirectoryRecursive(const char* path, bool continueOnFailure = true) NN_NOEXCEPT;
nn::Result DumpDirectoryRecursive(const char* path, bool continueOnFailure, void* buffer, size_t bufferSize, bool silent = false, bool isBreadthFirst = false) NN_NOEXCEPT;


/**
    @brief ディレクトリを深さ優先探索し、エントリ名、ファイルサイズを列挙します。
*/
nn::Result ListDirectoryRecursive(const char* path) NN_NOEXCEPT;

/**
    @brief ディレクトリを深さ優先探索し、ディレクトリとファイルのフルパスのリストを取得します。
*/
nn::Result ListDirectoryRecursive(Vector<String>* outFileList, Vector<String>* outDirList, const char* path) NN_NOEXCEPT;

/**
@brief ディレクトリを深さ優先探索し、ハッシュを生成します。エントリの列挙順に依存します。ファイルの内容も対象とします。
*/
nn::Result CalculateDirectoryTreeHash(Hash *outValue, const char* path, void* buffer, size_t bufferSize) NN_NOEXCEPT;

/**
@brief ディレクトリを深さ優先探索し、ハッシュを生成します。エントリの列挙順に依存します。ファイルの内容を対象としません。
*/
nn::Result CalculateDirectoryTreeHashLite(Hash *outValue, const char* path) NN_NOEXCEPT;

/**
@brief ディレクトリ内の総ファイルサイズを取得します。
*/
nn::Result GetDirectorySize(int64_t* outValue, const char* path) NN_NOEXCEPT;

/**
    @brief 2 つのディレクトリを深さ優先探索し、ファイルのハッシュを比較します。
*/
void CompareDirectoryRecursive(const char* path1, const char* path2) NN_NOEXCEPT;

/**
@brief 2 つのディレクトリを深さ優先探索し、ファイルのハッシュを比較します。(ログ出力なし)
*/
void CompareDirectoryRecursiveSilently(const char* path1, const char* path2) NN_NOEXCEPT;

/**
    @brief 渡された変数を 0x5A で埋めます。
*/
template <typename T>
void InvalidateVariable(T* pValue) NN_NOEXCEPT
{
    memset(pValue, 0x5A, sizeof(T));
}

/**
    @brief 渡された配列を 0x5A で埋めます。
*/
template <typename T>
void InvalidateVariable(T* pValue, const int count) NN_NOEXCEPT
{
    memset(pValue, 0x5A, sizeof(T) * count);
}


/**
    @brief ゲームカードがアタッチされた状態になるまで待機します。
*/
void WaitGameCardAttach() NN_NOEXCEPT;

/**
    @brief ゲームカードがデタッチされた状態になるまで待機します。
*/
void WaitGameCardDetach() NN_NOEXCEPT;

/**
    @brief 実機：ゲームカードを一度デタッチアタッチするまで待ちます。
           Win ：ゲームカードのデタッチアタッチをエミュレートします。
*/
void DetachAttachGameCard() NN_NOEXCEPT;

/**
@brief 実機：SD カードを一度デタッチアタッチするまで待ちます。
Win ：SD カードのデタッチアタッチをエミュレートします。
*/
void DetachAttachSdCard() NN_NOEXCEPT;

/**
    @brief GetRandomSeed でスレッドごとに呼び出し順に応じて一意な系列を取得できるように、スレッド固有の RandomSeed を初期化します。
    @details 各スレッドで GetRandomSeed の呼び出し前に SetInitialThreadLocalRandomSeed を呼び出しておく必要があります。
*/
void InitializeThreadLocalRandomSeed() NN_NOEXCEPT;

/**
    @brief 呼び出しスレッドの初期 RandomSeed を設定します。
    @details initialSeed が 0 の場合はシードの初期値を時刻から生成します。
*/
void SetInitialThreadLocalRandomSeed(int initialSeed=0) NN_NOEXCEPT;

/**
    @brief 乱数生成クラスに渡すシードを生成します。
    @details InitializeThreadLocalRandomSeed を事前に呼び出した場合は、スレッドごとに呼び出し順に応じて一意な値を返します。
             そうでない場合は、プログラム全体で呼び出し順に応じて一意な値を返します。
             シードの初期値は SetInitialThreadLocalRandomSeed で指定されたもの、もしくは初回呼び出し時に時刻から生成されたものが利用されます。
*/
int GetRandomSeed() NN_NOEXCEPT;

/**
* @brief 乱数生成クラス
*
* 線形合同法を用いて乱数を生成します。
* 過去互換用、今後削除予定
*/
class Random
{
private:
    uint64_t m_x;      //!< 乱数値
    uint64_t m_mul;    //!< 乗数
    uint64_t m_add;    //!< 加算する数

public:
    /**
    * @brief コンストラクタです。
    */
    explicit Random(uint64_t seed = 0)
    {
        m_x = seed;
        m_mul = (1566083941LL << 32) + 1812433253LL;
        m_add = 2531011;
    }

    /**
    * @brief 乱数シードを与えます。
    */
    void SetSeed(uint64_t seed) NN_NOEXCEPT
    {
        m_x = seed;
        m_mul = (1566083941LL << 32) + 1812433253LL;
        m_add = 2531011;
    }

    /**
    * @brief 範囲(0～max-1)を指定して乱数を取得します。
    */
    uint32_t Get32(uint32_t max = 0xFFFFFFFFU) NN_NOEXCEPT
    {
        m_x = m_mul * m_x + m_add;

        // 引数maxが定数ならばコンパイラにより最適化される。
        if( max == 0 )
        {
            return (uint32_t)(m_x >> 32);
        }
        else
        {
            return (uint32_t)(((m_x >> 32) * max) >> 32);
        }
    }

    /*!
    */
    void FillBytes(void *pBuffer0, size_t size, uint32_t max = 0xFFFFFFFFU) NN_NOEXCEPT
    {
        uint8_t *pBuffer = static_cast<uint8_t*>(pBuffer0);
        uint8_t *pBufferEnd = pBuffer + size;

        while (pBuffer < pBufferEnd)
        {
            *pBuffer++ = this->Get32(max) & 0xFF;
        }
    }
};

/**
* @brief AllocateBufferのカスタムデリータ
* @param[in] ptr malloc で確保されたバッファのポインタ
*/
void DeleterBuffer(void* ptr) NN_NOEXCEPT;

/**
* @brief カスタムデリータを用いたローカルのメモリ確保
* @param[in] size malloc で確保するバッファのサイズ
*/
std::unique_ptr<char, decltype(&DeleterBuffer)> AllocateBuffer(size_t size) NN_NOEXCEPT;

std::unique_ptr<char, decltype(&DeleterBuffer)> AllocateAlignedBuffer(size_t size, int alignment) NN_NOEXCEPT;


/**
@brief 与えられたバッファでファイルを作成します。
*/
nn::Result CreateAndWriteFileAtOnce(const char* path, const void* pBuffer, size_t bufferSize) NN_NOEXCEPT;

/**
@brief ファイルを読み込んだバッファを返します。
*/
nn::Result ReadFileAtOnce(size_t* outSize, decltype(AllocateBuffer(0)) * outBuffer, const char* path) NN_NOEXCEPT;


/**
* @brief Write したデータを Read し、データの比較をします。
* @param[in]  pFile       ファイル操作のためのインターフェース
* @param[in]  writeBuffer 書込みデータ
* @param[in]  size        サイズ
* @retval     true        確認 ok の場合に返ります。
* @retval     false       確認 ng の場合に返ります。
* @details                Write、Flush、Read、比較の一連の処理を行うAPI
*/
template<typename FileClass>
void ConfirmWriteRead(FileClass* pFile, uint8_t *writeBuffer, const size_t size) NN_NOEXCEPT;

/**
* @brief 指定のファイル・フォルダの存在確認をします。
* @param[in]  pfilesSystem ファイルシステム操作のためのインターフェース
* @param[in]  pPath         対象パス
* @param[in]  entryType     タイプ(ファイル又は、ディレクトリ)
* @retval     true          存在する場合に返ります。
* @retval     false         存在しない場合に返ります。
* @details                  GetEntryType() をラッピングして、指定したパス・エントリータイプを確認する API。
*/
template<typename FileSystemClass>
bool IsEntryExist(FileSystemClass* pFilesSystem, const char* pPath, const nn::fs::DirectoryEntryType entryType) NN_NOEXCEPT;

/**
* @brief 指定エントリタイプが存在するか確認する関数です
* @param[in]  pFilesSystem           ファイルシステム
* @param[in]  name                   確認するパスを指定します
* @param[in]  entryType              確認するファイル／ディレクトリのエントリタイプを指定します
*
* @return     指定エントリタイプの結果が返ります。
* @retval     true                   指定のエントリタイプで存在した場合に返ります。
* @retval     false                  指定のエントリタイプで存在しない場合に返ります。
*/
template<typename FileSystemClass>
bool IsEntryExist(FileSystemClass* pFilesSystem, const char* pPath, const nn::fs::DirectoryEntryType entryType) NN_NOEXCEPT;

// ファイルパスサイズ定義
const int32_t NameMaxLength = 260;

/**
* @brief テスト対象のファイルを作成します。
* @param[in]  pFileSystem ファイルシステム
* @param[out] pFilePath   ファイルパスの配列（nullptr を指定する場合は、作成のみを行い、パスの配列は返さない。）
* @param[in]  pPath       対象のベースパス
* @param[in]  count       作成数
* @param[in]  size        サイズ
* @details                作成場所(パス)、作成数、サイズを指定し、ファイルの作成処理を行う API
*/
template<typename FileSystemClass>
nn::Result CreateTestFile(FileSystemClass* pFileSystem, char (*pFilePath)[NameMaxLength], const char* pPath, const int count, const int64_t size) NN_NOEXCEPT;

/**
* @brief ディレクトリ構成の種類です。
*/
enum DirType
{
    DIRTYPE_FLAT,
    DIRTYPE_NEST
};

/**
* @brief テスト対象のディレクトリを作成します。
* @param[in]  pFileSystem  ファイルシステム
* @param[out] pDirPath     ディレクトリパスの配列（nullptr を指定する場合は、作成のみを行い、パスの配列は返さない。）
* @param[in]  pPath        対象のベースパス
* @param[in]  count        作成数
* @param[in]  type         構成 (階層構造又は、フラット)
*                            DIRTYPE_FLAT : 同ディレクトリ内にcount数の子ディレクトリ
*                            DIRTYPE_NEST : count階層の子ディレクトリ
* @param[in]  isFileCreate ファイル作成の有無
* @param[in]  size         ファイルサイズ
* @details                 作成場所(パス)、作成数、構成(階層又は、フラット)、ファイル作成の有無・サイズを指定し、フォルダの作成処理を行う API
*/
template<typename FileSystemClass>
nn::Result CreateTestDirectory(FileSystemClass* pFileSystem, char (*pDirPath)[NameMaxLength],
                               const char* pPath, const int count, const int type,
                               const bool isFileCreate, const int size) NN_NOEXCEPT;








nn::Result CreateTestDirectoryTreeRandomly(const char* path, int64_t totalSize, int64_t totalEntryCount);

nn::Result CreateTestDirectoryTreeRandomlyByFileNameOffset(const char* path, int64_t totalSize, int64_t totalEntryCount);

int64_t GetOffsetByFileName(const char* fileName, size_t fileNameLength);
int64_t GetOffsetByFileName(const char* filePath);

const nn::fs::SaveDataId UserSaveDataApplicationId =
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    0x0000000000000000ULL;
#else
    0x0005000C10000000ULL;
#endif

const nn::ncm::ApplicationId ApplicationId =
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    {0x0000000000000000ULL};
#else
    {0x0005000C10000000ULL};
#endif


const nn::ncm::ApplicationId TestApplicationId0 = { 0x0001000C10000000ULL };
const nn::ncm::ApplicationId TestApplicationId1 = { 0x0001000C10000001ULL };

const nn::fs::UserId TestUserId0 = { { 1 } };
const nn::fs::UserId TestUserId1 = { { 2 } };

bool IsTestSaveData(const nn::fs::SaveDataInfo& info) NN_NOEXCEPT;
void ListAllSaveData(nn::fs::SaveDataSpaceId spaceId) NN_NOEXCEPT;
void ListAllSaveData() NN_NOEXCEPT;
void DeleteAllTestSaveData() NN_NOEXCEPT;

/**
* @brief 固定のセーブデータサイズでセーブデータを作成し、マウントします。
*/
nn::Result CreateAndMountSystemSaveData(const char* mountName, nn::fs::SystemSaveDataId id);
nn::Result CreateAndMountSystemSaveData(const char* mountName, nn::fs::SaveDataSpaceId spaceId, nn::fs::SystemSaveDataId id) NN_NOEXCEPT;
nn::Result CreateAndMountSystemSaveData(const char* mountName, nn::fs::SystemSaveDataId id, nn::fs::UserId userId);
nn::Result CreateAndMountSystemSaveData(const char* mountName, nn::fs::SaveDataSpaceId spaceId, nn::fs::SystemSaveDataId id, nn::fs::UserId userId) NN_NOEXCEPT;
nn::Result CreateAndMountSystemSaveData(const char* mountName, nn::fs::SystemSaveDataId id, nn::account::Uid userId) NN_NOEXCEPT;
nn::Result CreateAndMountDeviceSaveData(const char* mountName);
nn::Result CreateAndMountDeviceSaveData(const char* mountName, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;
nn::Result CreateAndMountBcatSaveData(const char* mountName, nn::ncm::ApplicationId applicationId);
nn::Result CreateAndMountSaveData(const char* mountName, nn::fs::UserId userId);
nn::Result CreateAndMountSaveData(const char* mountName, nn::ncm::ApplicationId applicationId, nn::fs::UserId userId) NN_NOEXCEPT;
nn::Result CreateAndMountSaveData(const char* mountName, nn::account::Uid userId) NN_NOEXCEPT;
nn::Result CreateAndMountSaveData(const char* mountName, nn::ApplicationId applicationId, nn::account::Uid userId) NN_NOEXCEPT;
nn::Result CreateAndMountSaveDataReadOnly(const char* mountName, nn::ncm::ApplicationId applicationId, nn::fs::UserId userId) NN_NOEXCEPT;
nn::Result CreateAndMountSaveDataReadOnly(const char* mountName, nn::ApplicationId applicationId, nn::account::Uid userId) NN_NOEXCEPT;
nn::Result CreateAndMountSystemBcatSaveData(const char* mountName, nn::fs::SystemBcatSaveDataId id);
nn::Result CreateAndMountCacheStorage(const char* mountName) NN_NOEXCEPT;
nn::Result CreateAndMountCacheStorage(const char* mountName, int index) NN_NOEXCEPT;
nn::Result CreateAndMountCacheStorage(const char* mountName, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;
nn::Result CreateAndMountCacheStorage(const char* mountName, nn::ncm::ApplicationId applicationId, int index) NN_NOEXCEPT;
nn::Result CreateAndMountTemporaryStorage(const char* mountName) NN_NOEXCEPT;
nn::Result CreateAndMountSaveDataInternalStorage(const char* mountName, nn::fs::SaveDataId id) NN_NOEXCEPT;


/**
* @brief ユーザセーブを作成し、SaveDataInfo を返します。
*/
nn::Result CreateAndFindSaveData(
    nn::fs::SaveDataInfo* pOutValue,
    nn::ncm::ApplicationId applicationId,
    nn::fs::UserId userId,
    int64_t availableSize, int64_t journalSize
);

/**
* @brief TestApplicationId0, TestUserId0, 1MB, 1MB のパラメータでユーザセーブを作成し、SaveDataInfo を返します。
*/
inline nn::Result CreateAndFindSaveData(nn::fs::SaveDataInfo* pOutValue)
{
    return CreateAndFindSaveData(pOutValue, TestApplicationId0, TestUserId0, 1024 * 1024, 1024 * 1024);
}


/**
* @brief 初期サイズでユーザセーブを作成したのち、 extendCount 回の拡張を経て最終サイズまで拡張します。
*/
nn::Result CreateAndExtendSaveData(
    nn::fs::SaveDataInfo* pOutValue,
    nn::ncm::ApplicationId applicationId,
    nn::fs::UserId userId,
    int64_t initialSize, int64_t initialJournalSize,
    int64_t finalSize, int64_t finalJournalSize,
    int extendCount
    );



// VerifyMount で使用する構造体、関数
struct VerifyEntry
{
    char path[nn::fs::EntryNameLengthMax];
    nnt::fs::util::Hash hash;
};

bool operator<(const VerifyEntry& left, const VerifyEntry& right) NN_NOEXCEPT;

nn::Result ListVerifyEntry(Vector<VerifyEntry>* pOutValue, const char* path) NN_NOEXCEPT;

/**
* @brief        マウントしたファイルシステムの内容と、
*               その元となったホスト側ディレクトリの内容が一致するかのテスト
*
* @param[out]   pOutValue               内容が一致していれば true, そうでなければ false
* @param[in]    mountFunc               指定されたマウント名でマウントする関数
* @param[in]    srcHostDirectoryPath    ホスト側ディレクトリのパス
* @param[in]    mountName               マウント名
* @param[in]    directoryPath           比較するディレクトリのパス
*
*/
template<typename MountFunc>
nn::Result VerifyMount(
    bool* pOutValue,
    MountFunc mountFunc,
    const char* srcHostDirectoryPath,
    const char* mountName,
    const char* directoryPath
) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutValue);
    NN_ASSERT_NOT_NULL(srcHostDirectoryPath);
    NN_ASSERT_NOT_NULL(mountName);
    NN_ASSERT_NOT_NULL(directoryPath);

    // 追加コンテンツの元となったディレクトリを列挙
    Vector<VerifyEntry> treeListSourceDirectory;
    {
        // パスの比較を簡単にするためにマウント名を比較対象と同一にする
        NN_RESULT_DO(nn::fs::MountHost(mountName, srcHostDirectoryPath));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(mountName);
        };

        NN_RESULT_DO(ListVerifyEntry(&treeListSourceDirectory, directoryPath));
        NN_RESULT_DO(nnt::fs::util::DumpDirectoryRecursive(directoryPath));
    }

    // マウントしたファイルシステムの内容を列挙
    Vector<VerifyEntry> treeListMountData;
    {
        NN_RESULT_DO(mountFunc(mountName));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(mountName);
        };

        NN_RESULT_DO(ListVerifyEntry(&treeListMountData, directoryPath));
        NN_RESULT_DO(nnt::fs::util::DumpDirectoryRecursive(directoryPath));
    }

    // 出現順は問わない
    std::sort(treeListSourceDirectory.begin(), treeListSourceDirectory.end());
    std::sort(treeListMountData.begin(), treeListMountData.end());

    // ベリファイ
    const int memcmpResult = memcmp(
                                 &treeListMountData[0],
                                 &treeListSourceDirectory[0],
                                 treeListMountData.size() * sizeof(VerifyEntry)
                             );
    if( memcmpResult != 0 )
    {
        for( const VerifyEntry& e : treeListMountData )
        {
            NN_LOG("%s %x\n", e.path, e.hash);
        }
        for( const VerifyEntry& e : treeListSourceDirectory )
        {
            NN_LOG("%s %x\n", e.path, e.hash);
        }
        nnt::fs::util::DumpBufferDiff(
            &treeListMountData[0],
            &treeListSourceDirectory[0],
            treeListMountData.size() * sizeof(VerifyEntry)
        );
    }

    *pOutValue = memcmpResult == 0;
    NN_RESULT_SUCCESS;
}

/**
* @brief ディレクトリを巡回して処理を行います。
*/
template< typename TFunc >
nn::Result TraverseDirectory(const char* path) NN_NOEXCEPT
{
    Vector<nn::fs::DirectoryEntry> entries;

    // ディレクトリ内の全エントリ取得
    {
        nn::fs::DirectoryHandle directory;
        NN_RESULT_DO(nn::fs::OpenDirectory(&directory, path, nn::fs::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory(directory);
        };

        int64_t entryCount = 0;
        NN_RESULT_DO(nn::fs::GetDirectoryEntryCount(&entryCount, directory));
        const bool result = nn::util::IsIntValueRepresentable<size_t, int64_t>(entryCount);
        NN_ABORT_UNLESS(result);

        entries.resize(static_cast<size_t>(entryCount));

        int64_t readCount = 0;
        NN_RESULT_DO(nn::fs::ReadDirectory(&readCount, entries.data(), directory, entryCount));
        NN_ABORT_UNLESS(entryCount == readCount);
    }

    // ファイルの一覧表示＆関数実行
    {
        TFunc func;

        for( const auto& entry : entries )
        {
            if( entry.directoryEntryType == nn::fs::DirectoryEntryType_File )
            {
                String name(path);
                name.append(entry.name);

                NN_LOG("%s - 0x%llX\n", name.c_str(), entry.fileSize);

                NN_RESULT_DO(func(name.c_str(), entry.fileSize));
            }
        }
    }

    // ディレクトリの子階層へ
    for( const auto& entry : entries )
    {
        if( entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory )
        {
            String name(path);
            name.append(entry.name);
            name.push_back('/');

            // TODO: 再帰で処理するがスタックの状況次第で別の方法を考える
            NN_RESULT_DO(TraverseDirectory<TFunc>(name.c_str()));
        }
    }

    NN_RESULT_SUCCESS;
}


// 空き容量埋めるユーティリティ
// @pre path 下に対して CreateFile(), DeleteFile(), GetFreeSpaceSize() が利用可能
class ScopedPadder{
public:
    ScopedPadder(const char* path, int64_t freeSpaceSize)
    {
        int64_t size;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFreeSpaceSize(&size, path));

        int64_t paddingSize = std::max(static_cast<int64_t>(0), size - freeSpaceSize);

        m_FilePath = path + nnt::fs::util::GenerateRandomEntryName();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CreateFile(m_FilePath.c_str(), paddingSize, nn::fs::CreateFileOptionFlag_BigFile));
    }

    ~ScopedPadder()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::DeleteFile(m_FilePath.c_str()));
    }
private:
    nnt::fs::util::String m_FilePath;
};

static const int64_t LargeOffsetList[] = {
    0,
    static_cast<int64_t>(4) * 1024 * 1024 * 1024,
    static_cast<int64_t>(8) * 1024 * 1024 * 1024,
    static_cast<int64_t>(16) * 1024 * 1024 * 1024,
    static_cast<int64_t>(32) * 1024 * 1024 * 1024,
    static_cast<int64_t>(64) * 1024 * 1024 * 1024,
};

static const size_t LargeOffsetListLength = sizeof(LargeOffsetList) / sizeof(int64_t);

static const int64_t LargeOffsetMax = LargeOffsetList[LargeOffsetListLength - 1];

// 4 GB を超えるオフセットで IFile にアクセスする（AllowAppend 無効時）
void TestFileAccessWithLargeOffset(nn::fs::fsa::IFile* pFile, size_t bufferSize) NN_NOEXCEPT;

// IFile の AllowAppend 動作テスト
void TestFileAccessAllowAppend(nn::fs::fsa::IFile* pFile, size_t bufferSize) NN_NOEXCEPT;

// IFile の AllowAppend 無効の動作テスト
void TestFileAccessNotAllowAppend(nn::fs::fsa::IFile* pFile, size_t bufferSize) NN_NOEXCEPT;

// 4 GB を超えるオフセットでファイルにアクセスする共通テスト
void TestFileSystemAccessWithLargeOffset(nn::fs::fsa::IFileSystem* pFileSystem, const char* filePath) NN_NOEXCEPT;

// 4 GB を超えるオフセットでファイルハンドルにアクセスする
void TestFileHandleAccessWithLargeOffset(nn::fs::FileHandle file, size_t bufferSize) NN_NOEXCEPT;

// ファイルハンドルの AllowAppend 動作テスト
void TestFileHandleAccessAllowAppend(nn::fs::FileHandle file, size_t bufferSize) NN_NOEXCEPT;

// ファイルハンドルの AllowAppend 無効の動作テスト
void TestFileHandleAccessNotAllowAppend(
    size_t bufferSize,
    std::function<void(std::function<void(nn::fs::FileHandle)>)> fileAccessor) NN_NOEXCEPT;

// 4 GB を超えるオフセットでストレージにアクセスする共通テスト
void TestStorageAccessWithLargeOffset(nn::fs::IStorage* pStorage, size_t bufferSize) NN_NOEXCEPT;

// 4 GB を超えるオフセットでストレージにアクセスする共通テスト（読み込みテストを独自関数で行う）
void TestStorageAccessWithLargeOffset(
    nn::fs::IStorage* pStorage,
    size_t bufferSize,
    std::function<void(int64_t, char*, const char*, size_t)> readTestFunc) NN_NOEXCEPT;

// 指定したパスのファイル、ディレクトリを削除します。対象が存在していなくても成功扱いにします。
nn::Result DeleteFileOrDirectoryIfExists(const char* path) NN_NOEXCEPT;

/**
 * @brief バッファの初期値領域を確認する関数です。
 *   @param[in]  pBuffer                初期値を確認するバッファの先頭ポインタを指定します
 *   @param[in]  bufferSize             指定しているバッファのサイズ（バイト単位）を指定します
 *   @param[in]  offsetSize             バッファをオフセットするサイズ（バイト単位）を指定します
 *   @param[in]  compareSize            書き込み・読み込みを比較するサイズ（バイト単位）を指定します
 *
 *   @return     指定バッファの比較結果が返ります。
 *   @retval     true                   指定バッファには初期値のみ含まれています。
 *   @retval     false                  指定バッファには初期値以外の内容が含まれています。
 */
bool IsValidInitializedBound(const uint8_t* pBuffer, const size_t bufferSize, const size_t offsetSize, const size_t compareSize) NN_NOEXCEPT;

/**
    @brief HostFs テストで使用する一時パスを管理します。
           Create で Host PC 上にユニークな一時ディレクトリを作成し、デストラクタで破棄します。
           MountHost にはこのクラスで生成したパスを指定します。
*/
class TemporaryHostDirectory
{
public:
    explicit TemporaryHostDirectory() NN_NOEXCEPT;

    ~TemporaryHostDirectory() NN_NOEXCEPT;

    // 一時ディレクトリを作成します。
    void Create() NN_NOEXCEPT;

    // 一時ディレクトリを削除します。
    void Delete() NN_NOEXCEPT;

    // 一時ディレクトリのパスを返します。
    String GetPath() const NN_NOEXCEPT;

private:
    String m_Path;
    bool m_IsCreated;
};


// テスト用署名検証鍵 (PublicModulus)
const unsigned char TestPublicModulus[] =
{
    0xd8, 0xf1, 0x18, 0xef, 0x32, 0x72, 0x4c, 0xa7, 0x47, 0x4c, 0xb9, 0xea, 0xb3, 0x04, 0xa8, 0xa4,
    0xac, 0x99, 0x08, 0x08, 0x04, 0xbf, 0x68, 0x57, 0xb8, 0x43, 0x94, 0x2b, 0xc7, 0xb9, 0x66, 0x49,
    0x85, 0xe5, 0x8a, 0x9b, 0xc1, 0x00, 0x9a, 0x6a, 0x8d, 0xd0, 0xef, 0xce, 0xff, 0x86, 0xc8, 0x5c,
    0x5d, 0xe9, 0x53, 0x7b, 0x19, 0x2a, 0xa8, 0xc0, 0x22, 0xd1, 0xf3, 0x22, 0x0a, 0x50, 0xf2, 0x2b,
    0x65, 0x05, 0x1b, 0x9e, 0xec, 0x61, 0xb5, 0x63, 0xa3, 0x6f, 0x3b, 0xba, 0x63, 0x3a, 0x53, 0xf4,
    0x49, 0x2f, 0xcf, 0x03, 0xcc, 0xd7, 0x50, 0x82, 0x1b, 0x29, 0x4f, 0x08, 0xde, 0x1b, 0x6d, 0x47,
    0x4f, 0xa8, 0xb6, 0x6a, 0x26, 0xa0, 0x83, 0x3f, 0x1a, 0xaf, 0x83, 0x8f, 0x0e, 0x17, 0x3f, 0xfe,
    0x44, 0x1c, 0x56, 0x94, 0x2e, 0x49, 0x83, 0x83, 0x03, 0xe9, 0xb6, 0xad, 0xd5, 0xde, 0xe3, 0x2d,
    0xa1, 0xd9, 0x66, 0x20, 0x5d, 0x1f, 0x5e, 0x96, 0x5d, 0x5b, 0x55, 0x0d, 0xd4, 0xb4, 0x77, 0x6e,
    0xae, 0x1b, 0x69, 0xf3, 0xa6, 0x61, 0x0e, 0x51, 0x62, 0x39, 0x28, 0x63, 0x75, 0x76, 0xbf, 0xb0,
    0xd2, 0x22, 0xef, 0x98, 0x25, 0x02, 0x05, 0xc0, 0xd7, 0x6a, 0x06, 0x2c, 0xa5, 0xd8, 0x5a, 0x9d,
    0x7a, 0xa4, 0x21, 0x55, 0x9f, 0xf9, 0x3e, 0xbf, 0x16, 0xf6, 0x07, 0xc2, 0xb9, 0x6e, 0x87, 0x9e,
    0xb5, 0x1c, 0xbe, 0x97, 0xfa, 0x82, 0x7e, 0xed, 0x30, 0xd4, 0x66, 0x3f, 0xde, 0xd8, 0x1b, 0x4b,
    0x15, 0xd9, 0xfb, 0x2f, 0x50, 0xf0, 0x9d, 0x1d, 0x52, 0x4c, 0x1c, 0x4d, 0x8d, 0xae, 0x85, 0x1e,
    0xea, 0x7f, 0x86, 0xf3, 0x0b, 0x7b, 0x87, 0x81, 0x98, 0x23, 0x80, 0x63, 0x4f, 0x2f, 0xb0, 0x62,
    0xcc, 0x6e, 0xd2, 0x46, 0x13, 0x65, 0x2b, 0xd6, 0x44, 0x33, 0x59, 0xb5, 0x8f, 0xb9, 0x4a, 0xa9
};

// テスト用署名鍵 (PrivateExponent)
const unsigned char TestPrivateExponent[] =
{
    0x0c, 0x05, 0xb5, 0x6d, 0xe9, 0x0f, 0xe6, 0x41, 0x55, 0x6d, 0x52, 0x36, 0xc8, 0x57, 0xb3, 0x60,
    0x57, 0xdb, 0xcd, 0xb3, 0x03, 0x0f, 0x57, 0xf1, 0x17, 0x8a, 0x30, 0x33, 0x8a, 0x68, 0x92, 0xfb,
    0x73, 0x57, 0x04, 0x8a, 0xcb, 0xe3, 0xf4, 0x8a, 0xbf, 0xe3, 0xf2, 0xac, 0x38, 0x23, 0x30, 0x26,
    0x95, 0x42, 0x3d, 0x50, 0xfa, 0xb4, 0xaf, 0x60, 0x21, 0x75, 0xdc, 0xd9, 0x57, 0xb4, 0xc3, 0x6c,
    0xe5, 0xf6, 0xe5, 0xe0, 0x55, 0x65, 0x77, 0x4b, 0xc7, 0xa6, 0x7e, 0x0a, 0xfe, 0xdd, 0x80, 0x42,
    0x4f, 0x0d, 0x7e, 0x15, 0x8d, 0xf4, 0x27, 0x37, 0x24, 0x99, 0xf2, 0x12, 0x31, 0xdb, 0xd7, 0x7f,
    0x1e, 0x92, 0x21, 0x14, 0xca, 0x21, 0xf6, 0x50, 0x08, 0x92, 0xae, 0x31, 0xde, 0xf4, 0x29, 0x24,
    0xd6, 0x41, 0xb3, 0x47, 0x18, 0x37, 0x14, 0xf9, 0x8d, 0x5d, 0x95, 0xf4, 0xf5, 0x7f, 0x99, 0xfb,
    0x86, 0xda, 0x65, 0xe9, 0x72, 0xa9, 0x77, 0x65, 0xc8, 0xc5, 0x29, 0x5a, 0x19, 0x2b, 0x51, 0x1c,
    0x72, 0xeb, 0x49, 0xd1, 0x0b, 0x73, 0x8b, 0x3e, 0x2e, 0xc8, 0x7e, 0xff, 0xd8, 0xfe, 0xf4, 0xf4,
    0xf6, 0x92, 0x27, 0x7f, 0xa0, 0xdb, 0xc1, 0x25, 0xbc, 0xec, 0x5f, 0x0b, 0x2d, 0x99, 0xeb, 0xdd,
    0x9e, 0x5d, 0x42, 0x75, 0xb5, 0xe3, 0x24, 0xcb, 0xe9, 0xeb, 0xd9, 0x00, 0x4b, 0x12, 0x5d, 0xa3,
    0xa6, 0x25, 0xac, 0x20, 0x82, 0x25, 0x53, 0x1f, 0xc6, 0x2f, 0x27, 0xf1, 0x99, 0x7a, 0x99, 0xdc,
    0xa5, 0xc0, 0x5e, 0x63, 0x0f, 0x78, 0x03, 0x2a, 0x18, 0xd9, 0xe1, 0x06, 0x3b, 0xdf, 0xb2, 0x95,
    0x19, 0x32, 0xb4, 0x65, 0xd2, 0xd0, 0xfe, 0x18, 0xc7, 0x54, 0x5c, 0xa4, 0xf6, 0xd8, 0xfd, 0xdb,
    0x6d, 0xd8, 0xda, 0xf2, 0x9a, 0x55, 0x5c, 0x3e, 0xec, 0x17, 0x72, 0x09, 0xa3, 0x1a, 0x0a, 0xc1
};


/**
@brief セーブデータ移行・セーブデータクラウドバックアップ用の署名検証鍵を TestPublicModulus に差し替えます。
*/
nn::Result OverrideSaveDataTransferTokenSignVerificationKeyByTestKey();

/**
@brief セーブデータクラウドバックアップ用の KeySeedPackage を TestPublicModulus, TestPrivateExponent で生成します。
*/
void GenerateKeySeedPackage(
    nn::fs::SaveDataTransferManagerVersion2::KeySeedPackage* pOutKsp,
    const nn::ncm::ApplicationId applicationId,
    const nn::fs::detail::SdaId sdaId,
    const nn::fs::detail::KeySeed& keySeed,
    const nn::fs::detail::InitialDataMac initialDataMac,
    const nn::fs::SaveDataTransferManagerVersion2::Challenge& challenge,
    bool falsify) NN_NOEXCEPT;


/**
* @brief （内部用）pred にマッチするセーブデータを countMax 個まで取得します。countMax == 0 : 無限
*/
template <typename Pred>
void FindSaveData(Vector<nn::fs::SaveDataInfo>* pOutValue, nn::fs::SaveDataSpaceId spaceId, Pred pred, int countMax)
{
    using nn::util::nullopt;
    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    auto result = nn::fs::OpenSaveDataIterator(&iter, spaceId, nn::fs::SaveDataFilter::Make(nullopt, nullopt, nullopt, nullopt, nullopt, nn::fs::SaveDataEnumerateOption_AllRank));
    if (nn::fs::ResultPermissionDenied::Includes(result))
    {
        // フルパーミッション権限が無いテストではフィルタ指定無しで列挙できる範囲を列挙する
        result = nn::fs::OpenSaveDataIterator(&iter, spaceId);
    }
    if (nn::fs::ResultPortSdCardNoDevice::Includes(result))
    {
        // Sd が挿入されていない場合は無視する
        return;
    }
    NNT_EXPECT_RESULT_SUCCESS(result);

    pOutValue->clear();
    nn::fs::SaveDataInfo info;
    while (countMax == 0 || static_cast<int>(pOutValue->size()) < countMax )
    {
        int64_t count = 0;
        NNT_EXPECT_RESULT_SUCCESS(iter->ReadSaveDataInfo(&count, &info, 1));
        if (count == 0)
        {
            return;
        }
        if (pred(info))
        {
            pOutValue->push_back(info);
        }
    }
}


/**
* @brief pred にマッチするセーブデータを全て取得します。
*/
template <typename Pred>
void FindSaveData(Vector<nn::fs::SaveDataInfo>* pOutValue, nn::fs::SaveDataSpaceId spaceId, Pred pred)
{
    FindSaveData(pOutValue, spaceId, pred, 0);
}

/**
* @brief pred に最初にマッチするセーブデータを取得します。
*/
template <typename Pred>
void FindSaveData(nn::util::optional<nn::fs::SaveDataInfo>* pOutValue, nn::fs::SaveDataSpaceId spaceId, Pred pred)
{
    Vector<nn::fs::SaveDataInfo> infoArray;
    FindSaveData(&infoArray, spaceId, pred, 1);
    if (infoArray.size() == 0)
    {
        *pOutValue = nn::util::nullopt;
        return;
    }
    else
    {
        *pOutValue = infoArray[0];
        return;
    }
}

// セーブデータ取得ユーティリティ
nn::Result FindSaveDataBySystemSaveDataId(nn::fs::SaveDataInfo* outValue, nn::fs::SystemSaveDataId systemSaveDataId, nn::fs::UserId userId) NN_NOEXCEPT;
nn::Result FindSaveDataByApplicationId(nn::fs::SaveDataInfo* outValue, nn::ncm::ApplicationId applicationId, nn::fs::UserId userId) NN_NOEXCEPT;


/**
@brief actual が指すセーブデータの内容・userId を除く extraData が expected と一致すること、
       サムネイルファイルヘッダが IsFilledWith8BitCount(thumbnailFillValueOffset) であることをチェックします。
@pre expected, actual はユーザアカウント毎セーブデータ
*/
void CheckSaveData(const nn::fs::SaveDataInfo& expected, const nn::fs::SaveDataInfo& actual, int thumbnailFillValueOffset) NN_NOEXCEPT;


}}}


#if defined(NN_BUILD_CONFIG_OS_WIN)

// TORIAEZU:
namespace nn { namespace gc {
void SetStorage(nn::fs::IStorage* pStorage) NN_NOEXCEPT;
void Detach() NN_NOEXCEPT;
void Attach() NN_NOEXCEPT;
}}

namespace nn { namespace fs { namespace detail {
void InvalidateSdCardForWin() NN_NOEXCEPT;
}}}

namespace nn { namespace spl {

void InitializeForCrypto() NN_NOEXCEPT;
void Finalize() NN_NOEXCEPT;

}}

#endif




namespace nn { namespace fs {

bool operator==(const ReadOption& lhs, const ReadOption& rhs) NN_NOEXCEPT;
bool operator==(const WriteOption& lhs, const WriteOption& rhs) NN_NOEXCEPT;

}}

#include <nnt/fsUtil/testFs_util_function.Impl.h>

