﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <random>

#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fssystem/buffers/fs_BufferManagerUtility.h>
#include <nn/result/result_HandlingUtility.h>

namespace nnt { namespace fs { namespace util {

    class RandomAllocationFailureStorage : public nn::fs::IStorage
    {
    public:
        RandomAllocationFailureStorage(nn::fs::IStorage* pBaseStorage, std::mt19937* pMt) NN_NOEXCEPT
            : m_pBaseStorage(pBaseStorage),
            m_pMt(pMt),
            m_Divider(0),
            m_DividerOriginal(0)
        {
        }

        virtual ~RandomAllocationFailureStorage() NN_NOEXCEPT NN_OVERRIDE {}

    public:
        virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(OnPreOperation());
            NN_RESULT_DO(m_pBaseStorage->Read(offset, buffer, size));
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(OnPreOperation());
            NN_RESULT_DO(m_pBaseStorage->Flush());
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(OnPreOperation());
            NN_RESULT_DO(m_pBaseStorage->GetSize(outValue));
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result OperateRange(
            void* outBuffer,
            size_t outBufferSize,
            nn::fs::OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(OnPreOperation());
            NN_RESULT_DO(
                m_pBaseStorage->OperateRange(
                    outBuffer,
                    outBufferSize,
                    operationId,
                    offset,
                    size,
                    inBuffer,
                    inBufferSize
                )
            );
            NN_RESULT_SUCCESS;
        }

    public:
        void SetDivider(int divider) NN_NOEXCEPT
        {
            m_DividerOriginal = divider;
            m_Divider = m_DividerOriginal;
            m_OperationCount = 0;
        }

        int GetOperationCount() NN_NOEXCEPT
        {
            return m_OperationCount;
        }

    private:
        nn::Result OnPreOperation() NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_EQUAL(0, m_DividerOriginal);
            NN_SDK_ASSERT_NOT_EQUAL(0, m_Divider);

            ++m_OperationCount;

            auto *pContext = nn::fssystem::buffers::GetBufferManagerContext();
            if( (pContext == nullptr || !pContext->IsNeedBlocking())
                && std::uniform_int_distribution<>(1, m_Divider)(*m_pMt) <= 1 )
            {
                // スタックが巻き戻ると予測してリセット
                // 確実にテストを終わらせるために確率を下げる
                m_DividerOriginal *= 2;
                m_Divider = m_DividerOriginal;
                m_OperationCount = 0;

                NN_RESULT_THROW(nn::fs::ResultBufferAllocationFailed());
            }
            else
            {
                if( m_Divider >= 3 )
                {
                    // 深いところでも浅いところと同確率で発生させたい
                    --m_Divider;
                }
                NN_RESULT_SUCCESS;
            }
        }

    private:
        nn::fs::IStorage* m_pBaseStorage;
        std::mt19937* m_pMt;
        int m_Divider;
        int m_DividerOriginal;
        int m_OperationCount;
    };

}}}
