﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/fs/fs_QueryRange.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/nntest.h>

using namespace nn;

namespace nnt { namespace fs { namespace api {

enum class FsApiTestType
{
    Unit,
    Integration
};

enum FileSystemType
{
    FileSystemType_HostFileSystem = 0,
    FileSystemType_FatFileSystem,
    FileSystemType_SaveDataFileSystem,
    FileSystemType_RomFileSystem,
    FileSystemType_AesXtsFileSystem,
    FileSystemType_HtmlDocumentFileSystem,
};

struct FileSystemAttribute
{
    int fileNameLengthMax;
    int directoryNameLengthMax;
    int filePathLengthMax;
    int directoryPathLengthMax;

    int fileOpenMax;
    int directoryOpenMax;

    int32_t aesCtrKeyTypeFlag;
    int32_t speedEmulationTypeFlag;

    int64_t fileSizeMax;
    size_t fileSizeAlignment;
    int64_t storageSize;

    bool isSupportedMultiBytePath;
    bool isSupportedGetFreeSpaceSize;
    bool isConcatenationFileSystem;
    bool isSupportedGetFileTimeStamp;
    bool isSaveFileSystem;
    bool isSupportedQueryRange;
    bool isMemoryStorage;
    bool isSupportedFileOpenCheckOnCommit;
    bool isRootExcessFailureResultPathNotFound;
    bool isCommitSupported;
    bool isReadOverloadsSupported;

    explicit FileSystemAttribute(FsApiTestType testType) NN_NOEXCEPT
    {
        // 全変数に初期値を入れる
        fileNameLengthMax = 0;
        directoryNameLengthMax = 0;
        filePathLengthMax = 0;
        directoryPathLengthMax = 0;
        fileOpenMax = 0;
        directoryOpenMax = 0;
        aesCtrKeyTypeFlag = 0;
        speedEmulationTypeFlag = 0;
        fileSizeMax = 0;
        fileSizeAlignment = 0;
        storageSize = 0;
        isSupportedMultiBytePath = false;
        isSupportedGetFreeSpaceSize = false;
        isConcatenationFileSystem = false;
        isSupportedGetFileTimeStamp = false;
        isSaveFileSystem = false;
        isSupportedQueryRange = false;
        isMemoryStorage = false;
        isSupportedFileOpenCheckOnCommit = false;
        isRootExcessFailureResultPathNotFound = false;
        isCommitSupported = true;

        switch( testType )
        {
        case FsApiTestType::Integration:
            isReadOverloadsSupported = true;
            break;

        case FsApiTestType::Unit:
            isReadOverloadsSupported = false;
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    ~FileSystemAttribute() NN_NOEXCEPT
    {
    }
};

class ITestFile
{
public:
    virtual ~ITestFile() NN_NOEXCEPT {}

public:
    virtual void Close() NN_NOEXCEPT = 0;

    virtual Result Read(int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT = 0;

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT = 0;

    virtual Result Read(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT = 0;

    virtual Result Read(size_t* outValue, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT = 0;

    virtual Result Write(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT = 0;

    virtual Result Flush() NN_NOEXCEPT = 0;

    virtual Result SetSize(int64_t size) NN_NOEXCEPT = 0;

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT = 0;

    virtual Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        nn::fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT = 0;

    virtual Result OperateRange(
        nn::fs::OperationId operationId,
        int64_t offset,
        int64_t size) NN_NOEXCEPT = 0;

    virtual nn::Result QueryRange(nn::fs::QueryRangeInfo* outValue, int64_t offset, int64_t size) NN_NOEXCEPT = 0;
};

class ITestDirectory
{
public:
    virtual ~ITestDirectory() NN_NOEXCEPT {}

public:
    virtual void Close() NN_NOEXCEPT = 0;

    virtual Result Read(int64_t* outValue, nn::fs::DirectoryEntry* outEntries, int64_t count) NN_NOEXCEPT = 0;

    virtual Result GetEntryCount(int64_t* outValue) NN_NOEXCEPT = 0;
};

class ITestFileSystem
{
public:
    virtual ~ITestFileSystem() NN_NOEXCEPT {}

public:
    virtual Result CreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT = 0;

    virtual Result CreateFile(const char* path, int64_t size) NN_NOEXCEPT = 0;

    virtual Result DeleteFile(const char* path) NN_NOEXCEPT = 0;

    virtual Result CreateDirectory(const char* path) NN_NOEXCEPT = 0;

    virtual Result DeleteDirectory(const char* path) NN_NOEXCEPT = 0;

    virtual Result DeleteDirectoryRecursively(const char* path) NN_NOEXCEPT = 0;

    virtual Result CleanDirectoryRecursively(const char* path) NN_NOEXCEPT = 0;

    virtual Result RenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT = 0;

    virtual Result RenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT = 0;

    virtual Result GetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT = 0;

    virtual Result OpenFile(std::unique_ptr<ITestFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT = 0;

    virtual Result OpenDirectory(std::unique_ptr<ITestDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT = 0;

    virtual Result Commit() NN_NOEXCEPT = 0;

    virtual Result CommitSaveData() NN_NOEXCEPT = 0;

    virtual Result GetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT = 0;

    virtual Result GetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT = 0;

    virtual Result Flush() NN_NOEXCEPT = 0;

    virtual Result GetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT = 0;

    virtual Result QueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, nn::fs::fsa::QueryId queryId, const char* path) NN_NOEXCEPT = 0;
};

class TestFileSystemInfo
{
public:
    explicit TestFileSystemInfo(FsApiTestType type) NN_NOEXCEPT
        : attribute(type)
    {
    }

public:
    std::unique_ptr<ITestFileSystem> fileSystem;
    FileSystemType type;
    const char* rootDirPath;

    FileSystemAttribute attribute;

public:
    static void* operator new(size_t size) NN_NOEXCEPT
    {
        return nnt::fs::util::Allocate(size);
    }
    static void operator delete(void* p, size_t size) NN_NOEXCEPT
    {
        nnt::fs::util::Deallocate(p, size);
    }
};

/**
   @brief テスト対象のファイルシステム情報を取得します。この API はテストプロセス内で実装する必要があります。
*/
void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, int index) NN_NOEXCEPT;

/**
   @brief SetUp() で GetTestFileSystemInfo(index = 0) を実行するフィクスチャです。
          また、CheckGlobalNewDeleteFlagTestFixture を継承しその SetUp() も実行します。
          SetUp() 後は、GetFs() を利用してファイルシステムを操作でき、
          GetFsType() でファイルシステムの種類を取得できます。
          また、GetTestRootPath() でテストのルートディレクトリパスを取得できます。
*/
class GetFileSystemTestFixture : public nnt::fs::util::CheckGlobalNewDeleteFlagTestFixture, public nnt::fs::util::CheckMemoryLeakFixture
{
public:
    ITestFileSystem& GetFs() NN_NOEXCEPT
    {
        return *m_Info->fileSystem.get();
    }
    FileSystemType GetFsType() NN_NOEXCEPT
    {
        return m_Info->type;
    }
    nnt::fs::util::String GetTestRootPath() NN_NOEXCEPT
    {
        return nnt::fs::util::String(m_Info->rootDirPath);
    }
    const FileSystemAttribute* GetFsAttribute() const NN_NOEXCEPT
    {
        return &m_Info->attribute;
    }
    void RemountFs() NN_NOEXCEPT
    {
        // Unmount
        m_Info->fileSystem.reset(nullptr);

        // Mount
        std::unique_ptr<TestFileSystemInfo> info;
        GetTestFileSystemInfo(&info, 0);
        m_Info->fileSystem.reset(info->fileSystem.release());
    }

protected:
    virtual void SetUp() NN_NOEXCEPT
    {
        nnt::fs::util::CheckGlobalNewDeleteFlagTestFixture::SetUp();
        std::unique_ptr<TestFileSystemInfo> info;
        GetTestFileSystemInfo(&info, 0);
        if( info->attribute.fileSizeMax > info->attribute.storageSize )
        {
            info->attribute.fileSizeMax = info->attribute.storageSize;
        }

        if( info->attribute.directoryNameLengthMax > info->attribute.directoryPathLengthMax )
        {
            info->attribute.directoryNameLengthMax = info->attribute.directoryPathLengthMax;
        }
        if( info->attribute.fileNameLengthMax > info->attribute.filePathLengthMax )
        {
            info->attribute.fileNameLengthMax = info->attribute.filePathLengthMax;
        }

        m_Info = std::move(info);
        nnt::fs::util::CheckMemoryLeakFixture::StartMemoryCheck();
    }
    virtual void TearDown() NN_NOEXCEPT
    {
        nnt::fs::util::CheckMemoryLeakFixture::StopMemoryLeakCheck();
        nnt::fs::util::CheckGlobalNewDeleteFlagTestFixture::TearDown();
    }
private:
    std::unique_ptr<TestFileSystemInfo> m_Info;
};

class CleanupFileSystemTestFixture : public nnt::fs::api::GetFileSystemTestFixture
{
public:
    nnt::fs::util::String GetTestRootPath() NN_NOEXCEPT
    {
        return testDirPath;
    }

protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        GetFileSystemTestFixture::SetUp();
        testDirPath = GetFileSystemTestFixture::GetTestRootPath() + "/FsTest";
        GetFs().DeleteDirectoryRecursively(testDirPath.c_str());
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(testDirPath.c_str()));
        if( GetFsAttribute()->isSaveFileSystem )
        {
            NNT_ASSERT_RESULT_SUCCESS(GetFs().Commit());
        }
    }
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // エラーをキャッシュする Fs があるのでエラーはチェックしない
        (void) GetFs().DeleteDirectoryRecursively(testDirPath.c_str());
        if( GetFsAttribute()->isSaveFileSystem )
        {
            (void) GetFs().Commit();
        }
        GetFileSystemTestFixture::TearDown();
    }
private:
    nnt::fs::util::String testDirPath;
};

}}}
