﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cinttypes> // int16_t
#include <cstddef> // nullptr, std::size_t

#include <nn/codec/codec_OpusCommon.h>
#include <nn/codec/codec_OpusEncoderTypes.h>
#include <nn/codec/codec_OpusUtility.h>

namespace nnt {
namespace codec {

template <typename EncoderType>
class NativeOpusEncoder
{
public:
    NativeOpusEncoder() NN_NOEXCEPT
        : m_SampleRate(0)
        , m_ChannelCount(0)
        , m_StreamCount(0)
        , m_CoupledStreamCount(0)
        , m_WorkBufferAddress(nullptr)
        , m_WorkBufferSize(0)
        , m_IsInitialized(false)
    {}

    inline int GetSampleRate() const NN_NOEXCEPT
    {
        return m_SampleRate;
    }
    inline int GetChannelCount() const NN_NOEXCEPT
    {
        return m_ChannelCount;
    }
    inline int GetStreamCount() const NN_NOEXCEPT
    {
        return m_StreamCount;
    }
    inline int GetCoupledStreamCount() const NN_NOEXCEPT
    {
        return m_CoupledStreamCount;
    }
    inline bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }
    inline std::size_t GetWorkBufferSize(int sampleRate, int channelCount) NN_NOEXCEPT
    {
        return GetWorkBufferSize(sampleRate, 1, channelCount - 1);
    }
    virtual std::size_t GetWorkBufferSize(int sampleRate, int streamCount, int coupledStreamCount) NN_NOEXCEPT;
    inline int Initialize(int sampleRate, int channelCount, void* workBufferAddress, std::size_t workBufferSize) NN_NOEXCEPT
    {
        const uint8_t channelMappingTable[2] = { 0, 1 };
        return Initialize(
            sampleRate,
            channelCount,
            1,
            channelCount - 1,
            channelMappingTable,
            workBufferAddress,
            workBufferSize
        );
    }
    virtual nn::codec::OpusResult Initialize(int sampleRate, int channelCount, int streamCount, int coupledStreamCount, const uint8_t* pChannelMappingTable, void* workBufferAddress, std::size_t workBufferSize) NN_NOEXCEPT;
    virtual void Finalize() NN_NOEXCEPT;
    virtual nn::codec::OpusResult EncodeInterleaved(size_t* pOutputSize, void* outputBuffer, size_t outputBufferSize, const int16_t* inputBuffer, int inputSampleCountPerChannel) NN_NOEXCEPT;
    virtual void SetBitRate(int bitRate) NN_NOEXCEPT;
    virtual void SetBitRateControl(nn::codec::OpusBitRateControl bitRateControl) NN_NOEXCEPT;
    virtual void BindCodingMode(nn::codec::OpusCodingMode codingMode) NN_NOEXCEPT;
    virtual void GetCodingMode() NN_NOEXCEPT;
    virtual int CalculateFrameSampleCount(int frameInMicroSeconds) const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT(frameInMicroSeconds == 2500
            || frameInMicroSeconds == 5000
            || frameInMicroSeconds == 10000
            || frameInMicroSeconds == 20000
            || frameInMicroSeconds == 40000
            || frameInMicroSeconds == 60000);
        return static_cast<int>(static_cast<int64_t>(frameInMicroSeconds) * GetSampleRate() / 1000 / 1000);
    }

private:
    int m_SampleRate;
    int m_ChannelCount;
    int m_StreamCount;
    int m_CoupledStreamCount;
    void* m_WorkBufferAddress;
    std::size_t m_WorkBufferSize;
    bool m_IsInitialized;
};

}} // nnt::codec
