﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cinttypes> // int16_t
#include <cstddef> // nullptr, std::size_t

// sdk
#include <nn/nn_Assert.h>
#include <nn/codec/codec_OpusCommon.h>

namespace nnt {
namespace codec {

template <typename DecoderType>
class NativeOpusDecoder
{
public:
    NativeOpusDecoder() NN_NOEXCEPT
        : m_SampleRate(0)
        , m_ChannelCount(0)
        , m_StreamCount(0)
        , m_CoupledStreamCount(0)
        , m_WorkBufferAddress(nullptr)
        , m_WorkBufferSize(0)
        , m_IsInitialized(false)
    {}

    inline int GetSampleRate() const NN_NOEXCEPT
    {
        return m_SampleRate;
    }
    inline int GetChannelCount() const NN_NOEXCEPT
    {
        return m_ChannelCount;
    }
    inline int GetStreamCount() const NN_NOEXCEPT
    {
        return m_StreamCount;
    }
    inline int GetCoupledStreamCount() const NN_NOEXCEPT
    {
        return m_CoupledStreamCount;
    }
    inline bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }
    inline std::size_t GetWorkBufferSize(
        int sampleRate,
        int channelCount) NN_NOEXCEPT
    {
        NN_ASSERT(channelCount == 2 || channelCount == 1);
        return GetWorkBufferSize(sampleRate, 1, channelCount - 1);
    }

    inline int Initialize(
        int sampleRate,
        int channelCount,
        void* workBufferAddress,
        std::size_t workBufferSize) NN_NOEXCEPT
    {
        NN_ASSERT(channelCount == 2 || channelCount == 1);
        return Initialize(
            sampleRate,
            channelCount,
            1,
            channelCount - 1,
            nullptr,
            workBufferAddress,
            workBufferSize);
    }

    virtual std::size_t GetWorkBufferSize(
        int sampleRate,
        int streamCount,
        int coupledStreamCount) NN_NOEXCEPT;

    virtual nn::codec::OpusResult Initialize(
        int sampleRate,
        int channelCount,
        int streamCount,
        int coupledStreamCount,
        const uint8_t* channelMapping,
        void* workBufferAddress,
        std::size_t workBufferSize) NN_NOEXCEPT;

    virtual nn::codec::OpusResult DecodeInterleaved(size_t* pOutConsumed, int* pOutSampleCount, int16_t* outputBuffer, size_t outputSize, const void* inputBuffer, size_t inputSize) NN_NOEXCEPT;

    virtual void Finalize() NN_NOEXCEPT
    {
        if (!IsInitialized())
        {
            return ;
        }
        m_IsInitialized = false;
        m_WorkBufferAddress = nullptr;
        m_WorkBufferSize = 0;
    }

private:
    int m_SampleRate;
    int m_ChannelCount;
    int m_StreamCount;
    int m_CoupledStreamCount;
    void* m_WorkBufferAddress;
    std::size_t m_WorkBufferSize;
    bool m_IsInitialized;
};

}} // nnt::codec
