﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <type_traits>
#include <nn/util/util_IntrusiveList.h>

#include <nnt.h>
#include <nn/nn_Result.h>
#include <nn/audio.h>

namespace nnt {
namespace audio {
namespace util {

class AudioNode : public nn::util::IntrusiveListBaseNode<AudioNode>
{
public:
    enum class NodeTypeInfo
    {
        Voice,
        MultiDestinationVoice,
        Splitter,
        SubMix,
        MultiDestinationSubMix,
        FinalMix,
    };
    virtual nn::audio::NodeId  GetNodeId() const = 0;
    virtual void SetVolume(float volume) = 0;
    virtual NodeTypeInfo GetType() const = 0;
    virtual int GetChannelCount() const = 0;

    template <typename T>
    T* As()
    {
        if (T::_TypeInfo == this->GetType())
        {
            return static_cast<T*>(this);
        }
        else
        {
            return nullptr;
        }
    }
    bool operator==(const AudioNode& rhs) const
    {
        return GetNodeId() == rhs.GetNodeId();
    }
    bool operator!=(const AudioNode& rhs) const
    {
        return !(*this == rhs);
    }
};

class AudioNodeList : public nn::util::IntrusiveList<AudioNode, nn::util::IntrusiveListBaseNodeTraits<AudioNode>>
{
public:
    bool Contains(const AudioNode* node)
    {
        for (auto itr = begin(); itr != end(); ++itr)
        {
            if (&(*itr) == node)
            {
                return true;
            }
        }
        return false;
    }
};

class Voice;
class SubMix;
class FinalMix;
class MultiDestinationVoice;
class MultiDestinationSubMix;

class AudioEffect;
using AudioEffectList = nn::util::IntrusiveList<AudioEffect, nn::util::IntrusiveListBaseNodeTraits<AudioEffect>>;
class Aux;

struct NodeDescription
{
    int index;
    AudioNode::NodeTypeInfo type;
    int channel;
    int sampleRate;
    int destinationCount;
};

struct EdgeDescription
{
    int sourceNodeIndex;
    int sourceDestIndex;
    int sourceChannel;
    int destNodeIndex;
    int destChannel;
    float volume;
};

//////////////////////////////////////////////////////////////////////////
// node base class
//////////////////////////////////////////////////////////////////////////
template <typename BaseType>
class BaseHolder
{
private:
    BaseType m_Base;
public:
    virtual BaseType* GetBase()
    {
        return &m_Base;
    }
    virtual const BaseType* GetBase() const
    {
        return &m_Base;
    }
};

template <typename SelfType, typename BaseType>
class MultiDestinationBase : public BaseHolder<BaseType>
{
protected:
    nn::audio::SplitterType m_Splitter;
    AudioNodeList m_DestinationList;
    struct DestinationInfo
    {
        AudioNode* dest;
    };
    DestinationInfo m_DestinationArray[20]; // TODO: temp val

    template <typename DestType>
    void SetMixVolumeMatrixImpl(int destinationIndex, const float* volumes, int sourceChannelCount, int destChannelCount, DestType* destination);
    template <typename DestType>
    void SetMixVolumeImpl(int destinationIndex, float volume, int sourceChannel, int destChannel, DestType* destination);
    template <typename DestType>
    SelfType* SetDestinationImpl(nn::audio::AudioRendererConfig* pConfig, int destinationIndex, DestType* destination);

public:
    nn::audio::SplitterType* GetSplitterBase() { return &m_Splitter; };
    const nn::audio::SplitterType* GetSplitterBase() const { return &m_Splitter; };
};


//////////////////////////////////////////////////////////////////////////
// Concrete node classes
//////////////////////////////////////////////////////////////////////////
class FinalMix : public AudioNode, public BaseHolder<nn::audio::FinalMixType>
{
public:
    static const NodeTypeInfo _TypeInfo = NodeTypeInfo::FinalMix;

    FinalMix();
    virtual NodeTypeInfo GetType() const NN_OVERRIDE;
    virtual nn::audio::NodeId GetNodeId() const NN_OVERRIDE;
    virtual void SetVolume(float volume) NN_OVERRIDE;
    virtual int GetChannelCount() const NN_OVERRIDE;
};

class SubMix : public AudioNode, public BaseHolder<nn::audio::SubMixType>
{
private:
    AudioNode* m_Destination;
    AudioEffectList m_EffectList;

public:
    static const NodeTypeInfo _TypeInfo = NodeTypeInfo::SubMix;

    SubMix();
    virtual NodeTypeInfo GetType() const NN_OVERRIDE;
    virtual nn::audio::NodeId GetNodeId() const NN_OVERRIDE;
    virtual void SetVolume(float volume) NN_OVERRIDE;
    virtual int GetChannelCount() const NN_OVERRIDE;

    SubMix* SetMixVolume(float volume, int sourceChannel, int destChannel);
    SubMix* SetDestination(nn::audio::AudioRendererConfig* pConfig, AudioNode* destination);

private:
    template <typename DestType>
    void SetMixVolumeImpl(float volume, int sourceChannel, int destChannel, DestType* destination);
    template <typename DestType>
    SubMix* SetDestinationImpl(nn::audio::AudioRendererConfig* pConfig, DestType* destination);
};

class MultiDestinationSubMix : public AudioNode, public MultiDestinationBase<MultiDestinationSubMix, nn::audio::SubMixType>
{
private:
    AudioEffectList m_EffectList;

public:
    static const NodeTypeInfo _TypeInfo = NodeTypeInfo::MultiDestinationSubMix;

    MultiDestinationSubMix();
    virtual NodeTypeInfo GetType() const NN_OVERRIDE;
    virtual nn::audio::NodeId  GetNodeId() const NN_OVERRIDE;
    virtual void SetVolume(float volume) NN_OVERRIDE;
    virtual int GetChannelCount() const NN_OVERRIDE;
    MultiDestinationSubMix* SetDestination(nn::audio::AudioRendererConfig* pConfig, int destinationIndex, AudioNode* destination);
    MultiDestinationSubMix* SetMixVolume(int destinationIndex, float volume, int sourceChannel, int destChannel);
};

class Voice : public AudioNode, public BaseHolder<nn::audio::VoiceType>
{
private:
    AudioNode* m_Destination;

public:
    static const NodeTypeInfo _TypeInfo = NodeTypeInfo::Voice;

    Voice();
    virtual NodeTypeInfo GetType() const NN_OVERRIDE;
    virtual nn::audio::NodeId GetNodeId() const NN_OVERRIDE;
    virtual void SetVolume(float volume) NN_OVERRIDE;
    virtual int GetChannelCount() const NN_OVERRIDE;

    Voice* SetDestination(nn::audio::AudioRendererConfig* pConfig, AudioNode* destination);
    Voice* SetMixVolume(float volume, int sourceChannel, int destChannel);
    Voice* Start();
    Voice* Stop();
    Voice* AppendWaveBuffer(nn::audio::WaveBuffer* pWaveBuffer);

private:
    template <typename DestType>
    void SetMixVolumeImpl(float volume, int sourceChannel, int destChannel, DestType* destination);
    template <typename DestType>
    Voice* SetDestinationImpl(nn::audio::AudioRendererConfig* pConfig, DestType* destination);
};

class MultiDestinationVoice : public AudioNode, public MultiDestinationBase<MultiDestinationVoice, nn::audio::VoiceType>
{
public:
    static const NodeTypeInfo _TypeInfo = NodeTypeInfo::MultiDestinationVoice;

    MultiDestinationVoice();
    virtual NodeTypeInfo GetType() const NN_OVERRIDE;
    virtual nn::audio::NodeId  GetNodeId() const NN_OVERRIDE;
    virtual void SetVolume(float volume) NN_OVERRIDE;
    virtual int GetChannelCount() const NN_OVERRIDE;
    MultiDestinationVoice* SetDestination(nn::audio::AudioRendererConfig* pConfig, int destinationIndex, AudioNode* destination);
    MultiDestinationVoice* SetMixVolume(int destinationIndex, float volume, int sourceChannel, int destChannel);
    MultiDestinationVoice* Start();
    MultiDestinationVoice* Stop();
    MultiDestinationVoice* AppendWaveBuffer(nn::audio::WaveBuffer* pWaveBuffer);

};

class Splitter : public AudioNode, public MultiDestinationBase<Splitter, nn::audio::SplitterType>
{
public:
    static const NodeTypeInfo _TypeInfo = NodeTypeInfo::Splitter;

    Splitter();
    virtual NodeTypeInfo GetType() const NN_OVERRIDE;
    virtual nn::audio::NodeId  GetNodeId() const NN_OVERRIDE;
    virtual void SetVolume(float volume) NN_OVERRIDE;
    virtual int GetChannelCount() const NN_OVERRIDE;
    virtual nn::audio::SplitterType* GetBase() NN_OVERRIDE;
    virtual const nn::audio::SplitterType* GetBase() const NN_OVERRIDE;

    Splitter* SetDestination(nn::audio::AudioRendererConfig* pConfig, int destinationIndex, AudioNode* destination);
    Splitter* SetMixVolume(int destinationIndex, float volume, int sourceChannel, int destChannel);
};


//////////////////////////////////////////////////////////////////////////
// effect base classes
//////////////////////////////////////////////////////////////////////////
class AudioEffect : public nn::util::IntrusiveListBaseNode<AudioEffect>
{
public:
    enum class EffectTypeInfo
    {
        Aux,
    };

    AudioEffect() {}
    virtual EffectTypeInfo GetType() const = 0;
    template <typename T>
    T* As()
    {
        if (T::_TypeInfo == this->GetType())
        {
            return static_cast<T*>(this);
        }
        else
        {
            return nullptr;
        }
    }
};

//////////////////////////////////////////////////////////////////////////
// Concrete effect classes
//////////////////////////////////////////////////////////////////////////
class Aux : public AudioEffect, public BaseHolder<nn::audio::AuxType>
{
private:
    void* m_SendBuffer;
    void* m_ReturnBuffer;
    size_t m_BufferSize;

public:
    static const EffectTypeInfo _TypeInfo = AudioEffect::EffectTypeInfo::Aux;
    static const int DefaultFrameBufferCount = 10;

    virtual EffectTypeInfo GetType() const NN_OVERRIDE;
    static size_t GetWorkBufferSize(const nn::audio::AudioRendererParameter* param, int channelCount, int frameCount = DefaultFrameBufferCount);
    Aux* SetupBuffers(void* sendBuffer, void* returnBuffer, size_t bufferSize);
    Aux* SetInOut(const int8_t* input, const int8_t* output, const int count);
    int Read(int32_t* readBuffer, int count);
    void Write(const int32_t* writeBuffer, int count);
    Aux* AddTo(nn::audio::AudioRendererConfig* pConfig, AudioNode* node);
    int GetSampleCount() const NN_NOEXCEPT;

private:
    template <typename TargetNode>
    Aux* AddToImpl(nn::audio::AudioRendererConfig* pConfig, TargetNode* node);
};


//////////////////////////////////////////////////////////////////////////
// utility class
//////////////////////////////////////////////////////////////////////////
class AudioNodeAllocator
{
    AudioNodeList m_FreeVoice;
    AudioNodeList m_FreeMultidestinationVoice;
    AudioNodeList m_FreeSplitter;
    AudioNodeList m_FreeSubMix;
    AudioNodeList m_FreeMultidestinationSubMix;
    AudioNodeList m_FreeFinalMix;

    Voice* m_pVoices;
    int m_VoiceCount;
    MultiDestinationVoice* m_pMultiDestinationVoices;
    int m_MultiDestVoiceCount;
    Splitter* m_pSplitters;
    int m_SplitterCount;
    SubMix* m_pSubMixes;
    int m_SubMixCount;
    MultiDestinationSubMix* m_pMultiDestinationSubMixes;
    int m_MultiDestSubMixCount;
    FinalMix* m_pFinalMixes;
    int m_FinalMixCount;

    template<typename NodeBase>
    void InitializeList(AudioNodeList& list, NodeBase* base, int baseCount)
    {
        list.clear();
        for (auto i = 0; i < baseCount; ++i)
        {
            new(&base[i]) NodeBase();
            list.push_back(base[i]);
        }
    }

public:
    AudioNodeAllocator(void* voiceBuffer, int voiceCount,
                       void* multiVoiceBuffer, int multiVoiceCount,
                       void* splitterBuffer, int splitterCount,
                       void* subMixBuffer, int subMixCount,
                       void* multiSubMixBuffer, int multiSubMixCount,
                       void* finalMixBuffer, int finalMixCount);
    Voice* AllocateVoice(nn::audio::AudioRendererConfig* pConfig, int sampleRate, int channelCount, nn::audio::SampleFormat format, int priority, const void* pParameter, size_t paramSize);
    MultiDestinationVoice* AllocateMultiDestinationVoice(nn::audio::AudioRendererConfig* pConfig, int sampleRate, int channelCount, nn::audio::SampleFormat format, int priority, const void* pParameter, size_t paramSize, int destinationCount);
    Splitter* AllocateSplitter(nn::audio::AudioRendererConfig* pConfig, int sampleRate, int bufferCount, int destinationCount);
    SubMix* AllocateSubMix(nn::audio::AudioRendererConfig* pConfig, int sampleRate, int bufferCount);
    MultiDestinationSubMix* AllocateMultiDestinationSubMix(nn::audio::AudioRendererConfig* pConfig, int sampleRate, int bufferCount, int destinationCount);
    FinalMix* AllocateFinalMix(nn::audio::AudioRendererConfig* pConfig, int bufferCount);
};

} // namespace nnt
} // namespace audio
} // namespace util
