﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <functional>

#include <nn/nn_SdkAssert.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/os/os_Tick.h>

namespace nnt { namespace image { namespace jpeg {

#define NNT_SF_PERF_TEST_ARG_0(e, s, t, method) \
    ::nnt::image::jpeg::TestEntity( \
        ::nnt::image::jpeg::TestId::Get(e, s, t, #method "()"), \
        [&] { method(); })

#define NNT_SF_PERF_TEST_ARG_1(e, s, t, method, a0) \
    ::nnt::image::jpeg::TestEntity( \
        ::nnt::image::jpeg::TestId::Get(e, s, t, #method "(" #a0 ")"), \
        [&] { method(a0); })

#define NNT_SF_PERF_TEST_ARG_2(e, s, t, method, a0, a1) \
    ::nnt::image::jpeg::TestEntity( \
        ::nnt::image::jpeg::TestId::Get(e, s, t, #method "(" #a0 ", ...)"), \
        [&] { method(a0, a1); })

#define NNT_SF_PERF_TEST_ARG_3(e, s, t, method, a0, a1, a2) \
    ::nnt::image::jpeg::TestEntity( \
        ::nnt::image::jpeg::TestId::Get(e, s, t, #method "(" #a0 ", ...)"), \
        [&] { method(a0, a1, a2); })

#define NNT_SF_PERF_TEST_ARG_4(e, s, t, method, a0, a1, a2, a3) \
    ::nnt::image::jpeg::TestEntity( \
        ::nnt::image::jpeg::TestId::Get(e, s, t, #method "(" #a0 ", ...)"), \
        [&] { method(a0, a1, a2, a3); })

#define NNT_SF_PERF_TEST_ARG_5(e, s, t, method, a0, a1, a2, a3, a4) \
    ::nnt::image::jpeg::TestEntity( \
        ::nnt::image::jpeg::TestId::Get(e, s, t, #method "(" #a0 ", ...)"), \
        [&] { method(a0, a1, a2, a3, a4); })

#define NNT_SF_PERF_TEST_ARG_6(e, s, t, method, a0, a1, a2, a3, a4, a5) \
    ::nnt::image::jpeg::TestEntity( \
        ::nnt::image::jpeg::TestId::Get(e, s, t, #method "(" #a0 ", ...)"), \
        [&] { method(a0, a1, a2, a3, a4, a5); })

#define NNT_SF_PERF_TEST_ARG_7(e, s, t, method, a0, a1, a2, a3, a4, a5, a6) \
    ::nnt::image::jpeg::TestEntity( \
        ::nnt::image::jpeg::TestId::Get(e, s, t, #method "(" #a0 ", ...)"), \
        [&] { method(a0, a1, a2, a3, a4, a5, a6); })

#define NNT_SF_PERF_TEST_ARG_8(e, s, t, method, a0, a1, a2, a3, a4, a5, a6, a7) \
    ::nnt::image::jpeg::TestEntity( \
        ::nnt::image::jpeg::TestId::Get(e, s, t, #method "(" #a0 ", ...)"), \
        [&] { method(a0, a1, a2, a3, a4, a5, a6, a7); })

#define NNT_SF_PERF_TEST_BLOCK_WITH_PROCESS(env, set, testId, testName, pre, test, post) \
    ::nnt::image::jpeg::TestEntity( \
        ::nnt::image::jpeg::TestId::Get(env, set, testId, testName), \
        test, pre, post)

struct Statistic
{
    int64_t average;
};

struct TestId
{
    const char* env;
    const char* set;
    int32_t entity;
    const char *alt;

    static TestId Get(const char* e, const char* s, int32_t t, const char *n)
    {
        NN_SDK_REQUIRES(e != nullptr);
        NN_SDK_REQUIRES(s != nullptr);
        NN_SDK_REQUIRES(n != nullptr);
        TestId id = {e, s, t, n};
        return id;
    }

    inline void ToString(char *str, size_t s)
    {
        auto len = nn::util::SNPrintf(str, s, "%s_%s_%d", env, set, entity);
        NN_UNUSED(len);
        NN_SDK_ASSERT(len >= 0 && static_cast<uint32_t>(len) < s);
    }
    inline const char* GetAlternativeString()
    {
        return alt;
    }
    NN_IMPLICIT operator bool() const NN_NOEXCEPT
    {
        return true
            && env != nullptr
            && set != nullptr
            && alt != nullptr;
    }
};

struct TestResult
{
    TestId testId;
    Statistic stat;
};

struct TestEntity
{
    TestId testId;
    std::function<void()> test;
    std::function<void()> preprocess;
    std::function<void()> postprocess;

    TestEntity()
        : testId()
        , test(std::function<void()>())
        , preprocess(std::function<void()>())
        , postprocess(std::function<void()>())
    {
    }
    TestEntity(
        TestId id,
        std::function<void()> t = std::function<void()>(),
        std::function<void()> pre = std::function<void()>(),
        std::function<void()> post = std::function<void()>())
        : testId(id)
        , test(t)
        , preprocess(pre)
        , postprocess(post)
    {
    }
    NN_IMPLICIT operator bool() const NN_NOEXCEPT
    {
        return true
            && testId
            && test
            && preprocess
            && postprocess;
    }
    inline TestResult Invoke(const int LoopCount)
    {
        nn::os::Tick sum(0);
        for (auto i = 0; i < LoopCount; ++ i)
        {
            if (this->preprocess)
            {
                this->preprocess();
            }
            auto start = nn::os::GetSystemTick();
            this->test();
            auto end = nn::os::GetSystemTick();
            if (this->postprocess)
            {
                this->postprocess();
            }
            sum += end - start;
        }

        // 処理時間の統計情報の作成
        auto average = static_cast<double>(sum.GetInt64Value()) / LoopCount;;
        Statistic stat = {nn::os::Tick(static_cast<int64_t>(average + 0.5)).ToTimeSpan().GetNanoSeconds()};

        TestResult r = {testId, stat};
        return r;
    }
};

}}} // ~namespace nnt::image::jpeg
