﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <csetjmp>
#include <jpeglib.h>

/**
    @file libjpegのラッパー関数と関連するデータ型を定義します。
 */

namespace nnt { namespace image { namespace jpeg {

/**
    @brief 内部的に使用しがちな定数を定義します。
 */
enum LibjpegProperty
{
    LibjpegProperty_NumLinesToProcAtOnce = 4, ///< 同時に処理するライン数。メモリがある限り沢山でも可。
    LibjpegProperty_BytesPerPixel = 3, ///< libjpeg内部の1ピクセルあたりのバイト数。変更禁止。
};

/**
    @brief 矩形を表す型
 */
struct Dimension
{
    uint16_t width;   ///< 矩形幅
    uint16_t height;  ///< 矩形高さ

    Dimension GetAlignedDimension(size_t pxAlign)
    {
        Dimension r = {
            (width % pxAlign == 0)?
                width:
                static_cast<uint16_t>(width - width % pxAlign + pxAlign),
            (height % pxAlign == 0)?
                height:
                static_cast<uint16_t>(height - height % pxAlign + pxAlign)
        };
        return r;
    }
};

/**
    @brief Compress/ Decompress で使用する、エラーハンドル用の関数型です。
    @details    主に異常系のテストで、
                libjpegから返ってくる errorCode と reason が期待値かどうかを評価します。
 */
typedef bool (*ErrorHandler)(int errorCode, int reason);

/**
    @brief Decompress をするときに渡すテストの設定を表す型。
 */
struct TestArgDecode
{
    ErrorHandler errorHandler;      ///< エラーハンドラ。libjpegのエラーが期待値かどうかを検査します。
    int resolutionDenom;   ///< Decompress時のスケーリング設定値
    Dimension expectedDim; ///< Decompressされるピクセルデータの幅と高さの期待値
};

/**
    @brief Compress をするときに渡すテストの設定を表す型。
 */
struct TestArgEncode
{
    ErrorHandler errorHandler;  ///< エラーハンドラ。libjpegのエラーが期待値かどうかを検査します。
    int quality;       ///< Compress時の品質設定
    uint16_t yHSample;    ///< Compress時のY成分サンプリング比(水平方向)
    uint16_t yVSample;    ///< Compress時のY成分サンプリング比(垂直方向)
};

/**
    @brief JPEGデータのデコード
    @param[out] outPixel デコードされたピクセルデータの格納先
    @param[in] kPixelSize outPixelのバイト数
    @param[in] jpeg デコード対象
    @param[in] kJpegSize jpegのバイト数
    @param[in] decodeArg デコードの設定と期待値
    @param[in] temp ワークメモリ
    @param[in] kTempSize ワークメモリのバイト数
    @return decodeArgで期待通りの結果になったか、あるいは正常にデコードが終了したときにtrueが返ります。
 */
bool DecompressBinary(
    uint32_t *outPixel,
    const size_t kPixelSize,
    const void *jpeg,
    const size_t kJpegSize,
    const TestArgDecode &decodeArg,
    void *temp,
    const size_t kTempSize);

/**
    @brief JPEGデータへのエンコード
    @param[out] pOutActualCodedSize 実際にエンコードされたJPEGデータのバイト数
    @param[out] outJpegBuf エンコードされたJPEGデータの格納先
    @param[in] kJpegBufSize outJpegBufのバイト数
    @param[in] pixel エンコード対象のピクセルデータ
    @param[in] kDim pixelの幅と高さ
    @param[in] encodeArg エンコードの設定と期待値
    @param[in] temp ワークメモリ
    @param[in] kTempSize ワークメモリのバイト数
 */
bool CompressBinary(
    size_t *pOutActualCodedSize,
    void *outJpegBuf,
    const size_t kJpegBufSize,
    const uint32_t *pixel,
    const Dimension &kDim,
    const TestArgEncode &encodeArg,
    void *temp,
    const size_t kTempSize);

}}}
