﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <utility>  // std::pair
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/teamcity/testTeamcity_Logger.h>

#include <nn/settings/system/settings_Sleep.h>
#include <nn/settings/system/settings_Tv.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <nn/hid/system/hid_InputDetection.h>
#include <nn/lbl/lbl.h>
#include <nn/lbl/lbl_Switch.h>
#include <nn/vi.h>
#include <nn/vi/vi_CmuLuma.h>

// テスト対象
#include <nn/idle/idle_SystemApi.h>
#include <nn/idle/server/idle_HandlerManager.h>
#include <server/idle_IHandler.h> // ライブラリ内部実装ヘッダ

#include "HidStub.h"

namespace {

    using namespace ::nn::idle;

    struct SettingsParam
    {
        nn::settings::system::HandheldSleepPlan     handheldSleepPlan;
        nn::settings::system::ConsoleSleepPlan      consoleSleepPlan;
        bool                                        isAutoSleepInMediaPlaybackEnabled;
        bool                                        isTvDimmingEnabled;
    };

    struct ExpectedResults
    {
        nn::TimeSpan  dimTimeInHandheld;
        nn::TimeSpan  sleepTimeInHandheld;
        nn::TimeSpan  dimTimeInConsole;
        nn::TimeSpan  sleepTimeInConsole;
        nn::TimeSpan  dimTimeInHandheldPlayingMedia;
        nn::TimeSpan  sleepTimeInHandheldPlayingMedia;
        nn::TimeSpan  dimTimeInConsolePlayingMedia;
        nn::TimeSpan  sleepTimeInConsolePlayingMedia;
    };

    const auto UpdateIdleHandlerInterval = nn::TimeSpan::FromSeconds(1);
    const auto TimeSpanNever = nn::TimeSpan::FromHours(16); // テスト対象の仕様で最大の閾値が 12 時間なので、それより長い適当な時間

    const nn::hid::system::InputSourceIdSet AllInputSourceIdSet = nn::hid::system::InputSourceIdSet().Set();

    void SetSettings(const SettingsParam& settingsParam)
    {
        nn::settings::system::SleepSettings sleepSettings;
        nn::settings::system::GetSleepSettings(&sleepSettings);
        if ( settingsParam.isAutoSleepInMediaPlaybackEnabled )
        {
            sleepSettings.flags |= nn::settings::system::SleepFlag::SleepsWhilePlayingMedia::Mask;
        }
        else
        {
            sleepSettings.flags &= ~(nn::settings::system::SleepFlag::SleepsWhilePlayingMedia::Mask);
        }
        sleepSettings.handheldSleepPlan = settingsParam.handheldSleepPlan;
        sleepSettings.consoleSleepPlan= settingsParam.consoleSleepPlan;
        nn::settings::system::SetSleepSettings(sleepSettings);

        nn::settings::system::TvSettings tvSettings;
        nn::settings::system::GetTvSettings(&tvSettings);
        if ( settingsParam.isTvDimmingEnabled )
        {
            tvSettings.flags |= nn::settings::system::TvFlag::PreventsScreenBurnIn::Mask;
        }
        else
        {
            tvSettings.flags &= ~(nn::settings::system::TvFlag::PreventsScreenBurnIn::Mask);
        }
        nn::settings::system::SetTvSettings(tvSettings);
    }

    void UpdateHandlers(nn::TimeSpan* pOutApdTriggeredTime, nn::TimeSpan* pOutDimmingTriggeredTime, nn::TimeSpan elapsedIdleTimeBegin, nn::TimeSpan elapsedIdleTimeEnd)
    {
        nn::TimeSpan sleepRequestEventSignaledTime = TimeSpanNever;
        nn::TimeSpan dimmingTriggeredTime = TimeSpanNever;
        bool wasSleepRequestEventSignaled = false;
        bool wasDimmingTriggered = false;

        for ( auto elapsedIdleTime = elapsedIdleTimeBegin; elapsedIdleTime < elapsedIdleTimeEnd; )
        {
            elapsedIdleTime += UpdateIdleHandlerInterval; // 先に加算
            nn::idle::server::UpdateHandlerStatus(UpdateIdleHandlerInterval);

            if ( nn::idle::server::IsAutoPowerDownOn() )
            {
                nn::os::TryWaitSystemEvent(nn::idle::server::GetAutoPowerDownEvent());
                if ( !wasSleepRequestEventSignaled )
                {
                    wasSleepRequestEventSignaled = true;
                    sleepRequestEventSignaledTime = elapsedIdleTime;
                }
            }
            if ( nn::idle::server::IsDimmingOn() )
            {
                if ( !wasDimmingTriggered )
                {
                    wasDimmingTriggered = true;
                    dimmingTriggeredTime = elapsedIdleTime;
                }
            }
        }

        if ( pOutApdTriggeredTime )
        {
            *pOutApdTriggeredTime = sleepRequestEventSignaledTime;
        }
        if ( pOutDimmingTriggeredTime )
        {
            *pOutDimmingTriggeredTime = dimmingTriggeredTime;
        }
    }

    void TestTriggerTimeWithPeriodicInput(const nn::TimeSpan& expectedAutoSleepTime, const nn::TimeSpan& expectedDimmingTime, const HandlingContext& handlingContext, const int firstInputInSeconds, const int inputIntervalInSeconds)
    {
        nn::idle::server::InitializeHandlerManager(); // settings がリロードされる
        nn::idle::server::EnableHandlers();
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(handlingContext));

        nnt::hid::system::SetInputSourceState(nn::hid::system::InputSourceState{ 0LL });

        auto* pAutoSleepRequestEvent = nn::idle::server::GetAutoPowerDownEvent();

        // 事前状態のチェック
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(pAutoSleepRequestEvent));
        ASSERT_FALSE(nn::idle::server::IsDimmingOn());
        ASSERT_FALSE(nn::lbl::IsDimmingEnabled());

        {
            nn::vi::Display* displayTv = nullptr;
            float curLuma;
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&displayTv, "External"));
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayCmuLuma(&curLuma, displayTv));
            ASSERT_EQ(.0f, curLuma);
            nn::vi::CloseDisplay(displayTv);
        }

        nn::TimeSpan sleepRequestEventSignaledTime = TimeSpanNever;
        nn::TimeSpan dimmingTriggeredTime = TimeSpanNever;
        bool wasSleepRequestEventSignaled = false;
        bool wasDimmingTriggered = false;

        nn::TimeSpan elapsedIdleTime = 0;
        while ( elapsedIdleTime < TimeSpanNever )
        {
            elapsedIdleTime += UpdateIdleHandlerInterval;

            // 入力エミュレーション間隔が指定されている場合、その間隔で入力があったことにする
            bool isInputDetected = false;
            if ( 0 < firstInputInSeconds &&
                 firstInputInSeconds <= elapsedIdleTime.GetSeconds() &&
                 (elapsedIdleTime.GetSeconds() - firstInputInSeconds) % inputIntervalInSeconds == 0 )
            {
                isInputDetected = true;
                nn::hid::system::InputSourceState state = { nn::os::ConvertToTick(elapsedIdleTime).GetInt64Value() };
                nnt::hid::system::SetInputSourceState(state);
            }

            nn::idle::server::UpdateHandlerStatus(UpdateIdleHandlerInterval);

            if ( isInputDetected )
            {
                // 入力があったフレームの直後は解除されていなければならない
                EXPECT_FALSE(nn::idle::server::IsAutoPowerDownOn());
                EXPECT_FALSE(nn::idle::server::IsDimmingOn());
            }

            // 自動スリープ発動状態のテスト
            if ( nn::idle::server::IsAutoPowerDownOn() )
            {
                bool wasSignaled = nn::os::TryWaitSystemEvent(pAutoSleepRequestEvent);
                if ( !wasSleepRequestEventSignaled )
                {
                    EXPECT_TRUE(wasSignaled); // 一回 TryWait 成功したらクリアされるので、このテストはここで一回だけ
                    wasSleepRequestEventSignaled = true;
                    sleepRequestEventSignaledTime = elapsedIdleTime;
                }
            }
            else
            {
                EXPECT_FALSE(nn::os::TryWaitSystemEvent(pAutoSleepRequestEvent)); // こっちは毎フレームチェックしてよい
            }

            // 自動低輝度化発動状態のテスト
            if ( nn::idle::server::IsDimmingOn() )
            {
                EXPECT_TRUE(nn::lbl::IsDimmingEnabled());

                nn::vi::Display* displayTv = nullptr;
                float curLuma;
                NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&displayTv, "External"));
                NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayCmuLuma(&curLuma, displayTv));
                EXPECT_GT(0.0f, curLuma); // settings に依存しないよう、 0 未満なら成功扱いにする
                nn::vi::CloseDisplay(displayTv);

                if ( !wasDimmingTriggered )
                {
                    wasDimmingTriggered = true;
                    dimmingTriggeredTime = elapsedIdleTime;
                }
            }
            else
            {
                EXPECT_FALSE(nn::lbl::IsDimmingEnabled());

                nn::vi::Display* displayTv = nullptr;
                float curLuma;
                NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&displayTv, "External"));
                NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayCmuLuma(&curLuma, displayTv));
                EXPECT_EQ(0.0f, curLuma);
                nn::vi::CloseDisplay(displayTv);
            }
        }

        if ( expectedAutoSleepTime == TimeSpanNever )
        {
            EXPECT_FALSE(wasSleepRequestEventSignaled);
        }
        else
        {
            EXPECT_TRUE(wasSleepRequestEventSignaled);
            EXPECT_EQ(expectedAutoSleepTime.GetSeconds(), sleepRequestEventSignaledTime.GetSeconds());
        }

        if ( expectedDimmingTime == TimeSpanNever )
        {
            EXPECT_FALSE(wasDimmingTriggered);
        }
        else
        {
            EXPECT_TRUE(wasDimmingTriggered);
            EXPECT_EQ(expectedDimmingTime.GetSeconds(), dimmingTriggeredTime.GetSeconds());
        }

        nn::idle::server::DisableHandlers();
    }

    void TestTriggerTimeWithNoInput(const nn::TimeSpan& expectedAutoSleepTime, const nn::TimeSpan& expectedDimmingTime, const HandlingContext& handlingContext)
    {
        TestTriggerTimeWithPeriodicInput(expectedAutoSleepTime, expectedDimmingTime, handlingContext, 0, 0); // No input
    }

} // anonymous namespace

// --------------------------------------------------------------------------------------------
// 本体設定ごとに正しいポリシーがセットされるかをテストするシリーズ

class testIdle_Settings : public ::testing::TestWithParam<std::pair<SettingsParam, ExpectedResults>>
{};

TEST_P(testIdle_Settings, CheckTriggerTime)
{
    SetSettings(GetParam().first);

    NN_LOG("Handheld / non-MediaPlayback\n");
    TestTriggerTimeWithNoInput(
        GetParam().second.sleepTimeInHandheld,
        GetParam().second.dimTimeInHandheld,
        {
            nn::os::Tick(0), 0, 0, 0, 0,
            nn::idle::IdleTimeDetectionExtension_Disabled,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Handheld,
            { 0 },
            false, false, // 特殊設定はなし
            false, // isInMediaPlayback
        }
    );

    NN_LOG("Handheld / MediaPlayback\n");
    TestTriggerTimeWithNoInput(
        GetParam().second.sleepTimeInHandheldPlayingMedia,
        GetParam().second.dimTimeInHandheldPlayingMedia,
        {
            nn::os::Tick(0), 0, 0, 0, 0,
            nn::idle::IdleTimeDetectionExtension_Disabled,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Handheld,
            { 0 },
            false, false, // 特殊設定はなし
            true, // isInMediaPlayback
        }
    );

    NN_LOG("Console / non-MediaPlayback\n");
    TestTriggerTimeWithNoInput(
        GetParam().second.sleepTimeInConsole,
        GetParam().second.dimTimeInConsole,
        {
            nn::os::Tick(0), 0, 0, 0, 0,
            nn::idle::IdleTimeDetectionExtension_Disabled,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Console,
            { 0 },
            false, false, // 特殊設定はなし
            false, // isInMediaPlayback
        }
    );

    NN_LOG("Console / MediaPlayback\n");
    TestTriggerTimeWithNoInput(
        GetParam().second.sleepTimeInConsolePlayingMedia,
        GetParam().second.dimTimeInConsolePlayingMedia,
        {
            nn::os::Tick(0), 0, 0, 0, 0,
            nn::idle::IdleTimeDetectionExtension_Disabled,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Console,
            { 0 },
            false, false, // 特殊設定はなし
            true, // isInMediaPlayback
        }
    );
}

// 本体設定と、期待される挙動の組み合わせ
const std::pair<SettingsParam, ExpectedResults> SettingsTestParameterSet[] = {
    std::make_pair<SettingsParam, ExpectedResults>(
        {
            nn::settings::system::HandheldSleepPlan_1Min, // handheldSleepPlan
            nn::settings::system::ConsoleSleepPlan_1Hour, // consoleSleepPlan
            false, // isAutoSleepInMediaPlaybackEnabled
            false, // isTvDimmingEnabled
        },
        {
            nn::TimeSpan::FromSeconds(45), // dimTimeInHandheld
            nn::TimeSpan::FromMinutes(1), // sleepTimeInHandheld
            TimeSpanNever, // dimTimeInConsole
            nn::TimeSpan::FromHours(1), // sleepTimeInConsole
            TimeSpanNever, // dimTimeInHandheldPlayingMedia
            TimeSpanNever, // sleepTimeInHandheldPlayingMedia
            TimeSpanNever, // dimTimeInConsolePlayingMedia
            TimeSpanNever, // sleepTimeInConsolePlayingMedia
        }
    ),
    std::make_pair<SettingsParam, ExpectedResults>(
        {
            nn::settings::system::HandheldSleepPlan_3Min, // handheldSleepPlan
            nn::settings::system::ConsoleSleepPlan_2Hour, // consoleSleepPlan
            false, // isAutoSleepInMediaPlaybackEnabled
            false, // isTvDimmingEnabled
        },
        {
            nn::TimeSpan::FromMinutes(2), // dimTimeInHandheld
            nn::TimeSpan::FromMinutes(3), // sleepTimeInHandheld
            TimeSpanNever, // dimTimeInConsole
            nn::TimeSpan::FromHours(2), // sleepTimeInConsole
            TimeSpanNever, // dimTimeInHandheldPlayingMedia
            TimeSpanNever, // sleepTimeInHandheldPlayingMedia
            TimeSpanNever, // dimTimeInConsolePlayingMedia
            TimeSpanNever, // sleepTimeInConsolePlayingMedia
        }
    ),
    std::make_pair<SettingsParam, ExpectedResults>(
        {
            nn::settings::system::HandheldSleepPlan_5Min, // handheldSleepPlan
            nn::settings::system::ConsoleSleepPlan_3Hour, // consoleSleepPlan
            false, // isAutoSleepInMediaPlaybackEnabled
            false, // isTvDimmingEnabled
        },
        {
            nn::TimeSpan::FromMinutes(4), // dimTimeInHandheld
            nn::TimeSpan::FromMinutes(5), // sleepTimeInHandheld
            TimeSpanNever, // dimTimeInConsole
            nn::TimeSpan::FromHours(3), // sleepTimeInConsole
            TimeSpanNever, // dimTimeInHandheldPlayingMedia
            TimeSpanNever, // sleepTimeInHandheldPlayingMedia
            TimeSpanNever, // dimTimeInConsolePlayingMedia
            TimeSpanNever, // sleepTimeInConsolePlayingMedia
        }
    ),
    std::make_pair<SettingsParam, ExpectedResults>(
        {
            nn::settings::system::HandheldSleepPlan_10Min, // handheldSleepPlan
            nn::settings::system::ConsoleSleepPlan_6Hour, // consoleSleepPlan
            false, // isAutoSleepInMediaPlaybackEnabled
            false, // isTvDimmingEnabled
        },
        {
            nn::TimeSpan::FromMinutes(5), // dimTimeInHandheld
            nn::TimeSpan::FromMinutes(10), // sleepTimeInHandheld
            TimeSpanNever, // dimTimeInConsole
            nn::TimeSpan::FromHours(6), // sleepTimeInConsole
            TimeSpanNever, // dimTimeInHandheldPlayingMedia
            TimeSpanNever, // sleepTimeInHandheldPlayingMedia
            TimeSpanNever, // dimTimeInConsolePlayingMedia
            TimeSpanNever, // sleepTimeInConsolePlayingMedia
        }
    ),
    std::make_pair<SettingsParam, ExpectedResults>(
        {
            nn::settings::system::HandheldSleepPlan_30Min, // handheldSleepPlan
            nn::settings::system::ConsoleSleepPlan_12Hour, // consoleSleepPlan
            false, // isAutoSleepInMediaPlaybackEnabled
            false, // isTvDimmingEnabled
        },
        {
            nn::TimeSpan::FromMinutes(5), // dimTimeInHandheld
            nn::TimeSpan::FromMinutes(30), // sleepTimeInHandheld
            TimeSpanNever, // dimTimeInConsole
            nn::TimeSpan::FromHours(12), // sleepTimeInConsole
            TimeSpanNever, // dimTimeInHandheldPlayingMedia
            TimeSpanNever, // sleepTimeInHandheldPlayingMedia
            TimeSpanNever, // dimTimeInConsolePlayingMedia
            TimeSpanNever, // sleepTimeInConsolePlayingMedia
        }
    ),
    std::make_pair<SettingsParam, ExpectedResults>(
        {
            nn::settings::system::HandheldSleepPlan_Never, // handheldSleepPlan
            nn::settings::system::ConsoleSleepPlan_Never, // consoleSleepPlan
            false, // isAutoSleepInMediaPlaybackEnabled
            false, // isTvDimmingEnabled
        },
        {
            nn::TimeSpan::FromMinutes(5), // dimTimeInHandheld
            TimeSpanNever, // sleepTimeInHandheld
            TimeSpanNever, // dimTimeInConsole
            TimeSpanNever, // sleepTimeInConsole
            TimeSpanNever, // dimTimeInHandheldPlayingMedia
            TimeSpanNever, // sleepTimeInHandheldPlayingMedia
            TimeSpanNever, // dimTimeInConsolePlayingMedia
            TimeSpanNever, // sleepTimeInConsolePlayingMedia
        }
    ),
    std::make_pair<SettingsParam, ExpectedResults>(
        {
            nn::settings::system::HandheldSleepPlan_1Min, // handheldSleepPlan
            nn::settings::system::ConsoleSleepPlan_1Hour, // consoleSleepPlan
            true, // isAutoSleepInMediaPlaybackEnabled
            true, // isTvDimmingEnabled
        },
        {
            nn::TimeSpan::FromSeconds(45), // dimTimeInHandheld
            nn::TimeSpan::FromMinutes(1), // sleepTimeInHandheld
            nn::TimeSpan::FromMinutes(5), // dimTimeInConsole
            nn::TimeSpan::FromHours(1), // sleepTimeInConsole
            TimeSpanNever, // dimTimeInHandheldPlayingMedia
            nn::TimeSpan::FromHours(4), // sleepTimeInHandheldPlayingMedia
            TimeSpanNever, // dimTimeInConsolePlayingMedia
            nn::TimeSpan::FromHours(4), // sleepTimeInConsolePlayingMedia
        }
    ),
    std::make_pair<SettingsParam, ExpectedResults>(
        {
            nn::settings::system::HandheldSleepPlan_3Min, // handheldSleepPlan
            nn::settings::system::ConsoleSleepPlan_2Hour, // consoleSleepPlan
            true, // isAutoSleepInMediaPlaybackEnabled
            true, // isTvDimmingEnabled
        },
        {
            nn::TimeSpan::FromMinutes(2), // dimTimeInHandheld
            nn::TimeSpan::FromMinutes(3), // sleepTimeInHandheld
            nn::TimeSpan::FromMinutes(5), // dimTimeInConsole
            nn::TimeSpan::FromHours(2), // sleepTimeInConsole
            TimeSpanNever, // dimTimeInHandheldPlayingMedia
            nn::TimeSpan::FromHours(4), // sleepTimeInHandheldPlayingMedia
            TimeSpanNever, // dimTimeInConsolePlayingMedia
            nn::TimeSpan::FromHours(4), // sleepTimeInConsolePlayingMedia
        }
    ),
    std::make_pair<SettingsParam, ExpectedResults>(
        {
            nn::settings::system::HandheldSleepPlan_5Min, // handheldSleepPlan
            nn::settings::system::ConsoleSleepPlan_3Hour, // consoleSleepPlan
            true, // isAutoSleepInMediaPlaybackEnabled
            true, // isTvDimmingEnabled
        },
        {
            nn::TimeSpan::FromMinutes(4), // dimTimeInHandheld
            nn::TimeSpan::FromMinutes(5), // sleepTimeInHandheld
            nn::TimeSpan::FromMinutes(5), // dimTimeInConsole
            nn::TimeSpan::FromHours(3), // sleepTimeInConsole
            TimeSpanNever, // dimTimeInHandheldPlayingMedia
            nn::TimeSpan::FromHours(4), // sleepTimeInHandheldPlayingMedia
            TimeSpanNever, // dimTimeInConsolePlayingMedia
            nn::TimeSpan::FromHours(4), // sleepTimeInConsolePlayingMedia
        }
    ),
    std::make_pair<SettingsParam, ExpectedResults>(
        {
            nn::settings::system::HandheldSleepPlan_10Min, // handheldSleepPlan
            nn::settings::system::ConsoleSleepPlan_6Hour, // consoleSleepPlan
            true, // isAutoSleepInMediaPlaybackEnabled
            true, // isTvDimmingEnabled
        },
        {
            nn::TimeSpan::FromMinutes(5), // dimTimeInHandheld
            nn::TimeSpan::FromMinutes(10), // sleepTimeInHandheld
            nn::TimeSpan::FromMinutes(5), // dimTimeInConsole
            nn::TimeSpan::FromHours(6), // sleepTimeInConsole
            TimeSpanNever, // dimTimeInHandheldPlayingMedia
            nn::TimeSpan::FromHours(4), // sleepTimeInHandheldPlayingMedia
            TimeSpanNever, // dimTimeInConsolePlayingMedia
            TimeSpanNever, // sleepTimeInConsolePlayingMedia
        }
    ),
    std::make_pair<SettingsParam, ExpectedResults>(
        {
            nn::settings::system::HandheldSleepPlan_30Min, // handheldSleepPlan
            nn::settings::system::ConsoleSleepPlan_12Hour, // consoleSleepPlan
            true, // isAutoSleepInMediaPlaybackEnabled
            true, // isTvDimmingEnabled
        },
        {
            nn::TimeSpan::FromMinutes(5), // dimTimeInHandheld
            nn::TimeSpan::FromMinutes(30), // sleepTimeInHandheld
            nn::TimeSpan::FromMinutes(5), // dimTimeInConsole
            nn::TimeSpan::FromHours(12), // sleepTimeInConsole
            TimeSpanNever, // dimTimeInHandheldPlayingMedia
            nn::TimeSpan::FromHours(4), // sleepTimeInHandheldPlayingMedia
            TimeSpanNever, // dimTimeInConsolePlayingMedia
            TimeSpanNever, // sleepTimeInConsolePlayingMedia
        }
    ),
    std::make_pair<SettingsParam, ExpectedResults>(
        {
            nn::settings::system::HandheldSleepPlan_Never, // handheldSleepPlan
            nn::settings::system::ConsoleSleepPlan_Never, // consoleSleepPlan
            true, // isAutoSleepInMediaPlaybackEnabled
            true, // isTvDimmingEnabled
        },
        {
            nn::TimeSpan::FromMinutes(5), // dimTimeInHandheld
            TimeSpanNever, // sleepTimeInHandheld
            nn::TimeSpan::FromMinutes(5), // dimTimeInConsole
            TimeSpanNever, // sleepTimeInConsole
            TimeSpanNever, // dimTimeInHandheldPlayingMedia
            TimeSpanNever, // sleepTimeInHandheldPlayingMedia
            TimeSpanNever, // dimTimeInConsolePlayingMedia
            TimeSpanNever, // sleepTimeInConsolePlayingMedia
        }
    )
};

INSTANTIATE_TEST_CASE_P(
    testIdle_SettingsInst,
    testIdle_Settings,
    ::testing::ValuesIn(SettingsTestParameterSet)
);

// --------------------------------------------------------------------------------------------
// 一時的に設定をオーバーライドする機能のテスト

TEST(testIdle_PolicyOverride, CheckTriggerTimeWithTriggerTimeOverride)
{
    // 適当な本体設定を使用
    SetSettings({
        nn::settings::system::HandheldSleepPlan_3Min, // handheldSleepPlan
        nn::settings::system::ConsoleSleepPlan_3Hour, // consoleSleepPlan
        true, // isAutoSleepInMediaPlaybackEnabled
        true, // isTvDimmingEnabled
    });

    const int ExpectedOverrideAutoSleepTimeInHandheldInSeconds = 10;
    const int ExpectedOverrideAutoSleepTimeInConsoleInSeconds = 15;
    const int ExpectedOverrideDimmingTimeInHandheldInSeconds = 5;
    const int ExpectedOverrideDimmingTimeInConsoleInSeconds = 8;

    NN_LOG("Handheld / non-MediaPlayback\n");
    TestTriggerTimeWithNoInput(
        nn::TimeSpan::FromSeconds(ExpectedOverrideAutoSleepTimeInHandheldInSeconds),
        nn::TimeSpan::FromSeconds(ExpectedOverrideDimmingTimeInHandheldInSeconds),
        {
            nn::os::Tick(0),
            ExpectedOverrideAutoSleepTimeInHandheldInSeconds,
            ExpectedOverrideAutoSleepTimeInConsoleInSeconds,
            ExpectedOverrideDimmingTimeInHandheldInSeconds,
            ExpectedOverrideDimmingTimeInConsoleInSeconds,
            nn::idle::IdleTimeDetectionExtension_Disabled,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Handheld,
            { 0 },
            false, false, // 一時無効化はなし
            false, // isInMediaPlayback
        }
    );

    NN_LOG("Handheld / MediaPlayback\n");
    TestTriggerTimeWithNoInput(
        nn::TimeSpan::FromSeconds(ExpectedOverrideAutoSleepTimeInHandheldInSeconds),
        nn::TimeSpan::FromSeconds(ExpectedOverrideDimmingTimeInHandheldInSeconds),
        {
            nn::os::Tick(0),
            ExpectedOverrideAutoSleepTimeInHandheldInSeconds,
            ExpectedOverrideAutoSleepTimeInConsoleInSeconds,
            ExpectedOverrideDimmingTimeInHandheldInSeconds,
            ExpectedOverrideDimmingTimeInConsoleInSeconds,
            nn::idle::IdleTimeDetectionExtension_Disabled,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Handheld,
            { 0 },
            false, false, // 一時無効化はなし
            true, // isInMediaPlayback
        }
    );

    NN_LOG("Console / non-MediaPlayback\n");
    TestTriggerTimeWithNoInput(
        nn::TimeSpan::FromSeconds(ExpectedOverrideAutoSleepTimeInConsoleInSeconds),
        nn::TimeSpan::FromSeconds(ExpectedOverrideDimmingTimeInConsoleInSeconds),
        {
            nn::os::Tick(0),
            ExpectedOverrideAutoSleepTimeInHandheldInSeconds,
            ExpectedOverrideAutoSleepTimeInConsoleInSeconds,
            ExpectedOverrideDimmingTimeInHandheldInSeconds,
            ExpectedOverrideDimmingTimeInConsoleInSeconds,
            nn::idle::IdleTimeDetectionExtension_Disabled,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Console,
            { 0 },
            false, false, // 一時無効化はなし
            false, // isInMediaPlayback
        }
    );

    NN_LOG("Console / MediaPlayback\n");
    TestTriggerTimeWithNoInput(
        nn::TimeSpan::FromSeconds(ExpectedOverrideAutoSleepTimeInConsoleInSeconds),
        nn::TimeSpan::FromSeconds(ExpectedOverrideDimmingTimeInConsoleInSeconds),
        {
            nn::os::Tick(0),
            ExpectedOverrideAutoSleepTimeInHandheldInSeconds,
            ExpectedOverrideAutoSleepTimeInConsoleInSeconds,
            ExpectedOverrideDimmingTimeInHandheldInSeconds,
            ExpectedOverrideDimmingTimeInConsoleInSeconds,
            nn::idle::IdleTimeDetectionExtension_Disabled,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Console,
            { 0 },
            false, false, // 一時無効化はなし
            true, // isInMediaPlayback
        }
    );
}

// --------------------------------------------------------------------------------------------
// 一時無効化機能のテスト

TEST(testIdle_PolicyOverride, CheckTriggerTimeWithDisable)
{
    // 適当な本体設定を使用
    SetSettings({
        nn::settings::system::HandheldSleepPlan_1Min, // handheldSleepPlan
        nn::settings::system::ConsoleSleepPlan_1Hour, // consoleSleepPlan
        false, // isAutoSleepInMediaPlaybackEnabled
        true, // isTvDimmingEnabled
    });

    NN_LOG("Handheld / non-MediaPlayback\n");
    TestTriggerTimeWithNoInput(
        TimeSpanNever,
        nn::TimeSpan::FromSeconds(45), // 本体設定に沿うはず
        {
            nn::os::Tick(0),
            0, 0, 0, 0, // 一時時間変更はなし
            nn::idle::IdleTimeDetectionExtension_Disabled,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Handheld,
            { 0 },
            true, false, // 自動スリープのみ一時無効化
            false, // isInMediaPlayback
        }
    );
    TestTriggerTimeWithNoInput(
        nn::TimeSpan::FromMinutes(1), // 本体設定に沿うはず
        TimeSpanNever,
        {
            nn::os::Tick(0),
            0, 0, 0, 0, // 一時時間変更はなし
            nn::idle::IdleTimeDetectionExtension_Disabled,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Handheld,
            { 0 },
            false, true, // 低輝度化だけ一時無効化
            false, // isInMediaPlayback
        }
    );
    TestTriggerTimeWithNoInput(
        TimeSpanNever,
        TimeSpanNever,
        {
            nn::os::Tick(0),
            0, 0, 0, 0, // 一時時間変更はなし
            nn::idle::IdleTimeDetectionExtension_Disabled,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Handheld,
            { 0 },
            true, true, // 両方一時無効化
            false, // isInMediaPlayback
        }
    );

    NN_LOG("Console / non-MediaPlayback\n");
    TestTriggerTimeWithNoInput(
        TimeSpanNever,
        nn::TimeSpan::FromMinutes(5), // 本体設定に沿うはず
        {
            nn::os::Tick(0),
            0, 0, 0, 0, // 一時時間変更はなし
            nn::idle::IdleTimeDetectionExtension_Disabled,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Console,
            { 0 },
            true, false, // 自動スリープのみ一時無効化
            false, // isInMediaPlayback
        }
    );
    TestTriggerTimeWithNoInput(
        nn::TimeSpan::FromHours(1), // 本体設定に沿うはず
        TimeSpanNever,
        {
            nn::os::Tick(0),
            0, 0, 0, 0, // 一時時間変更はなし
            nn::idle::IdleTimeDetectionExtension_Disabled,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Console,
            { 0 },
            false, true, // 低輝度化だけ一時無効化
            false, // isInMediaPlayback
        }
    );
    TestTriggerTimeWithNoInput(
        TimeSpanNever,
        TimeSpanNever,
        {
            nn::os::Tick(0),
            0, 0, 0, 0, // 一時時間変更はなし
            nn::idle::IdleTimeDetectionExtension_Disabled,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Console,
            { 0 },
            true, true, // 両方一時無効化
            false, // isInMediaPlayback
        }
    );
}

// --------------------------------------------------------------------------------------------
// 一時無効化機能の終了時に、すでに発動時間が来ていたら即発動することのテスト
TEST(testIdle_PolicyOverride, CheckTriggerAtDisableEnd)
{
    // 適当な本体設定を使用
    SetSettings({
        nn::settings::system::HandheldSleepPlan_1Min, // handheldSleepPlan
        nn::settings::system::ConsoleSleepPlan_1Hour, // consoleSleepPlan
        true, // isAutoSleepInMediaPlaybackEnabled
        true, // isTvDimmingEnabled
    });

    const HandlingContext HandlingContextBase =
    {
        nn::os::Tick(0),
        0, 0, 0, 0, // 一時時間変更はなし
        nn::idle::IdleTimeDetectionExtension_Disabled,
        AllInputSourceIdSet,
        nn::omm::OperationMode::Handheld,
        { 0 },
        false, false,
        false, // isInMediaPlayback
    };

    nn::TimeSpan apdTriggerTime;
    nn::TimeSpan dimmingTriggerTime;

    // 携帯モード & 自動スリープ無効化状態で数分放置してから無効化を解除すると即スリープする
    nn::idle::server::InitializeHandlerManager();
    nn::idle::server::EnableHandlers();
    {
        const auto PointEndApdDisabledSection = nn::TimeSpan::FromMinutes(10);
        const auto PointEndTest = PointEndApdDisabledSection + nn::TimeSpan::FromMinutes(1);
        const auto PointExpectedDimTriggeredTime = nn::TimeSpan::FromSeconds(45);
        const auto PointExpectedApdTriggeredTime = PointEndApdDisabledSection;

        auto hc = HandlingContextBase;

        // 自動スリープを無効にして 10m （経過する直前まで）再生
        // 低輝度化は当初ポリシー通りに発生するが自動スリープはまだ発生しない
        hc.isAutoSleepDisabled = true;
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(hc));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, 0, PointEndApdDisabledSection - UpdateIdleHandlerInterval);
        EXPECT_EQ(PointExpectedDimTriggeredTime.GetSeconds(), dimmingTriggerTime.GetSeconds());
        EXPECT_EQ(TimeSpanNever.GetSeconds(), apdTriggerTime.GetSeconds());

        // 自動スリープ無効化を解除すると即スリープ
        hc.isAutoSleepDisabled = false;
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(hc));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, PointEndApdDisabledSection - UpdateIdleHandlerInterval, PointEndTest);
        EXPECT_EQ(PointExpectedApdTriggeredTime.GetSeconds(), apdTriggerTime.GetSeconds());
    }
    nn::idle::server::DisableHandlers();

    // 携帯モード & 低輝度化無効化状態で数分放置してから無効化を解除すると即低輝度化する
    nn::idle::server::InitializeHandlerManager();
    nn::idle::server::EnableHandlers();
    {
        const auto PointEndDimDisabledSection = nn::TimeSpan::FromSeconds(50);
        const auto PointEndTest = PointEndDimDisabledSection + nn::TimeSpan::FromMinutes(1);
        const auto PointExpectedDimTriggeredTime = PointEndDimDisabledSection;
        const auto PointExpectedApdTriggeredTime = nn::TimeSpan::FromMinutes(1);

        auto hc = HandlingContextBase;

        // 低輝度化無効化にして 50s （経過する直前まで）再生
        hc.isDimmingDisabled = true;
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(hc));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, 0, PointEndDimDisabledSection - UpdateIdleHandlerInterval);
        EXPECT_EQ(TimeSpanNever.GetSeconds(), dimmingTriggerTime.GetSeconds());
        EXPECT_EQ(TimeSpanNever.GetSeconds(), apdTriggerTime.GetSeconds());

        // 低輝度化無効化を解除すると即低輝度化
        hc.isDimmingDisabled = false;
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(hc));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, PointEndDimDisabledSection - UpdateIdleHandlerInterval, PointEndTest);
        EXPECT_EQ(PointExpectedDimTriggeredTime.GetSeconds(), dimmingTriggerTime.GetSeconds());
        EXPECT_EQ(PointExpectedApdTriggeredTime.GetSeconds(), apdTriggerTime.GetSeconds());
    }
    nn::idle::server::DisableHandlers();
}

// --------------------------------------------------------------------------------------------
// 発動遅延機能のテスト

struct DetectionExtensionTestParam
{
    nn::TimeSpan expectedMaxApdTriggerTime;
    nn::TimeSpan expectedMaxDimTriggerTime;
    nn::idle::IdleTimeDetectionExtension extensionMode;
};

class testIdle_DetectionExtension : public ::testing::TestWithParam<DetectionExtensionTestParam>
{};

TEST_P(testIdle_DetectionExtension, CheckTriggerTimeWithAutoSleepDelay)
{
    auto expectedApdtime = [this](nn::TimeSpan apdTimePerSetting) {
        return (apdTimePerSetting > GetParam().expectedMaxApdTriggerTime) ?
            apdTimePerSetting : GetParam().expectedMaxApdTriggerTime;
    };
    auto expectedDimtime = [this](nn::TimeSpan dimTimePerSetting) {
        return (dimTimePerSetting > GetParam().expectedMaxDimTriggerTime) ?
            dimTimePerSetting : GetParam().expectedMaxDimTriggerTime;
    };

    SetSettings({
        nn::settings::system::HandheldSleepPlan_10Min, // handheldSleepPlan
        nn::settings::system::ConsoleSleepPlan_2Hour, // consoleSleepPlan
        true, // isAutoSleepInMediaPlaybackEnabled
        true, // isTvDimmingEnabled
    });

    NN_LOG("Handheld / non-MediaPlayback\n");
    TestTriggerTimeWithNoInput(
        expectedApdtime(nn::TimeSpan::FromMinutes(10)),
        expectedDimtime(nn::TimeSpan::FromMinutes(5)),
        {
            nn::os::Tick(0),
            0, 0, 0, 0, // 一時時間変更はなし
            GetParam().extensionMode,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Handheld,
            { 0 },
            false, false, // 一時無効化はなし
            false, // isInMediaPlayback
        }
    );

    NN_LOG("Handheld / MediaPlayback\n");
    TestTriggerTimeWithNoInput(
        expectedApdtime(nn::TimeSpan::FromHours(4)),
        expectedDimtime(TimeSpanNever),
        {
            nn::os::Tick(0),
            0, 0, 0, 0, // 一時時間変更はなし
            GetParam().extensionMode,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Handheld,
            { 0 },
            false, false, // 一時無効化はなし
            true, // isInMediaPlayback
        }
    );

    NN_LOG("Console / non-MediaPlayback\n");
    TestTriggerTimeWithNoInput(
        expectedApdtime(nn::TimeSpan::FromHours(2)),
        expectedDimtime(nn::TimeSpan::FromMinutes(5)),
        {
            nn::os::Tick(0),
            0, 0, 0, 0, // 一時時間変更はなし
            GetParam().extensionMode,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Console,
            { 0 },
            false, false, // 一時無効化はなし
            false, // isInMediaPlayback
        }
    );

    NN_LOG("Console / MediaPlayback\n");
    TestTriggerTimeWithNoInput(
        expectedApdtime(nn::TimeSpan::FromHours(4)),
        expectedDimtime(TimeSpanNever),
        {
            nn::os::Tick(0),
            0, 0, 0, 0, // 一時時間変更はなし
            GetParam().extensionMode,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Console,
            { 0 },
            false, false, // 一時無効化はなし
            true, // isInMediaPlayback
        }
    );

    // 別の本体設定（Never が含まれる設定）でもテスト
    SetSettings({
        nn::settings::system::HandheldSleepPlan_1Min, // handheldSleepPlan
        nn::settings::system::ConsoleSleepPlan_1Hour, // consoleSleepPlan
        false, // isAutoSleepInMediaPlaybackEnabled
        true, // isTvDimmingEnabled
    });

    NN_LOG("Handheld / MediaPlayback\n");
    TestTriggerTimeWithNoInput(
        expectedApdtime(TimeSpanNever),
        expectedDimtime(TimeSpanNever),
        {
            nn::os::Tick(0),
            0, 0, 0, 0, // 一時時間変更はなし
            GetParam().extensionMode,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Handheld,
            { 0 },
            false, false, // 一時無効化はなし
            true, // isInMediaPlayback
        }
    );
}

const DetectionExtensionTestParam DetectionExtensionTestParamSet[] = {
    {
        nn::TimeSpan::FromMinutes(30), // expectedMaxApdTriggerTime
        nn::TimeSpan::FromMinutes(15), // expectedMaxDimTriggerTime
        nn::idle::IdleTimeDetectionExtension_Enabled, // expectedMaxApdTriggerTime
    },
    {
        nn::TimeSpan::FromMinutes(60), // expectedMaxApdTriggerTime
        TimeSpanNever, // expectedMaxDimTriggerTime
        nn::idle::IdleTimeDetectionExtension_EnabledUnsafe, // expectedMaxApdTriggerTime
    },
};

INSTANTIATE_TEST_CASE_P(
    testIdle_DetectionExtensionInst,
    testIdle_DetectionExtension,
    ::testing::ValuesIn(DetectionExtensionTestParamSet)
);

// --------------------------------------------------------------------------------------------
// InputDetector からの入力を正しくハンドリングしていることのテスト

TEST(testIdle_InputDetection, Basic)
{
    // 適当な本体設定を使用
    SetSettings({
        nn::settings::system::HandheldSleepPlan_1Min, // handheldSleepPlan
        nn::settings::system::ConsoleSleepPlan_1Hour, // consoleSleepPlan
        true, // isAutoSleepInMediaPlaybackEnabled
        true, // isTvDimmingEnabled
    });

    // ずっと入力していれば寝ない
    TestTriggerTimeWithPeriodicInput(
        TimeSpanNever,
        TimeSpanNever,
        {
            nn::os::Tick(0), 0, 0, 0, 0,
            nn::idle::IdleTimeDetectionExtension_Disabled,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Handheld,
            { 0 },
            false, false, // 特殊設定はなし
            false, // isInMediaPlayback
        },
        1, 1 // 1 秒後から 1 秒に一回入力
    );

    // 低輝度化だけは発動するが、その後の入力で解除されるパターン
    TestTriggerTimeWithPeriodicInput(
        TimeSpanNever,
        nn::TimeSpan::FromSeconds(45),
        {
            nn::os::Tick(0), 0, 0, 0, 0,
            nn::idle::IdleTimeDetectionExtension_Disabled,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Handheld,
            { 0 },
            false, false, // 特殊設定はなし
            false, // isInMediaPlayback
        },
        50, 12 // 50 秒後から 12 秒に一回入力
    );

    // 途中の入力で発動は遅れるが、入力がなくなってちゃんと発動するパターン
    TestTriggerTimeWithPeriodicInput(
        nn::TimeSpan::FromSeconds(60 + 10), // 最初の入力タイミングの分だけ発動が遅れる
        nn::TimeSpan::FromSeconds(45 + 10), // 最初の入力タイミングの分だけ発動が遅れる
        {
            nn::os::Tick(0), 0, 0, 0, 0,
            nn::idle::IdleTimeDetectionExtension_Disabled,
            AllInputSourceIdSet,
            nn::omm::OperationMode::Handheld,
            { 0 },
            false, false, // 特殊設定はなし
            false, // isInMediaPlayback
        },
        10, (2 * 60 * 60) // 10 秒後から 2 時間に一回入力
    );
}

// --------------------------------------------------------------------------------------------
// メディアプレイバック期間の扱いに関する固有の仕様のテスト

TEST(testIdle_MediaPlayback, ChainedMediaPlayConsole)
{
    // 適当な本体設定を使用
    SetSettings({
        nn::settings::system::HandheldSleepPlan_1Min, // handheldSleepPlan
        nn::settings::system::ConsoleSleepPlan_1Hour, // consoleSleepPlan
        true, // isAutoSleepInMediaPlaybackEnabled
        true, // isTvDimmingEnabled
    });

    const HandlingContext NoPlaybackContextConsoleBase =
    {
        nn::os::Tick(0), 0, 0, 0, 0,
        nn::idle::IdleTimeDetectionExtension_Disabled,
        AllInputSourceIdSet,
        nn::omm::OperationMode::Console, // 1時間でスリープ発動
        { 0 },
        false, false, // 特殊設定はなし
        false, // isInMediaPlayback
    };
    const HandlingContext PlaybackContextConsoleBase =
    {
        nn::os::Tick(0), 0, 0, 0, 0,
        nn::idle::IdleTimeDetectionExtension_Disabled,
        AllInputSourceIdSet,
        nn::omm::OperationMode::Console, // 1時間でスリープ発動
        { 0 },
        false, false, // 特殊設定はなし
        true, // isInMediaPlayback
    };
    auto nopc = NoPlaybackContextConsoleBase;
    auto pc = PlaybackContextConsoleBase;

    nn::TimeSpan apdTriggerTime;
    nn::TimeSpan dimmingTriggerTime;

    // 5m 未満の間隔で再生し続けていると、低輝度化が発生しない & 4h のところでスリープする
    nn::idle::server::InitializeHandlerManager();
    nn::idle::server::EnableHandlers();
    {
        const nn::TimeSpan Point1stPlayEnd = nn::TimeSpan::FromMinutes(3 * 60 + 46);
        const nn::TimeSpan Point2ndPlayBegin = nn::TimeSpan::FromMinutes(3 * 60 + 50);
        const nn::TimeSpan PointApdTriggered = nn::TimeSpan::FromHours(4);

        // 3h46m 再生
        pc.timeMediaPlaybackFinished = nn::os::Tick(0);
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(pc));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, 0, Point1stPlayEnd);
        EXPECT_EQ(TimeSpanNever.GetSeconds(), dimmingTriggerTime.GetSeconds());
        EXPECT_EQ(TimeSpanNever.GetSeconds(), apdTriggerTime.GetSeconds());

        // 4m 再生停止
        nopc.timeMediaPlaybackFinished = nn::os::ConvertToTick(Point1stPlayEnd);
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(nopc));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, Point1stPlayEnd, Point2ndPlayBegin);
        EXPECT_EQ(TimeSpanNever.GetSeconds(), dimmingTriggerTime.GetSeconds());
        EXPECT_EQ(TimeSpanNever.GetSeconds(), apdTriggerTime.GetSeconds());

        // 1h 再生するが 10m 経ったところでスリープする
        pc.timeMediaPlaybackFinished = nn::os::ConvertToTick(Point1stPlayEnd);
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(pc));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, Point2ndPlayBegin, PointApdTriggered);
        EXPECT_EQ(TimeSpanNever.GetSeconds(), dimmingTriggerTime.GetSeconds());
        EXPECT_EQ(PointApdTriggered.GetSeconds(), apdTriggerTime.GetSeconds());
    }
    nn::idle::server::DisableHandlers();

    // メディア再生終了後元の設定に沿ってスリープする
    nn::idle::server::InitializeHandlerManager();
    nn::idle::server::EnableHandlers();
    {
        const nn::TimeSpan Point1stPlayEnd = nn::TimeSpan::FromHours(1);
        const nn::TimeSpan PointDimTriggered = nn::TimeSpan::FromMinutes(1 * 60 + 5);
        const nn::TimeSpan PointApdTriggered = nn::TimeSpan::FromMinutes(1 * 60 + 5);

        // 1h 再生
        pc.timeMediaPlaybackFinished = nn::os::Tick(0);
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(pc));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, 0, Point1stPlayEnd);
        EXPECT_EQ(TimeSpanNever.GetSeconds(), dimmingTriggerTime.GetSeconds());
        EXPECT_EQ(TimeSpanNever.GetSeconds(), apdTriggerTime.GetSeconds());

        // 再生停止
        // 5 分保留期間が満了した瞬間に元のポリシーに戻り、スリープ
        nopc.timeMediaPlaybackFinished = nn::os::ConvertToTick(Point1stPlayEnd);
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(nopc));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, Point1stPlayEnd, PointApdTriggered);
        EXPECT_EQ(PointDimTriggered.GetSeconds(), dimmingTriggerTime.GetSeconds());
        EXPECT_EQ(PointApdTriggered.GetSeconds(), apdTriggerTime.GetSeconds());
    }
    nn::idle::server::DisableHandlers();
}

TEST(testIdle_MediaPlayback, ChainedMediaPlayHandheld)
{
    // 適当な本体設定を使用
    SetSettings({
        nn::settings::system::HandheldSleepPlan_1Min, // handheldSleepPlan
        nn::settings::system::ConsoleSleepPlan_1Hour, // consoleSleepPlan
        true, // isAutoSleepInMediaPlaybackEnabled
        true, // isTvDimmingEnabled
    });

    const HandlingContext NoPlaybackContextHandheldBase =
    {
        nn::os::Tick(0), 0, 0, 0, 0,
        nn::idle::IdleTimeDetectionExtension_Disabled,
        AllInputSourceIdSet,
        nn::omm::OperationMode::Handheld, // 1m でスリープ発動
        { 0 },
        false, false, // 特殊設定はなし
        false, // isInMediaPlayback
    };
    const HandlingContext PlaybackContextHandheldBase =
    {
        nn::os::Tick(0), 0, 0, 0, 0,
        nn::idle::IdleTimeDetectionExtension_Disabled,
        AllInputSourceIdSet,
        nn::omm::OperationMode::Handheld, // 1m でスリープ発動
        { 0 },
        false, false, // 特殊設定はなし
        true, // isInMediaPlayback
    };
    auto nopc = NoPlaybackContextHandheldBase;
    auto pc = PlaybackContextHandheldBase;

    nn::TimeSpan apdTriggerTime;
    nn::TimeSpan dimmingTriggerTime;

    // 5m 未満の間隔で再生し続けていると、低輝度化が発生しない & 4h のところでスリープする
    nn::idle::server::InitializeHandlerManager();
    nn::idle::server::EnableHandlers();
    {
        const nn::TimeSpan Point1stPlayEnd = nn::TimeSpan::FromMinutes(3 * 60 + 46);
        const nn::TimeSpan Point2ndPlayBegin = nn::TimeSpan::FromMinutes(3 * 60 + 50);
        const nn::TimeSpan PointApdTriggered = nn::TimeSpan::FromHours(4);

        // 3h46m 再生
        pc.timeMediaPlaybackFinished = nn::os::Tick(0);
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(pc));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, 0, Point1stPlayEnd);
        EXPECT_EQ(TimeSpanNever.GetSeconds(), dimmingTriggerTime.GetSeconds());
        EXPECT_EQ(TimeSpanNever.GetSeconds(), apdTriggerTime.GetSeconds());

        // 4m 再生停止
        nopc.timeMediaPlaybackFinished = nn::os::ConvertToTick(Point1stPlayEnd);
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(nopc));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, Point1stPlayEnd, Point2ndPlayBegin);
        EXPECT_EQ(TimeSpanNever.GetSeconds(), dimmingTriggerTime.GetSeconds());
        EXPECT_EQ(TimeSpanNever.GetSeconds(), apdTriggerTime.GetSeconds());

        // 1h 再生するが 10m 経ったところでスリープする
        pc.timeMediaPlaybackFinished = nn::os::ConvertToTick(Point1stPlayEnd);
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(pc));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, Point2ndPlayBegin, PointApdTriggered);
        EXPECT_EQ(TimeSpanNever.GetSeconds(), dimmingTriggerTime.GetSeconds());
        EXPECT_EQ(PointApdTriggered.GetSeconds(), apdTriggerTime.GetSeconds());
    }
    nn::idle::server::DisableHandlers();

    // メディア再生終了後元の設定に沿ってスリープする
    nn::idle::server::InitializeHandlerManager();
    nn::idle::server::EnableHandlers();
    {
        const nn::TimeSpan Point1stPlayEnd = nn::TimeSpan::FromHours(1);
        const nn::TimeSpan PointDimTriggered = nn::TimeSpan::FromMinutes(1 * 60 + 5);
        const nn::TimeSpan PointApdTriggered = nn::TimeSpan::FromMinutes(1 * 60 + 5);

        // 1h 再生
        pc.timeMediaPlaybackFinished = nn::os::Tick(0);
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(pc));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, 0, Point1stPlayEnd);
        EXPECT_EQ(TimeSpanNever, dimmingTriggerTime);
        EXPECT_EQ(TimeSpanNever, apdTriggerTime);

        // 再生停止
        // 5 分保留期間が満了した瞬間に元のポリシーに戻り、スリープ
        nopc.timeMediaPlaybackFinished = nn::os::ConvertToTick(Point1stPlayEnd);
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(nopc));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, Point1stPlayEnd, PointApdTriggered);
        EXPECT_EQ(PointDimTriggered.GetSeconds(), dimmingTriggerTime.GetSeconds());
        EXPECT_EQ(PointApdTriggered.GetSeconds(), apdTriggerTime.GetSeconds());
    }
    nn::idle::server::DisableHandlers();
}

// フォアグラウンドの切り替えによりメディア再生期間外になる場合はメディア再生中ポリシー解除保留期間に入らないテスト
TEST(testIdle_MediaPlayback, CancelAfterglowByForegroundSwitch)
{
    // 適当な本体設定を使用
    SetSettings({
        nn::settings::system::HandheldSleepPlan_1Min, // handheldSleepPlan
        nn::settings::system::ConsoleSleepPlan_1Hour, // consoleSleepPlan
        true, // isAutoSleepInMediaPlaybackEnabled
        true, // isTvDimmingEnabled
    });

    const HandlingContext PlaybackContextHandheldBase =
    {
        nn::os::Tick(0), 0, 0, 0, 0,
        nn::idle::IdleTimeDetectionExtension_Disabled,
        AllInputSourceIdSet,
        nn::omm::OperationMode::Handheld, // 1m でスリープ発動
        { 0 },
        false, false, // 特殊設定はなし
        true, // isInMediaPlayback
    };
    const HandlingContext NoPlaybackContextHandheldOnAnotherApplication =
    {
        nn::os::Tick(0), 0, 0, 0, 0,
        nn::idle::IdleTimeDetectionExtension_Disabled,
        AllInputSourceIdSet,
        nn::omm::OperationMode::Handheld, // 1m でスリープ発動
        { 1 }, // 別の ARUID
        false, false, // 特殊設定はなし
        false, // isInMediaPlayback
    };
    auto pc = PlaybackContextHandheldBase;
    auto nopcOnAnotherApp = NoPlaybackContextHandheldOnAnotherApplication;

    nn::TimeSpan apdTriggerTime;
    nn::TimeSpan dimmingTriggerTime;

    // 以下テストケース本体
    nn::idle::server::InitializeHandlerManager();
    nn::idle::server::EnableHandlers();
    {
        const nn::TimeSpan Point1stPlayEnd = nn::TimeSpan::FromSeconds(30);
        const nn::TimeSpan PointDimTriggered = nn::TimeSpan::FromSeconds(45); // 補正なし
        const nn::TimeSpan PointApdTriggered = nn::TimeSpan::FromMinutes(1); // 補正なし

        // 30s 再生
        pc.timeMediaPlaybackFinished = nn::os::Tick(0);
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(pc));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, 0, Point1stPlayEnd);
        EXPECT_EQ(TimeSpanNever, dimmingTriggerTime);
        EXPECT_EQ(TimeSpanNever, apdTriggerTime);

        // ユーザ操作を伴わず FG 切替
        // この場合はメディア再生中ポリシー解除保留期間に入らないので、補正なく画面焼け軽減が発動する
        nopcOnAnotherApp.timeMediaPlaybackFinished = nn::os::ConvertToTick(Point1stPlayEnd);
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(nopcOnAnotherApp));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, Point1stPlayEnd, PointApdTriggered);
        EXPECT_EQ(PointDimTriggered.GetSeconds(), dimmingTriggerTime.GetSeconds());
        EXPECT_EQ(PointApdTriggered.GetSeconds(), apdTriggerTime.GetSeconds());
    }
    nn::idle::server::DisableHandlers();
}

// メディア再生中ポリシー解除保留期間中にユーザ操作があったら即座に期間が終了するテスト
TEST(testIdle_MediaPlayback, CancelAfterglowByUserInput)
{
    // 適当な本体設定を使用
    SetSettings({
        nn::settings::system::HandheldSleepPlan_1Min, // handheldSleepPlan
        nn::settings::system::ConsoleSleepPlan_1Hour, // consoleSleepPlan
        true, // isAutoSleepInMediaPlaybackEnabled
        true, // isTvDimmingEnabled
    });

    const HandlingContext NoPlaybackContextHandheldBase =
    {
        nn::os::Tick(0), 0, 0, 0, 0,
        nn::idle::IdleTimeDetectionExtension_Disabled,
        AllInputSourceIdSet,
        nn::omm::OperationMode::Handheld, // 1m でスリープ発動
        { 0 },
        false, false, // 特殊設定はなし
        false, // isInMediaPlayback
    };
    const HandlingContext PlaybackContextHandheldBase =
    {
        nn::os::Tick(0), 0, 0, 0, 0,
        nn::idle::IdleTimeDetectionExtension_Disabled,
        AllInputSourceIdSet,
        nn::omm::OperationMode::Handheld, // 1m でスリープ発動
        { 0 },
        false, false, // 特殊設定はなし
        true, // isInMediaPlayback
    };
    auto nopc = NoPlaybackContextHandheldBase;
    auto pc = PlaybackContextHandheldBase;

    nn::TimeSpan apdTriggerTime;
    nn::TimeSpan dimmingTriggerTime;

    // 以下テストケース本体
    nn::idle::server::InitializeHandlerManager();
    nn::idle::server::EnableHandlers();
    {
        const nn::TimeSpan Point1stPlayEnd = nn::TimeSpan::FromSeconds(30);
        const nn::TimeSpan PointUserInput = nn::TimeSpan::FromSeconds(40);
        const nn::TimeSpan PointDimTriggered = nn::TimeSpan::FromSeconds(40 + 45); // 補正なし
        const nn::TimeSpan PointApdTriggered = nn::TimeSpan::FromSeconds(40 + 60); // 補正なし

        // 30s 再生
        pc.timeMediaPlaybackFinished = nn::os::Tick(0);
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(pc));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, 0, Point1stPlayEnd);
        EXPECT_EQ(TimeSpanNever, dimmingTriggerTime);
        EXPECT_EQ(TimeSpanNever, apdTriggerTime);

        // 再生終了して 10s 待機
        nopc.timeMediaPlaybackFinished = nn::os::ConvertToTick(Point1stPlayEnd);
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(nopc));
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, Point1stPlayEnd, PointUserInput);
        EXPECT_EQ(TimeSpanNever, dimmingTriggerTime);
        EXPECT_EQ(TimeSpanNever, apdTriggerTime);

        // ユーザ操作を入れる
        // ユーザ操作があった瞬間にメディア再生中ポリシー解除保留期間が終了する
        nn::hid::system::InputSourceState state = { nn::os::ConvertToTick(PointUserInput).GetInt64Value() };
        nnt::hid::system::SetInputSourceState(state);
        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, PointUserInput, PointApdTriggered);
        EXPECT_EQ(PointDimTriggered.GetSeconds(), dimmingTriggerTime.GetSeconds());
        EXPECT_EQ(PointApdTriggered.GetSeconds(), apdTriggerTime.GetSeconds());
    }
    nn::idle::server::DisableHandlers();
}

// --------------------------------------------------------------------------------------------
// ユーザ操作の明示的宣言 API のテスト

TEST(testIdle_ReportUserIsActive, Basic)
{
    // 適当な本体設定を使用
    SetSettings({
        nn::settings::system::HandheldSleepPlan_1Min, // handheldSleepPlan
        nn::settings::system::ConsoleSleepPlan_1Hour, // consoleSleepPlan
        true, // isAutoSleepInMediaPlaybackEnabled
        true, // isTvDimmingEnabled
    });

    const HandlingContext TypicalHandlngContext =
    {
        nn::os::Tick(0), 0, 0, 0, 0,
        nn::idle::IdleTimeDetectionExtension_Disabled,
        AllInputSourceIdSet,
        nn::omm::OperationMode::Handheld, // 1m でスリープ発動
        { 0 },
        false, false, // 特殊設定はなし
        false, // isInMediaPlayback
    };

    nn::TimeSpan apdTriggerTime;
    nn::TimeSpan dimmingTriggerTime;

    nn::idle::server::InitializeHandlerManager();
    nn::idle::server::EnableHandlers();
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::idle::server::SetHandlingContext(TypicalHandlngContext));

        const nn::TimeSpan PointReportUserIsActive = nn::TimeSpan::FromSeconds(50);
        const nn::TimeSpan PointDimTriggered = PointReportUserIsActive + nn::TimeSpan::FromSeconds(45);
        const nn::TimeSpan PointApdTriggered = PointReportUserIsActive + nn::TimeSpan::FromSeconds(60);

        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, 0, PointReportUserIsActive);
        nn::idle::server::ResetIdleTimeCount(); // nn::idle::ReportUserIsActive() のサーバ側内部実装はこれ

        UpdateHandlers(&apdTriggerTime, &dimmingTriggerTime, PointReportUserIsActive, PointApdTriggered);
        EXPECT_EQ(PointDimTriggered.GetSeconds(), dimmingTriggerTime.GetSeconds());
        EXPECT_EQ(PointApdTriggered.GetSeconds(), apdTriggerTime.GetSeconds());
    }
    nn::idle::server::DisableHandlers();
}
