﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/i2c/i2c.h>
#include <nn/nn_Common.h>

#include <nnt/nntest.h>

#include "../Common/test_ThreadHelper.h"

namespace {

const int                        NumberOfThreads = 2;      // 生成するスレッドの数
const int                        NumberOfCalls   = 2000;  // CI環境向けには短めに値を設定。手元のローカル環境で数億オーダー程度に大きくしてテストする
const nn::i2c::I2cDevice         DeviceName      = nn::i2c::I2cDevice::I2cDevice_ClassicController; // クラコンのデバイス識別子
const nn::Bit8                   SendBuf[2]      = { 0xF0, 0x55 };                     // 適当なデータ
const nn::i2c::TransactionOption InOption        = static_cast<nn::i2c::TransactionOption>( nn::i2c::TransactionOption_StartCondition | nn::i2c::TransactionOption_StopCondition );

nn::i2c::I2cSession              g_Session;
nn::Bit8                         g_ReceiveBuf[8];
nn::os::BarrierType              g_Barrier;
nnt::i2c::util::ThreadHelper     g_ThreadHelper[NumberOfThreads];

void SendThread(void* arg)
{
    nn::os::AwaitBarrier( &g_Barrier );
    for(int i = 0; i < NumberOfCalls; i++)
    {
        auto result = nn::i2c::Send( g_Session, SendBuf, sizeof(SendBuf), InOption);
        ASSERT_TRUE(result.IsSuccess());
    }
}

void ReceiveThread(void* arg)
{
    nn::os::AwaitBarrier( &g_Barrier );
    for(int i = 0; i < NumberOfCalls; i++)
    {
        auto result = nn::i2c::Receive( g_ReceiveBuf, g_Session, sizeof(g_ReceiveBuf), InOption );
        ASSERT_TRUE(result.IsSuccess());
    }
}

// 引数指定された関数を登録して、複数のスレッドで実行する
void ExecuteThreads( nnt::i2c::util::ThreadFunction threadFunction )
{
    for(int i=0; i<NumberOfThreads; i++)
    {
        g_ThreadHelper[i].SetThreadFunction( threadFunction );
        g_ThreadHelper[i].CreateThread(i);
        g_ThreadHelper[i].StartThread();
    }

    for(int i=0; i<NumberOfThreads; i++)
    {
        g_ThreadHelper[i].WaitThread();
    }

    for(int i=0; i<NumberOfThreads; i++)
    {
        g_ThreadHelper[i].DestroyThread();
    }
}

}

namespace nnt { namespace i2c { namespace thread {

//---------------------------------------------------------------------------
//  テスト用関数
//---------------------------------------------------------------------------

bool TEST_Transaction()
{
    NN_LOG("[NOTE] This test needs a classic controller attached to a target board via connector.\nDon't care the result if it's not attached.\n");
    nn::os::InitializeBarrier( &g_Barrier, NumberOfThreads );

    nn::i2c::Initialize();
    nn::i2c::OpenSession( &g_Session, DeviceName );

    ExecuteThreads( SendThread );

    ExecuteThreads( ReceiveThread );

    return true;
}

}}}  // namespace nnt::i2c::thread
