﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>

#include <nn/os.h>

#include <nn/i2c/i2c.h>
#include <nn/util/util_BitPack.h>
#include <nnt/nntest.h>

#include "test_Clsic.h"

namespace {

const int                        ReadIntervalMicroSeconds = 150;  // ReadInterval [us]
const nn::i2c::TransactionOption InOption = static_cast<nn::i2c::TransactionOption>( nn::i2c::TransactionOption_StartCondition | nn::i2c::TransactionOption_StopCondition );
const nn::i2c::I2cDevice         DeviceName               = nn::i2c::I2cDevice::I2cDevice_ClassicController;

// 2 Bytes Send 処理を行う関数(指定したレジスタにデータを書き込む関数に相当します)
void MySend(nn::i2c::I2cSession session, nn::Bit8 reg, nn::Bit8 data)
{
    nn::Bit8 sendBuf[2] = {reg, data};
    auto result = nn::i2c::Send(session, sendBuf, sizeof(sendBuf), InOption);
    ASSERT_TRUE(result.IsSuccess());
}

// length 分のデータを受信する関数
void MyReceive(nn::Bit8* receiveBuf, nn::i2c::I2cSession session, size_t length)
{
    auto result = nn::i2c::Receive( receiveBuf, session, length, InOption );
    ASSERT_TRUE(result.IsSuccess());
}

// コントローラデータを要求する関数(指定したレジスタを始点とする)
void RequestData(nn::i2c::I2cSession session, nn::Bit8 reg)
{
    auto result = nn::i2c::Send(session, &reg, 1, InOption);
    ASSERT_TRUE(result.IsSuccess());
    nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(ReadIntervalMicroSeconds) );
}

// クラシックコントローラの初期化処理を行う関数
void InitializeClassicController(nn::i2c::I2cSession session)
{
    MySend(session, 0xF0, 0x55); // 暗号化を解く処理
    MySend(session, 0xFB, 0x00); // 暗号化を解く処理
    MySend(session, 0xFE, 0x03); // データフォーマット3に設定
}

// (デバッグ用途) 配列を整形して出力する関数

void Dump( nn::Bit8* receiveBuf, int length )
{
    for(int i = 0; i < length; i++ )
    {
        if( i % 8 == 0)
        {
            NN_LOG("0x%02x | ", i );
        }
        NN_LOG("%02x ", receiveBuf[i]);
        if( ( i + 1 ) % 8 == 0 )
        {
            NN_LOG("\n");
        }
    }
    NN_LOG("\n");
}

}

namespace nnt { namespace i2c { namespace clsic {

//---------------------------------------------------------------------------
//  テスト用関数
//---------------------------------------------------------------------------

bool TEST_BasicTransaction( int readTestTimes )
{
    nn::i2c::I2cSession        session;
    nn::Bit8                   receiveBuf[9];
    const nn::Bit8             StartReg = 0x00;

    // I2Cドライバライブラリの初期化
    nn::i2c::Initialize();

    // セッションをオープンして有効な session を取得します
    nn::i2c::OpenSession(&session, DeviceName);

    // クラコンの初期化を行います
    InitializeClassicController( session );

    // readTestTimes 回コントローラデータの取得テストを行います
    for(int i = 0; i < readTestTimes; i++ )
    {
        RequestData(session, StartReg);
        MyReceive(receiveBuf, session, sizeof(receiveBuf));
    }

    // セッションの終了
    nn::i2c::CloseSession(session);

    // ライブラリの終了
    nn::i2c::Finalize();

    return true;
}

bool TEST_AutoIncrementAccess( int autoIncrementTestTimes )
{
    nn::i2c::I2cSession        session;
    const nn::Bit8             StartReg  = 0x00;
    const size_t               BufLength = 0x100;
    nn::Bit8                   receiveBuf[BufLength];

    // I2Cドライバライブラリの初期化
    nn::i2c::Initialize();

    // セッションをオープンして有効な session を取得します
    nn::i2c::OpenSession(&session, DeviceName);

    // クラコンのレジスタ全体にわたってAutoIncrement アクセスを行います
    for(int i = 0; i < autoIncrementTestTimes; i++ )
    {
        InitializeClassicController( session ); // TORIAEZU: 多バイト長読み出しを行うとクラコン側の都合で都度再初期化が必要
        RequestData(session, StartReg);
        MyReceive( receiveBuf, session, BufLength);
        Dump( receiveBuf, BufLength );

    }

    // セッションの終了
    nn::i2c::CloseSession(session);

    // ライブラリの終了
    nn::i2c::Finalize();

    return true;
}

bool TEST_CommandList()
{
    nn::i2c::I2cSession        session;

    const size_t               ReceiveBufLength = 0x100;
    nn::Bit8                   receiveBuf[ReceiveBufLength];

    // I2Cドライバライブラリの初期化
    nn::i2c::Initialize();

    // セッションをオープンして有効な session を取得します
    nn::i2c::OpenSession(&session, DeviceName);

    // クラコンの初期化を行います
    InitializeClassicController( session );

    uint8_t                        sendBuf;
    uint8_t                        commandList[nn::i2c::CommandListLengthCountMax];
    nn::i2c::CommandListFormatter  commandListFormatter(commandList, sizeof(commandList));

    // 0xFF から 1 Byte 分読み出す
    sendBuf = 0xFF;
    commandListFormatter.EnqueueSendCommand( InOption, &sendBuf, 1 );
    commandListFormatter.EnqueueSleepCommand( ReadIntervalMicroSeconds );
    commandListFormatter.EnqueueReceiveCommand( InOption, 1 );

    // 0xFE から 2 Byte 分読み出す
    sendBuf = 0xFE;
    commandListFormatter.EnqueueSendCommand( InOption, &sendBuf, 1 );
    commandListFormatter.EnqueueSleepCommand( ReadIntervalMicroSeconds );
    commandListFormatter.EnqueueReceiveCommand( InOption, 2 );

    // コマンドリスト実行
    auto result = ExecuteCommandList( receiveBuf, ReceiveBufLength, session, commandList, commandListFormatter.GetCurrentLength() );
    NN_ABORT_UNLESS(result.IsSuccess());

    // コマンド結果チェック(ID チェック)
    // reg  | value
    //---------------
    // 0xFE | 0x03
    // 0xFF | 0x01
    NN_ABORT_UNLESS( receiveBuf[0] == 0x01 && receiveBuf[1] == 0x03 && receiveBuf[2] == 0x01 );


    // セッションの終了
    nn::i2c::CloseSession(session);

    // ライブラリの終了
    nn::i2c::Finalize();

    return true;
}

}}}  // namespace nnt::i2c::clsic
