﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  ユーティリティ関数
//---------------------------------------------------------------------------

#include <nn/os.h>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

#include <nn/gpio/gpio.h> // To supply LDO1_EN to ALC5639
#include <nn/i2c/i2c.h>
#include <nn/i2c/i2c_BusDev.h> // Include development header not in nn/i2c/i2c.h

#include <nnt/nntest.h>

#include "../Common/test_Util.h"
#include "test_Alc5639.h"

namespace {

const nn::Bit16               SlaveAddr              = 0x1c;

const uint8_t                 RegSpkoutVol           = 0x01;
const uint8_t                 DefaultValSpkoutVol[2] = {0xc8, 0xc8};
const uint8_t                 RegVendorId            = 0xfe;
const uint8_t                 DefaultValVendorId[2]  = {0x10, 0xec};

#if (defined(NN_BUILD_CONFIG_SPEC_GENERIC))
const nn::gpio::GpioPadName   Alc5639GpioPadName = nn::gpio::GpioPadName_AudioPowerEnable;
#elif (defined(NN_BUILD_CONFIG_SPEC_NX))
const nn::gpio::GpioPadName   Alc5639GpioPadName = nn::gpio::GpioPadName_CodecLdoEnTemp;
#else
    #error NN_BUILD_CONFIG_SPEC_ is not defined
#endif

void SetUpGpioForAlc5639(nn::gpio::GpioPadSession* pSession)
{
    nn::gpio::OpenSession(pSession, Alc5639GpioPadName);
    nn::gpio::SetDirection(pSession, nn::gpio::Direction_Output);
    nn::gpio::SetValue(pSession, nn::gpio::GpioValue_High );

    // ALC5639 requires minimum 400ms wait after enabling LDO1_EN
    // (see "ALC5639 Datasheet Rev 0.99", Sec 7.2, Figure 5)
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(400) );
}

}

namespace nnt { namespace i2c { namespace alc5639 {

//---------------------------------------------------------------------------
//  テスト用関数
//---------------------------------------------------------------------------

bool TEST_VariousSpeedModes( int targetBusIdx, const nn::i2c::SpeedMode* targetSpeedModes, int targetSpeedModeNum )
{
    // ALC5639 needs LDO1_EN input enabled via GPIO
    nn::gpio::Initialize();
    nn::gpio::GpioPadSession gpioSession;

    SetUpGpioForAlc5639(&gpioSession);

    nn::i2c::Initialize();

    nn::i2c::I2cSession i2c;

    for( int i = 0; i < targetSpeedModeNum; i++ )
    {
        NN_LOG("Test @ SpeedMode = %d [bps]\n", targetSpeedModes[i]);
        nn::i2c::OpenSessionForDev(&i2c, targetBusIdx, SlaveAddr, nn::i2c::AddressingMode_BitWidth7, targetSpeedModes[i]);

        nnt::i2c::util::TestReadRegister(i2c,
            reinterpret_cast<const char*>(&RegSpkoutVol),        sizeof(RegSpkoutVol),
            reinterpret_cast<const char*>(&DefaultValSpkoutVol), sizeof(DefaultValSpkoutVol)
            );
        if (::testing::Test::HasFatalFailure())
        {
            nn::i2c::CloseSession(i2c);
            nn::i2c::Finalize();
            return false;
        }

        nnt::i2c::util::TestReadRegister(i2c,
            reinterpret_cast<const char*>(&RegVendorId),         sizeof(RegVendorId),
            reinterpret_cast<const char*>(&DefaultValVendorId),  sizeof(DefaultValVendorId)
            );
        if (::testing::Test::HasFatalFailure())
        {
            nn::i2c::CloseSession(i2c);
            nn::i2c::Finalize();
            return false;
        }
        nn::i2c::CloseSession(i2c);
    }

    nn::i2c::Finalize();

    nn::gpio::CloseSession(&gpioSession);
    nn::gpio::Finalize();

    return true;
}

bool TEST_I2cDeviceAccess()
{
    // ALC5639 needs LDO1_EN input enabled via GPIO
    nn::gpio::Initialize();
    nn::gpio::GpioPadSession gpioSession;

    SetUpGpioForAlc5639(&gpioSession);

    nn::i2c::Initialize();

    nn::i2c::I2cSession i2c;

    nn::i2c::OpenSession(&i2c, nn::i2c::I2cDevice::I2cDevice_Alc5639);

    nnt::i2c::util::TestReadRegister(i2c,
        reinterpret_cast<const char*>(&RegSpkoutVol),        sizeof(RegSpkoutVol),
        reinterpret_cast<const char*>(&DefaultValSpkoutVol), sizeof(DefaultValSpkoutVol)
        );
    if (::testing::Test::HasFatalFailure())
    {
        nn::i2c::CloseSession(i2c);
        nn::i2c::Finalize();
        return false;
    }

    nnt::i2c::util::TestReadRegister(i2c,
        reinterpret_cast<const char*>(&RegVendorId),         sizeof(RegVendorId),
        reinterpret_cast<const char*>(&DefaultValVendorId),  sizeof(DefaultValVendorId)
        );
    if (::testing::Test::HasFatalFailure())
    {
        nn::i2c::CloseSession(i2c);
        nn::i2c::Finalize();
        return false;
    }
    nn::i2c::CloseSession(i2c);

    nn::i2c::Finalize();

    nn::gpio::CloseSession(&gpioSession);
    nn::gpio::Finalize();

    return true;
}


}}}  // namespace nnt::i2c::alc5639
