﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Net.Sockets;
using System.Threading;
using System.IO;

using Nintendo.Htcs;

namespace StressTestTool
{
    internal class StressTest
    {
        private readonly string m_KeyString;
        private readonly HtcsCommunicator m_HtcsCommunicator;

        private int m_TestCount = 0;
        private int m_FinishedTestCount = 0;

        //const int DataSize = 1024 * 1024;
        //const int RepeatCount = 10;

        private struct TestParams
        {
            public int  threadNum;
            public int  threadCount;
            public int  dataSize;
            public int  repeatCount;
        };

        public StressTest(string keyString, HtcsCommunicator htcsCommunicator)
        {
            m_KeyString = keyString;
            m_HtcsCommunicator = htcsCommunicator;
        }

        public void Start()
        {
            var tcpListener = new TcpListener(System.Net.IPAddress.Loopback, 0);
            tcpListener.Start();

            var htcsPortDesc = new HtcsPortDescriptor(HtcsPeerName.Any, "StressTestManager" + m_KeyString);
            var portMapItem = new PortMapItem(htcsPortDesc, ((System.Net.IPEndPoint)tcpListener.LocalEndpoint));
            while (m_HtcsCommunicator.RegisterPort(portMapItem) != 0)
            {
                Console.WriteLine("Registration error.");
                Thread.Sleep(1000);
            }

            var client = tcpListener.AcceptTcpClient();

            using (var reader = new StreamReader(client.GetStream(), Encoding.ASCII, true))
            {
                while (true)
                {
                    var data = reader.ReadLine().Trim('\0', ' ', '\n').Split(',');

                    int threadNum   = Int32.Parse(data[0]);
                    int threadCount = Int32.Parse(data[1]);
                    string testName = data[2];
                    int dataSize    = Int32.Parse(data[3]);
                    int repeatCount = Int32.Parse(data[4]);

                    Console.WriteLine("[Thread{0}-{1}] Name:{2}, Size:{3}, LoopCount:{4}", threadNum, threadCount, testName, dataSize, repeatCount);

                    if (testName == "End")
                    {
                        break;
                    }

                    TestParams param;
                    param.threadNum = threadNum;
                    param.threadCount = threadCount;
                    param.dataSize = dataSize;
                    param.repeatCount = repeatCount;

                    var callback = GetTestCallback(testName);

                    if (callback == null)
                    {
                        continue;
                    }

                    m_TestCount++;

                    ThreadPool.QueueUserWorkItem(callback, param);
                }
            }

            client.Close();

            m_HtcsCommunicator.UnregisterPort(htcsPortDesc);

            tcpListener.Stop();

            Console.WriteLine("[StressTest] Waiting all tests end.");
            while (m_FinishedTestCount < m_TestCount)
            {
                System.Threading.Thread.Sleep(1000);
            }
            Console.WriteLine("[StressTest] All tests finished.");
        }

        public WaitCallback GetTestCallback(string testName)
        {
            switch (testName)
            {
                case "RecvClient":
                    return new WaitCallback(TestRecvClient);
                case "SendClient":
                    return new WaitCallback(TestSendClient);
                case "EchoClient":
                    return new WaitCallback(TestEchoClient);
                case "RecvServer":
                    return new WaitCallback(TestRecvServer);
                case "SendServer":
                    return new WaitCallback(TestSendServer);
                case "EchoServer":
                    return new WaitCallback(TestEchoServer);
                default:
                    return null;
            }
        }

        public void TestRecvClient(object obj)
        {
            TestParams param = (TestParams)obj;
            var testServer = new TestServer(0, "Stress" + param.threadNum + "_" + param.threadCount + "_" + m_KeyString, m_HtcsCommunicator, (client) => Utils.Send(client, param.dataSize, param.repeatCount));
            testServer.Start();
            testServer.Wait();
            m_FinishedTestCount++;
        }

        public void TestSendClient(object obj)
        {
            TestParams param = (TestParams)obj;
            var testServer = new TestServer(0, "Stress" + param.threadNum + "_" + param.threadCount + "_" + m_KeyString, m_HtcsCommunicator, (client) => Utils.Recv(client, param.dataSize, param.repeatCount));
            testServer.Start();
            testServer.Wait();
            m_FinishedTestCount++;
        }

        public void TestEchoClient(object obj)
        {
            TestParams param = (TestParams)obj;
            var testServer = new TestServer(0, "Stress" + param.threadNum + "_" + param.threadCount + "_" + m_KeyString, m_HtcsCommunicator, (client) => Utils.Echo(client, param.dataSize, param.repeatCount));
            testServer.Start();
            testServer.Wait();
            m_FinishedTestCount++;
        }

        public void TestRecvServer(object obj)
        {
            TestParams param = (TestParams)obj;
            var testClient = new TestClient("Stress" + param.threadNum + "_" + param.threadCount + "_" + m_KeyString, m_HtcsCommunicator, (client) => Utils.Send(client, param.dataSize, param.repeatCount));
            testClient.Start();
            testClient.Wait();
            m_FinishedTestCount++;
        }

        public void TestSendServer(object obj)
        {
            TestParams param = (TestParams)obj;
            var testClient = new TestClient("Stress" + param.threadNum + "_" + param.threadCount + "_" + m_KeyString, m_HtcsCommunicator, (client) => Utils.Recv(client, param.dataSize, param.repeatCount));
            testClient.Start();
            testClient.Wait();
            m_FinishedTestCount++;
        }

        public void TestEchoServer(object obj)
        {
            TestParams param = (TestParams)obj;
            var testClient = new TestClient("Stress" + param.threadNum + "_" + param.threadCount + "_" + m_KeyString, m_HtcsCommunicator, (client) => Utils.Echo(client, param.dataSize, param.repeatCount));
            testClient.Start();
            testClient.Wait();
            m_FinishedTestCount++;
        }
    }
}
