﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Net.Sockets;
using System.Threading;
using System.IO;

using Nintendo.Htcs;

namespace StressTestTool
{
    internal class NonBlockingTest
    {
        private readonly string m_KeyString;
        private readonly HtcsCommunicator m_HtcsCommunicator;

        const int DataSize = 1024;

        public NonBlockingTest(string keyString, HtcsCommunicator htcsCommunicator)
        {
            m_KeyString = keyString;
            m_HtcsCommunicator = htcsCommunicator;
        }

        public void Start()
        {
#if false
            ConnectTest();
#endif
            AcceptTest();
            MultipleAcceptTest();
            SendAndReceiveTest();
        }

        public void ConnectTest()
        {
            var listner = new TcpListener(System.Net.IPAddress.Loopback, 0);
            listner.Start();
            var htcsPortDesc = new HtcsPortDescriptor(HtcsPeerName.Any, "Connect_" + m_KeyString);
            var portMapping = new PortMapItem(htcsPortDesc, ((System.Net.IPEndPoint)listner.LocalEndpoint));
            while (m_HtcsCommunicator.RegisterPort(portMapping) != 0)
            {
                Console.WriteLine("Registration error.");
                Thread.Sleep(1000);
            }

            var client = listner.AcceptTcpClient();
            Utils.Echo(client, DataSize);
            client.Close();

            listner.Stop();
            m_HtcsCommunicator.UnregisterPort(htcsPortDesc);
        }

        public void AcceptTest()
        {
            PortMapItem portMapItem;
            while (true)
            {
                try
                {
                    portMapItem = m_HtcsCommunicator.PortMap.First(x => x.HtcsPortDescriptor.HtcsPortName == "Accept_" + m_KeyString);
                    break;
                }
                catch (InvalidOperationException ioe)
                {
                    Console.WriteLine("Invalid operation occurred({0:X8})", ioe.HResult);
                }

                // 例外発生時、しばらく待ってやり直す
                Thread.Sleep(1000);
            }

            // Accept のエラーチェックのために一時停止してエラーを発生させる
            Thread.Sleep(10000);

            var client = new TcpClient();
            client.Connect(portMapItem.EndPoint);
            Utils.Echo(client, DataSize);
            client.Close();
        }

        public void MultipleAcceptTest()
        {
            PortMapItem portMapItem;
            while (true)
            {
                try
                {
                    portMapItem = m_HtcsCommunicator.PortMap.First(x => x.HtcsPortDescriptor.HtcsPortName == "MultipleAccept_" + m_KeyString);
                    break;
                }
                catch (InvalidOperationException ioe)
                {
                    Console.WriteLine("Invalid operation occurred({0:X8})", ioe.HResult);
                }

                // 例外発生時、しばらく待ってやり直す
                Thread.Sleep(1000);
            }

            const int ClientCount = 11; // テストする backlog 数 + 1
            TcpClient[] client = new TcpClient[ClientCount];
            for (int i = 0; i < ClientCount; i++)
            {
                client[i] = new TcpClient();

                int retryCount = 10;
                while(true)
                {
                    try
                    {
                        client[i].Connect(portMapItem.EndPoint);
                    }
                    catch (SocketException e)
                    {
                        Console.WriteLine("Connect failed : error code " + e.ErrorCode);
                        if (retryCount > 0)
                        {
                            client[i] = new TcpClient();
                            Console.WriteLine("Retry");
                            retryCount--;
                            Thread.Sleep(100);
                            continue;
                        }
                        throw new Exception("Connect failed : error code " + e.ErrorCode);
                    }
                    Console.WriteLine("Connect success ({0}/{1})", i + 1, ClientCount);
                    System.Threading.Thread.Sleep(100);
                    break;
                }
            }

            try
            {
                var tmpClient = new TcpClient();
                tmpClient.Connect(portMapItem.EndPoint);
            }
            catch (Exception e)
            {
                Console.WriteLine("Expected error");
                Console.WriteLine(e.Message);
            }

            for (int i = 0; i < ClientCount; i++)
            {
                Utils.Echo(client[i], DataSize);
                client[i].Close();
            }
        }

        public void SendAndReceiveTest()
        {
            var listner = new TcpListener(System.Net.IPAddress.Loopback, 0);
            listner.Start();
            var htcsPortDesc = new HtcsPortDescriptor(HtcsPeerName.Any, "SendAndReceive_" + m_KeyString);
            var portMapping = new PortMapItem(htcsPortDesc, ((System.Net.IPEndPoint)listner.LocalEndpoint));
            while (m_HtcsCommunicator.RegisterPort(portMapping) != 0)
            {
                Console.WriteLine("Registration error.");
                Thread.Sleep(1000);
            }

            var client = listner.AcceptTcpClient();
            Utils.Echo(client, DataSize);

            using (var reader = new BinaryReader(client.GetStream(), Encoding.ASCII, true))
            {
                byte[] data = reader.ReadBytes(4);
                if (data.Length != 4)
                {
                    Console.WriteLine("Error: failed to receive");
                }
                Console.WriteLine(System.Text.Encoding.UTF8.GetString(data));
            }

            client.Close();

            listner.Stop();
            m_HtcsCommunicator.UnregisterPort(htcsPortDesc);
        }
    }
}
