﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <nn/nn_Log.h>
#include <nnt/nntest.h>
#include <nnt/htcsUtil/testHtcs_util.h>
#include <nnt/htcsUtil/testHtcs_utilMemoryLeakDetector.h>

/*
*   色々なサイズのデータを送受信する Htcs のストレステスト
*   事前に Target Manager を立ち上げておく必要があります。
*   テスト時に Tests/Htcs/Sources/Tools/StressTestTool を実行する必要があります。
*/

namespace
{
    // テストデータの最大サイズ
    const size_t MaxDataSize = 16 * 1024;

    static const size_t KeyStringSize = 11;
    static char g_KeyString[KeyStringSize];

    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }

    class VariousSize : public ::testing::Test
    {
    protected:
        static void SetUpTestCase() {
            for (int i = 0; i < (KeyStringSize - 1); i++)
            {
                int seed = nn::os::GetSystemTick().GetInt64Value() % 26;
                g_KeyString[i] = static_cast<char>('a' + seed);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            }
            g_KeyString[KeyStringSize - 1] = '\0';
        }

        virtual void SetUp()
        {
            nn::htcs::Initialize(Allocate, Deallocate);
        }
        virtual void TearDown()
        {
            nn::htcs::Finalize();
        }
    };
}

TEST_F(VariousSize, Start)
{
    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    nnt::htcs::util::ConnectToTestServer(socket);

    const char* testName = "VariousSize";

    size_t bufferSize = std::strlen(g_KeyString) + 1 + std::strlen(testName) + 1;
    char* buffer = reinterpret_cast<char*>(std::malloc(bufferSize));
    ASSERT_NE(nullptr, buffer);
    buffer[0] = '\0';

    std::strcat(buffer, g_KeyString);
    std::strcat(buffer, ",");
    std::strcat(buffer, testName);

    ASSERT_EQ(nn::htcs::Send(socket, buffer, bufferSize, 0), bufferSize);

    std::free(buffer);
}

TEST_F(VariousSize, EchoClient)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    // HTCS ポート名
    nn::htcs::HtcsPortName portName;
    std::strcpy(portName.name, "Echo0_");
    std::strcat(portName.name, g_KeyString);

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    nnt::htcs::util::ConnectToAnyHost(socket, &portName);

    // テスト本体
    for (int size = 1; size <= MaxDataSize; size++)
    {
        uint8_t* sendBuffer = reinterpret_cast<uint8_t*>(std::malloc(size));
        ASSERT_NE(nullptr, sendBuffer);

        uint8_t* recvBuffer = reinterpret_cast<uint8_t*>(std::malloc(size));
        ASSERT_NE(nullptr, recvBuffer);

        nnt::htcs::util::FillBuffer(sendBuffer, size);
        nnt::htcs::util::Echo(socket, size, sendBuffer, recvBuffer);

        std::free(sendBuffer);
        std::free(recvBuffer);
    }

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(VariousSize, EchoServer)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    // HTCS ポート名
    nn::htcs::HtcsPortName portName;
    std::strcpy(portName.name, "Echo1_");
    std::strcat(portName.name, g_KeyString);

    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, portName.name);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);

    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &addr));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    int socket = nn::htcs::Accept(listenSocket, nullptr);
    EXPECT_LE(0, socket);

    // テスト本体
    for (int size = 1; size <= MaxDataSize; size++)
    {
        uint8_t* sendBuffer = reinterpret_cast<uint8_t*>(std::malloc(size));
        ASSERT_NE(nullptr, sendBuffer);

        uint8_t* recvBuffer = reinterpret_cast<uint8_t*>(std::malloc(size));
        ASSERT_NE(nullptr, recvBuffer);

        nnt::htcs::util::FillBuffer(sendBuffer, size);
        nnt::htcs::util::Echo(socket, size, sendBuffer, recvBuffer);

        std::free(sendBuffer);
        std::free(recvBuffer);
    }

    EXPECT_EQ(0, nn::htcs::Close(socket));
    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}
