﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <nn/nn_Log.h>
#include <nnt/nntest.h>
#include <nnt/htcsUtil/testHtcs_util.h>
#include <nnt/htcsUtil/testHtcs_utilMemoryLeakDetector.h>

/*
*   ノンブロッキングモード関係のテストです。
*   事前に Target Manager を立ち上げておく必要があります。
*   テスト時に Tests/Htcs/Sources/Tools/StressTestTool を実行する必要があります。
*/

namespace
{
    static const size_t KeyStringSize = 11;
    static char g_KeyString[KeyStringSize];

    // [不具合？] dataSize を 1 MB にすると失敗する
    static const nn::htcs::ssize_t dataSize = 1024;

    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }

    class NonBlockingTest : public ::testing::Test
    {
    protected:
        static void SetUpTestCase() {
            for (int i = 0; i < (KeyStringSize - 1); i++)
            {
                int seed = nn::os::GetSystemTick().GetInt64Value() % 26;
                g_KeyString[i] = static_cast<char>('a' + seed);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            }
            g_KeyString[KeyStringSize - 1] = '\0';
        }

        virtual void SetUp()
        {
            nn::htcs::Initialize(Allocate, Deallocate);
        }
        virtual void TearDown()
        {
            nn::htcs::Finalize();
        }
    };

    void NonBlockingSend(int socket, nn::htcs::ssize_t size, uint8_t* buffer)
    {
        nn::htcs::ssize_t sentByteCount = 0;
        do
        {
            auto n = nn::htcs::Send(socket, &buffer[sentByteCount], size - sentByteCount, 0);
            if (n < 0)
            {
                NN_LOG("Send Check\n");
                EXPECT_EQ(nn::htcs::HTCS_EWOULDBLOCK, nn::htcs::GetLastError());
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                continue;
            }
            sentByteCount += n;
        } while (sentByteCount < size);
        EXPECT_EQ(size, sentByteCount);
    }

    void NonBlockingRecv(int socket, nn::htcs::ssize_t size, uint8_t* buffer)
    {
        nn::htcs::ssize_t recievedByteCount = 0;
        do
        {
            auto n = nn::htcs::Recv(socket, &buffer[recievedByteCount], size - recievedByteCount, 0);
            if (n < 0)
            {
                NN_LOG("Recv Check\n");
                EXPECT_EQ(nn::htcs::HTCS_EWOULDBLOCK, nn::htcs::GetLastError());
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                continue;
            }
            recievedByteCount += n;
        } while (recievedByteCount < size);
        EXPECT_EQ(size, recievedByteCount);
    }

}

TEST_F(NonBlockingTest, Start)
{
    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    nnt::htcs::util::ConnectToTestServer(socket);

    const char* testName = "NonBlockingTest";

    size_t bufferSize = std::strlen(g_KeyString) + 1 + std::strlen(testName) + 1;
    char* buffer = reinterpret_cast<char*>(std::malloc(bufferSize));
    ASSERT_NE(nullptr, buffer);
    buffer[0] = '\0';

    std::strcat(buffer, g_KeyString);
    std::strcat(buffer, ",");
    std::strcat(buffer, testName);

    ASSERT_EQ(nn::htcs::Send(socket, buffer, bufferSize, 0), bufferSize);

    std::free(buffer);
}

#if 0
TEST_F(NonBlockingTest, Connect)
{
    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "Connect_");
    std::strcat(address.portName.name, g_KeyString);

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    EXPECT_EQ(0, nn::htcs::Fcntl(socket, nn::htcs::HTCS_F_SETFL, nn::htcs::HTCS_O_NONBLOCK));

    if (nn::htcs::Connect(socket, &address) < 0)
    {
        ASSERT_EQ(nn::htcs::HTCS_EWOULDBLOCK, nn::htcs::GetLastError());
    }

    // TORIAEZU: select がないので固定時間待つ
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    uint8_t* sendBuffer = reinterpret_cast<uint8_t*>(std::malloc(dataSize));
    uint8_t* recvBuffer = reinterpret_cast<uint8_t*>(std::malloc(dataSize));
    ASSERT_NE(nullptr, sendBuffer);
    ASSERT_NE(nullptr, recvBuffer);

    nnt::htcs::util::FillBuffer(sendBuffer, dataSize);
    std::memset(recvBuffer, 0, dataSize);

    NonBlockingSend(socket, dataSize, sendBuffer);
    NonBlockingRecv(socket, dataSize, recvBuffer);

    // 検証
    for (int i = 0; i < dataSize; i++)
    {
        EXPECT_EQ(recvBuffer[i], sendBuffer[i]);
        if (recvBuffer[i] != sendBuffer[i])
        {
            break;
        }
    }

    std::free(sendBuffer);
    std::free(recvBuffer);

    EXPECT_EQ(0, nn::htcs::Close(socket));
}
#endif

TEST_F(NonBlockingTest, Accept)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "Accept_");
    std::strcat(address.portName.name, g_KeyString);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);

    EXPECT_EQ(0, nn::htcs::Fcntl(listenSocket, nn::htcs::HTCS_F_SETFL, nn::htcs::HTCS_O_NONBLOCK));

    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &address));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 0));

    NN_LOG("Waiting for connection from host.\n");
    int socket;
    do
    {
        socket = nn::htcs::Accept(listenSocket, nullptr);
        if (socket < 0)
        {
            EXPECT_EQ(nn::htcs::HTCS_EWOULDBLOCK, nn::htcs::GetLastError());
        }
    } while (socket < 0);

    EXPECT_EQ(nn::htcs::HTCS_O_NONBLOCK, nn::htcs::Fcntl(socket, nn::htcs::HTCS_F_GETFL, 0));
    EXPECT_EQ(0, nn::htcs::Fcntl(socket, nn::htcs::HTCS_F_SETFL, 0));

    uint8_t* sendBuffer = reinterpret_cast<uint8_t*>(std::malloc(dataSize));
    uint8_t* recvBuffer = reinterpret_cast<uint8_t*>(std::malloc(dataSize));
    ASSERT_NE(nullptr, sendBuffer);
    ASSERT_NE(nullptr, recvBuffer);

    nnt::htcs::util::FillBuffer(sendBuffer, dataSize);
    std::memset(recvBuffer, 0, dataSize);

    nnt::htcs::util::Echo(socket, dataSize, sendBuffer, recvBuffer);

    std::free(sendBuffer);
    std::free(recvBuffer);

    EXPECT_EQ(0, nn::htcs::Close(socket));
    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}

TEST_F(NonBlockingTest, MultipleAccept)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    const int backlogCount = 10; // 検証する backlog の数

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "MultipleAccept_");
    std::strcat(address.portName.name, g_KeyString);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);

    EXPECT_EQ(0, nn::htcs::Fcntl(listenSocket, nn::htcs::HTCS_F_SETFL, nn::htcs::HTCS_O_NONBLOCK));

    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &address));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, backlogCount));

    for (int i = 0; i < backlogCount + 1; i++)
    {
        NN_LOG("Waiting for connection from host.\n");
        int socket = nn::htcs::Accept(listenSocket, nullptr);
        if (i == 0)
        {
            while (socket < 0)
            {
                EXPECT_EQ(nn::htcs::HTCS_EWOULDBLOCK, nn::htcs::GetLastError());
                socket = nn::htcs::Accept(listenSocket, nullptr);
            }
        }
        EXPECT_LE(0, socket);
        if (socket < 0)
        {
            NN_LOG("L%d LastError = %d\n", __LINE__, nn::htcs::GetLastError());
            break;
        }

        EXPECT_EQ(nn::htcs::HTCS_O_NONBLOCK, nn::htcs::Fcntl(socket, nn::htcs::HTCS_F_GETFL, 0));
        EXPECT_EQ(0, nn::htcs::Fcntl(socket, nn::htcs::HTCS_F_SETFL, 0));

        uint8_t* sendBuffer = reinterpret_cast<uint8_t*>(std::malloc(dataSize));
        uint8_t* recvBuffer = reinterpret_cast<uint8_t*>(std::malloc(dataSize));
        ASSERT_NE(nullptr, sendBuffer);
        ASSERT_NE(nullptr, recvBuffer);

        nnt::htcs::util::FillBuffer(sendBuffer, dataSize);
        std::memset(recvBuffer, 0, dataSize);

        nnt::htcs::util::Echo(socket, dataSize, sendBuffer, recvBuffer);

        std::free(sendBuffer);
        std::free(recvBuffer);

        EXPECT_EQ(0, nn::htcs::Close(socket));
    }

    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}

TEST_F(NonBlockingTest, SendAndReceive)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "SendAndReceive_");
    std::strcat(address.portName.name, g_KeyString);

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    NN_LOG("Trying to connect...\n");
    while (NN_STATIC_CONDITION(true))
    {
        if (nn::htcs::Connect(socket, &address) == 0)
        {
            NN_LOG("Connected to server on host.\n");
            break;
        }
        NN_LOG("L%d LastError = %d\n", __LINE__, nn::htcs::GetLastError());
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    }
    NN_LOG("Connected!\n");

    uint8_t* sendBuffer = reinterpret_cast<uint8_t*>(std::malloc(dataSize));
    uint8_t* recvBuffer = reinterpret_cast<uint8_t*>(std::malloc(dataSize));
    ASSERT_NE(nullptr, sendBuffer);
    ASSERT_NE(nullptr, recvBuffer);

    nnt::htcs::util::FillBuffer(sendBuffer, dataSize);
    std::memset(recvBuffer, 0, dataSize);

    EXPECT_EQ(0, nn::htcs::Fcntl(socket, nn::htcs::HTCS_F_SETFL, nn::htcs::HTCS_O_NONBLOCK));

    NonBlockingSend(socket, dataSize, sendBuffer);
    NonBlockingRecv(socket, dataSize, recvBuffer);
    EXPECT_EQ(4, nn::htcs::Send(socket, "End", 4, 0));

    // 検証
    for (int i = 0; i < dataSize; i++)
    {
        EXPECT_EQ(recvBuffer[i], sendBuffer[i]);
    }

    std::free(sendBuffer);
    std::free(recvBuffer);

    EXPECT_EQ(0, nn::htcs::Close(socket));
}
