﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/htcs.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/htcsUtil/testHtcs_utilMemoryLeakDetector.h>

/*
*   初期化テスト
*/

namespace
{
    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }
}

TEST(Initialize, Basic)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    EXPECT_EQ(false, nn::htcs::IsInitialized());
    nn::htcs::Initialize(Allocate, Deallocate);
    EXPECT_EQ(true, nn::htcs::IsInitialized());
    nn::htcs::Finalize();
    EXPECT_EQ(false, nn::htcs::IsInitialized());
}

TEST(Initialize, Repeat)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    for (int i = 0; i < 100; i++)
    {
        NN_LOG("count=%d\n", i);
        EXPECT_EQ(false, nn::htcs::IsInitialized());
        nn::htcs::Initialize(Allocate, Deallocate);
        int socket;
        while ((socket = nn::htcs::Socket()) == -1)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }
        nn::htcs::Close(socket);
        EXPECT_EQ(true, nn::htcs::IsInitialized());
        nn::htcs::Finalize();
    }
    EXPECT_EQ(false, nn::htcs::IsInitialized());
}

TEST(Initlialize, LowMemory)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    const int socketCount = 4;
    size_t bufferSize = nn::htcs::GetWorkingMemorySize(socketCount);
    void* buffer = Allocate(bufferSize);
    nn::htcs::Initialize(buffer, bufferSize);

    int socket[socketCount];
    for (int i = 0; i < socketCount; i++)
    {
        socket[i] = nn::htcs::Socket();
        EXPECT_LE(0, socket[i]);
    }
    EXPECT_EQ(-1, nn::htcs::Socket());
    EXPECT_EQ(nn::htcs::HTCS_EMFILE, nn::htcs::GetLastError());
    for (int i = 0; i < socketCount; i++)
    {
        EXPECT_EQ(0, nn::htcs::Close(socket[i]));
    }
    nn::htcs::Finalize();
}
