﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <functional>
#include <numeric>
#include <thread>
#include <vector>

#include <nn/nn_Log.h>
#include <nn/os.h>

#include <nn/htclow.h>
#include <nn/htclow/detail/htclow_InternalTypes.h>
#include <nn/htclow/detail/htclow_DebugApi.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include "../../../../../Programs/Eris/Sources/Libraries/htclow/server/htclow_Packet.h"

#include "../testHtclow_Util.h"
#include "testHtclow_HandshakeUtil.h"
#include "testHtclow_PacketUtil.h"

/*
 * @file htclow の境界値テスト
 */

namespace nnt { namespace htclow {

namespace {
    const nn::htclow::ModuleId TestModuleId = static_cast<nn::htclow::ModuleId>(0);
    const nn::htclow::ChannelId TestChannelId = static_cast<nn::htclow::ChannelId>(0);
}

class BoundaryTest : public ::testing::Test
{
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::htclow::detail::OpenDriver(nn::htclow::detail::DriverType::Debug);
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::htclow::detail::CloseDriver();
    }
};

// 0 バイトの Send() を混ぜても正しく送信できることをテストする
TEST_F(BoundaryTest, ZeroByteSend)
{
    nn::htclow::Module module(TestModuleId);
    nn::htclow::Channel channel(&module, TestChannelId);

    DoHandshake(&channel);

    // データの準備
    uint8_t dummy;

    const int dataSize1 = 1;
    const int dataSize2 = 16;
    const int dataSize3 = GetMaxBodySize();

    const auto data1 = MakeRandomArray(dataSize1, 1);
    const auto data2 = MakeRandomArray(dataSize2, 2);
    const auto data3 = MakeRandomArray(dataSize3, 3);

    // データの送信
    size_t size;

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Send(&size, &dummy, 0));
    ASSERT_EQ(0, size);

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Send(&size, data1.get(), dataSize1));
    ASSERT_EQ(dataSize1, size);

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Send(&size, &dummy, 0));
    ASSERT_EQ(0, size);

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Send(&size, data2.get(), dataSize2));
    ASSERT_EQ(dataSize2, size);

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Send(&size, &dummy, 0));
    ASSERT_EQ(0, size);

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Send(&size, &dummy, 0));
    ASSERT_EQ(0, size);

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Send(&size, &dummy, 0));
    ASSERT_EQ(0, size);

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Send(&size, data3.get(), dataSize3));
    ASSERT_EQ(dataSize3, size);

    // データの受信
    const auto channelInternal = nn::htclow::detail::ConvertChannelType(*(channel.GetBase()));

    const auto packet1 = ReceivePacketForTest();
    AssertDataPacket(*packet1, channelInternal, 1, data1.get(), dataSize1);

    const auto packet2 = ReceivePacketForTest();
    AssertDataPacket(*packet2, channelInternal, 2, data2.get(), dataSize2);

    const auto packet3 = ReceivePacketForTest();
    AssertDataPacket(*packet3, channelInternal, 3, data3.get(), dataSize3);
}

// 0 バイトの Receive() を混ぜても正しく受信できることをテストする
TEST_F(BoundaryTest, ZeroByteReceive)
{
    nn::htclow::Module module(TestModuleId);
    nn::htclow::Channel channel(&module, TestChannelId);

    DoHandshake(&channel);

    // データの準備
    const int dataSize1 = 1;
    const int dataSize2 = 16;
    const int dataSize3 = GetMaxBodySize();

    const auto data1 = MakeRandomArray(dataSize1, 1);
    const auto data2 = MakeRandomArray(dataSize2, 2);
    const auto data3 = MakeRandomArray(dataSize3, 3);

    uint8_t dummy;
    auto buffer1 = MakeZeroArray(dataSize1);
    auto buffer2 = MakeZeroArray(dataSize2);
    auto buffer3 = MakeZeroArray(dataSize3);

    const auto channelInternal = nn::htclow::detail::ConvertChannelType(*(channel.GetBase()));

    const auto packet1 = MakeDataPacket(channelInternal, 1, data1.get(), dataSize1);
    const auto packet2 = MakeDataPacket(channelInternal, 2, data2.get(), dataSize2);
    const auto packet3 = MakeDataPacket(channelInternal, 3, data3.get(), dataSize3);

    // ホスト側のデータ送信
    SendPacketForTest(*packet1);
    SendPacketForTest(*packet2);
    SendPacketForTest(*packet3);

    // ホスト側の ack 受信
    for (int sequenceId = 1; sequenceId <= 3; sequenceId++)
    {
        auto ack = ReceivePacketForTest();
        AssertDataAckPacket(*ack, channelInternal, sequenceId);
    }

    // ターゲット側のデータ受信
    size_t size;

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Receive(&size, &dummy, 0));
    ASSERT_EQ(0, size);

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Receive(&size, buffer1.get(), dataSize1));
    ASSERT_EQ(dataSize1, size);
    ASSERT_EQ(0, std::memcmp(data1.get(), buffer1.get(), dataSize1));

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Receive(&size, &dummy, 0));
    ASSERT_EQ(0, size);

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Receive(&size, buffer2.get(), dataSize2));
    ASSERT_EQ(dataSize2, size);
    ASSERT_EQ(0, std::memcmp(data2.get(), buffer2.get(), dataSize2));

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Receive(&size, &dummy, 0));
    ASSERT_EQ(0, size);

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Receive(&size, &dummy, 0));
    ASSERT_EQ(0, size);

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Receive(&size, &dummy, 0));
    ASSERT_EQ(0, size);

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Receive(&size, buffer3.get(), dataSize3));
    ASSERT_EQ(dataSize3, size);
    ASSERT_EQ(0, std::memcmp(data3.get(), buffer3.get(), dataSize3));
}

// 対向が 0 バイトのデータパケットを混ぜて送信しても、正しく受信できることをテストする
TEST_F(BoundaryTest, ReceiveEmptyDataPacket)
{
    nn::htclow::Module module(TestModuleId);
    nn::htclow::Channel channel(&module, TestChannelId);

    DoHandshake(&channel);

    // データの準備
    const int dataSize1 = 1;
    const int dataSize2 = 16;
    const int dataSize3 = GetMaxBodySize();

    const auto data1 = MakeRandomArray(dataSize1, 1);
    const auto data2 = MakeRandomArray(dataSize2, 2);
    const auto data3 = MakeRandomArray(dataSize3, 3);

    uint8_t dummy;
    auto buffer1 = MakeZeroArray(dataSize1);
    auto buffer2 = MakeZeroArray(dataSize2);
    auto buffer3 = MakeZeroArray(dataSize3);

    const auto channelInternal = nn::htclow::detail::ConvertChannelType(*(channel.GetBase()));

    const auto packet1 = MakeDataPacket(channelInternal, 1, &dummy, 0);
    const auto packet2 = MakeDataPacket(channelInternal, 2, data1.get(), dataSize1);
    const auto packet3 = MakeDataPacket(channelInternal, 3, &dummy, 0);
    const auto packet4 = MakeDataPacket(channelInternal, 4, data2.get(), dataSize2);
    const auto packet5 = MakeDataPacket(channelInternal, 5, &dummy, 0);
    const auto packet6 = MakeDataPacket(channelInternal, 6, &dummy, 0);
    const auto packet7 = MakeDataPacket(channelInternal, 7, &dummy, 0);
    const auto packet8 = MakeDataPacket(channelInternal, 8, data3.get(), dataSize3);

    // ホスト側のデータ送信
    SendPacketForTest(*packet1);
    SendPacketForTest(*packet2);
    SendPacketForTest(*packet3);
    SendPacketForTest(*packet4);
    SendPacketForTest(*packet5);
    SendPacketForTest(*packet6);
    SendPacketForTest(*packet7);
    SendPacketForTest(*packet8);

    // ホスト側の ack 受信
    for (int sequenceId = 1; sequenceId <= 8; sequenceId++)
    {
        auto ack = ReceivePacketForTest();
        AssertDataAckPacket(*ack, channelInternal, sequenceId);
    }

    // ターゲット側のデータ受信
    size_t size;

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Receive(&size, buffer1.get(), dataSize1));
    ASSERT_EQ(dataSize1, size);
    ASSERT_EQ(0, std::memcmp(data1.get(), buffer1.get(), dataSize1));

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Receive(&size, buffer2.get(), dataSize2));
    ASSERT_EQ(dataSize2, size);
    ASSERT_EQ(0, std::memcmp(data2.get(), buffer2.get(), dataSize2));

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Receive(&size, buffer3.get(), dataSize3));
    ASSERT_EQ(dataSize3, size);
    ASSERT_EQ(0, std::memcmp(data3.get(), buffer3.get(), dataSize3));
}

// 対向がプロトコルの MaxBodySize を超えるデータパケットを送信した際、正しくエラーになることをテストする
TEST_F(BoundaryTest, ReceiveTooBigPacket)
{
    nn::htclow::Module module(TestModuleId);
    nn::htclow::Channel channel(&module, TestChannelId);

    DoHandshake(&channel);

    // データ準備
    const int dataSize = GetMaxBodySize() + 1;
    const auto data = MakeRandomArray(dataSize, 1);

    const auto channelInternal = nn::htclow::detail::ConvertChannelType(*(channel.GetBase()));

    // データ送信
    const auto packet = MakeDataPacket(channelInternal, 1, data.get(), dataSize);
    SendPacketForTest(*packet);

    // データ受信
    uint8_t dummy;
    size_t size;
    const auto result = channel.Receive(&size, &dummy, 1);
    NNT_HTCLOW_ASSERT_RESULT_INCLUDED(nn::htclow::ResultChannelStateError(), result);
}

// 複数のパケットを1回の Receive() で取得できることをテストする
TEST_F(BoundaryTest, OneReceiveForMultiPacket)
{
    nn::htclow::Module module(TestModuleId);
    nn::htclow::Channel channel(&module, TestChannelId);

    DoHandshake(&channel);

    // データセット（送信サイズ）
    std::vector<int> dataSet[]{
        { 1, 2, 3, 4, 5, 6, 7, 8 },
        { GetMaxBodySize(), GetMaxBodySize(), GetMaxBodySize(), 1},
    };

    const auto channelInternal = nn::htclow::detail::ConvertChannelType(*(channel.GetBase()));
    int64_t sequenceId = 1;

    for (auto& sizes : dataSet)
    {
        // データ準備
        const auto totalDataSize = std::accumulate(sizes.begin(), sizes.end(), 0);
        const auto data = MakeRandomArray(totalDataSize, 1);
        const auto buffer = MakeZeroArray(totalDataSize);

        // 複数回に分けて送信
        int offset = 0;
        for (auto& size : sizes)
        {
            const auto packet = MakeDataPacket(channelInternal, sequenceId, &data.get()[offset], size);
            SendPacketForTest(*packet);

            sequenceId++;
            offset += size;
        }

        // 1回で受信
        size_t size;
        NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Receive(&size, buffer.get(), totalDataSize));

        // 比較
        ASSERT_EQ(0, std::memcmp(data.get(), buffer.get(), totalDataSize));
    }
}

// 1つのパケットを複数回の Receive() で取得できることをテストする
TEST_F(BoundaryTest, MultiReceiveForOnePacket)
{
    nn::htclow::Module module(TestModuleId);
    nn::htclow::Channel channel(&module, TestChannelId);

    DoHandshake(&channel);

    // データセット（受信サイズ）
    std::vector<int> dataSet[]{
        { 1, 2, 3, 4, 5, 6, 7, 8 },
        { GetMaxBodySize() - 1, 1 },
    };

    const auto channelInternal = nn::htclow::detail::ConvertChannelType(*(channel.GetBase()));
    int64_t sequenceId = 1;

    for (auto& sizes : dataSet)
    {
        // データ準備
        const auto totalDataSize = std::accumulate(sizes.begin(), sizes.end(), 0);
        const auto data = MakeRandomArray(totalDataSize, 1);
        const auto buffer = MakeZeroArray(totalDataSize);

        // 1回で送信
        const auto packet = MakeDataPacket(channelInternal, sequenceId, data.get(), totalDataSize);
        SendPacketForTest(*packet);

        sequenceId++;

        // 複数回に分けて受信
        int offset = 0;
        for (auto& size : sizes)
        {
            size_t n;
            NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Receive(&n, &buffer.get()[offset], size));
            ASSERT_EQ(size, n);

            offset += size;
        }

        // 比較
        ASSERT_EQ(0, std::memcmp(data.get(), buffer.get(), totalDataSize));
    }
}

}}
