﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/os.h>

#include <nn/htclow.h>
#include <nn/htclow/detail/htclow_InternalTypes.h>
#include <nn/htclow/detail/htclow_DebugApi.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include "../testHtclow_Util.h"

#include "testHtclow_HandshakeUtil.h"

namespace nnt { namespace htclow {

class ApiTest : public ::testing::Test
{
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::htclow::detail::OpenDriver(nn::htclow::detail::DriverType::Debug);
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::htclow::detail::CloseDriver();
    }
};

// シャットダウン後に Connect を実行し、エラーが返ることを確認する。
TEST_F(ApiTest, ConnectAfterShutdown)
{
    // ChannelId, ModuleId は任意の適当な値
    nn::htclow::Module module(static_cast<nn::htclow::ModuleId>(0));
    nn::htclow::Channel channel(&module, static_cast<nn::htclow::ChannelId>(1));

    DoHandshake(&channel);
    DoShutdownByHost(&channel, 1, 1);

    NNT_HTCLOW_ASSERT_RESULT_INCLUDED(nn::htclow::ResultChannelStateError(), channel.Connect());
}

// クローズ後に Connect を実行し、エラーが返ることを確認する。
TEST_F(ApiTest, ConnectAfterClose)
{
    // ChannelId, ModuleId は任意の適当な値
    nn::htclow::Module module(static_cast<nn::htclow::ModuleId>(1));
    nn::htclow::Channel channel(&module, static_cast<nn::htclow::ChannelId>(1));

    DoHandshake(&channel);
    DoShutdownByTarget(&channel, 1, 1);

    channel.Close();

    NNT_HTCLOW_ASSERT_RESULT_INCLUDED(nn::htclow::ResultChannelClosed(), channel.Connect());
}

// シャットダウン後に Send を実行し、エラーが返ることを確認する。
TEST_F(ApiTest, SendAfterShutdown)
{
    // ChannelId, ModuleId は任意の適当な値
    nn::htclow::Module module(static_cast<nn::htclow::ModuleId>(2));
    nn::htclow::Channel channel(&module, static_cast<nn::htclow::ChannelId>(2));

    DoHandshake(&channel);
    DoShutdownByTarget(&channel, 1, 1);

    // Send
    const int dataSize = 16;
    const auto data = MakeRandomArray(dataSize, 3);

    size_t size;
    auto result = channel.Send(&size, data.get(), dataSize);
    NNT_HTCLOW_ASSERT_RESULT_INCLUDED(nn::htclow::ResultConnectionFailure(), result);
}

// クローズ後に Send を実行し、エラーが返ることを確認する。
TEST_F(ApiTest, SendAfterClose)
{
    // ChannelId, ModuleId は任意の適当な値
    nn::htclow::Module module(static_cast<nn::htclow::ModuleId>(3));
    nn::htclow::Channel channel(&module, static_cast<nn::htclow::ChannelId>(3));

    DoHandshake(&channel);
    DoShutdownByHost(&channel, 1, 1);

    channel.Close();

    // Send
    const int dataSize = 16;
    const auto data = MakeRandomArray(dataSize, 3);

    size_t size;
    auto result = channel.Send(&size, data.get(), dataSize);
    NNT_HTCLOW_ASSERT_RESULT_INCLUDED(nn::htclow::ResultChannelClosed(), result);
}

// チャンネルをクローズ後に Receive を実行し、エラーが返ることを確認する。
// このとき、別のチャンネルを1つオープンした状態にしておく。
TEST_F(ApiTest, ReceiveAfterCloseWithAnotherChannel)
{
    // ChannelId, ModuleId は任意の適当な値
    nn::htclow::Module module(static_cast<nn::htclow::ModuleId>(3));
    nn::htclow::Channel channel1(&module, static_cast<nn::htclow::ChannelId>(1));
    nn::htclow::Channel channel2(&module, static_cast<nn::htclow::ChannelId>(2));

    DoHandshake(&channel1);
    DoHandshake(&channel2);

    DoShutdownSimultaneously(&channel1, 1, 1);

    channel1.Close();

    // Receive
    const int bufferSize = 1;
    uint8_t buffer[bufferSize];

    size_t size;
    auto result = channel1.Receive(&size, &buffer, bufferSize);
    NNT_HTCLOW_ASSERT_RESULT_INCLUDED(nn::htclow::ResultChannelClosed(), result);
}

// 多重に Shutdown や Close を呼んでもエラーにならないことを確認する
TEST_F(ApiTest, MultipleClose)
{
    // ChannelId, ModuleId は任意の適当な値
    nn::htclow::Module module(static_cast<nn::htclow::ModuleId>(0));
    nn::htclow::Channel channel(&module, static_cast<nn::htclow::ChannelId>(0));

    DoHandshake(&channel);
    DoShutdownSimultaneously(&channel, 1, 1);

    channel.Close();
    channel.Close();
    channel.Shutdown();
    channel.Close();
    channel.Shutdown();
}

}}
