﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * Rondeと排他の機能をON/OFFする為だけの機能
 * (ちなみに初期化だけはボタン関係無しにやる模様)
 */


#include "ExclusiveFunc.h"

const int NpadIdNum = 9;

nns::hidfw::layout::ButtonGroup    IrGroup[NpadIdNum];        //!< IRセンサー
nns::hidfw::layout::ButtonGroup    NfcGroup[NpadIdNum];       //!< NFC
nns::hidfw::layout::ButtonGroup    SixAxisGroup[NpadIdNum];   //!< 6軸
nns::hidfw::layout::ButtonGroup    VibGroup[NpadIdNum];       //!< HD振動

nn::hid::NpadStyleSet CurrentStyle[NpadIdNum];
bool CurrentConnectState[NpadIdNum];

nn::irsensor::IrCameraHandle IrHandles[NpadIdNum];
nn::irsensor::MomentProcessorConfig IrConfig;

int                   NfcDeviceHandleCount;
nn::nfc::DeviceHandle HandleList[nn::nfc::DeviceCountMax];
nn::nfc::DeviceHandle NfcHandles[nn::nfc::DeviceCountMax];

int SixAxisHandleCounts[NpadIdNum];
nn::hid::SixAxisSensorHandle SixAxishandles[NpadIdNum][nn::hid::NpadSixAxisSensorHandleCountMax];
nn::hid::SixAxisSensorState SixAxisStates[nn::hid::SixAxisSensorStateCountMax];

int vibrationDeviceCount[NpadIdNum];
const int VibrationDeviceCountMax = 2;
nn::hid::VibrationDeviceHandle HdVibHandles[NpadIdNum][VibrationDeviceCountMax];

bool    IrIsActive[NpadIdNum];        //!< IRセンサー
bool    NfcIsActive[NpadIdNum];       //!< NFC
bool    SixIsActive[NpadIdNum];       //!< 6軸
bool    VibIsActive[NpadIdNum];       //!< HD振動

bool    IrIsInit = false;            //!< IRの初期化が完了したかどうか
bool    NfcIsInit = false;            //!< NFCの初期化が完了したかどうか

int     NfcErrorPrintCounter = 0;

int     SixAxisPrintCounter = 0;

float xOriginEx;
float yOriginEx;

const char* buttonTextExclusive[] =
{
    "OFF",
    "ON",
};

/////////////////////////////////
// IndexのNpadId変換
/////////////////////////////////
nn::hid::NpadIdType ConvertNpadIdFromIndex(size_t index)
{
    if (index != 8)
    {
        return index;
    }
    else
    {
        return nn::hid::NpadId::Handheld;
    }
}
const char* ToString(nn::Result result)
{
    if (result.IsSuccess())
    {
        return "Success";
    }
    else if (nn::nfc::ResultNfcDeviceNotFound::Includes(result))
    {
        // コントローラーの Firmware Update が発生したが失敗しました。
        // リトライするか、アプリ側で Firmware Update アプレットを立ち上げてください。
        return "NfcDeviceNotFound";
    }
    else if (nn::nfc::ResultNfcDisabled::Includes(result))
    {
        // アプリケーションが InForcus 状態でないため、拡張デバイスを有効化できません。
        // アプリケーション側で nn::oe のライブラリ群を使用してアプリケーションが InForcus になるのを待ってください。
        return "NfcDisabled";
    }
    else if (nn::nfc::ResultMaxNfcDeviceActivated::Includes(result))
    {
        // 指定された Npad が拡張バス機能と排他で使用されるべき機能を使用しているため、有効化できませんでした。
        // アプリケーション側で拡張バスと排他で使用されるべき機能を終了して、再度本関数を実行してください。
        return "MaxNfcDeviceActivated";
    }
    else if (nn::nfc::ResultConflictFunction::Includes(result))
    {
        // 拡張バスに接続されたデバイスの取り外しや、アプリケーションが InForcus でなくなった等の理由により、
        // 拡張バスに接続されたデバイスとの通信機能が無効化されています。
        // 再度 Enable からやり直してください。
        return "ConflictFunction";
    }
    else
    {
        return "Unexpected Result";
    }
}

//======================================================================
// 各排他機能の初期化関数
//======================================================================
void nnt::InitializeExclusiveFunc(size_t index)
{
    nn::hid::NpadIdType id = ConvertNpadIdFromIndex(index);
    CurrentStyle[index] = nn::hid::GetNpadStyleSet(id);

    ///////////////////////////////////
    // IRセンサー
    ///////////////////////////////////
    IrHandles[index] = nn::irsensor::GetIrCameraHandle(id);
    nn::irsensor::Initialize(IrHandles[index]);

    if (!IrIsInit)
    {
        nn::irsensor::GetMomentProcessorDefaultConfig(&IrConfig);
        IrIsInit = true;
    }

    ///////////////////////////////////
    // NFC
    ///////////////////////////////////
    if (!NfcIsInit)
    {
        // Framework内で既に初期化済
        //nn::nfc::Initialize();
        nn::nfc::ListDevices(HandleList, &NfcDeviceHandleCount, nn::nfc::DeviceCountMax);
        NfcIsInit = true;
    }

    // NFC デバイスハンドルの更新を行います
    for (int i = 0; i < NfcDeviceHandleCount; i++)
    {
        nn::hid::NpadIdType t_id;
        if ((nn::nfc::GetNpadId(&t_id, HandleList[i]).IsSuccess()) && (t_id == id))
        {
            NfcHandles[index] = HandleList[i];
        }
    }

    ///////////////////////////////////
    // 6軸
    ///////////////////////////////////
    SixAxisHandleCounts[index] = nn::hid::GetSixAxisSensorHandles(SixAxishandles[index],
                                        nn::hid::NpadSixAxisSensorHandleCountMax,
                                        id,
                                        CurrentStyle[index]);

    ///////////////////////////////////
    // HD振動
    ///////////////////////////////////
    vibrationDeviceCount[index] = nn::hid::GetVibrationDeviceHandles(HdVibHandles[index], VibrationDeviceCountMax, id, CurrentStyle[index]);

    // 振動が勝手に止まってしまうのを無効化
    auto ctrl = gController.GetControllerFromIndex(index);
    ctrl->GetVibration().Disable();

    for (int i = 0; i < vibrationDeviceCount[index]; i++)
    {
        // 振動子の初期化
        nn::hid::InitializeVibrationDevice(HdVibHandles[index][i]);
    }
}

//======================================================================
// ボタンの設定
//======================================================================
void nnt::MakeExclusiveFuncButtons(float x, float y)
{
    xOriginEx = x;
    yOriginEx = y + 10.0f;

    ///////////////////////////////
    // IRセンサー
    ///////////////////////////////
    for (size_t idx = 0; idx < NpadIdNum; ++idx)
    {
        for (size_t i = 0; i < NN_ARRAY_SIZE(buttonTextExclusive); ++i)
        {
            nns::hidfw::layout::Button button;
            button.SetPos((50 + xOriginEx) + 70 * i, yOriginEx + 330 * (idx % 2));
            button.SetSize(60, 25);
            button.SetText(buttonTextExclusive[i]);
            button.SetTextColor(nn::util::Color4u8::Black());
            button.ThroughCancel(true);                             // キャンセルを無効化します
            button.ThroughSelect(true);
            button.ThroughChoose(true);
            IrGroup[idx].Add(new nns::hidfw::layout::Button(button));
        }
        IrGroup[idx].FocusItem(0);
        IrGroup[idx].Select(0);
    }

    ///////////////////////////////
    // NFC
    ///////////////////////////////
    for (size_t idx = 0; idx < NpadIdNum; ++idx)
    {
        for (size_t i = 0; i < NN_ARRAY_SIZE(buttonTextExclusive); ++i)
        {
            nns::hidfw::layout::Button button;
            button.SetPos((50 + xOriginEx) + 70 * i, yOriginEx + 40 + 330 * (idx % 2));
            button.SetSize(60, 25);
            button.SetText(buttonTextExclusive[i]);
            button.SetTextColor(nn::util::Color4u8::Black());
            button.ThroughCancel(true);                             // キャンセルを無効化します
            button.ThroughSelect(true);
            button.ThroughChoose(true);
            NfcGroup[idx].Add(new nns::hidfw::layout::Button(button));
        }
        NfcGroup[idx].FocusItem(0);
        NfcGroup[idx].Select(0);
    }

    ///////////////////////////////
    // 6軸
    ///////////////////////////////
    for (size_t idx = 0; idx < NpadIdNum; ++idx)
    {
        for (size_t i = 0; i < NN_ARRAY_SIZE(buttonTextExclusive); ++i)
        {
            nns::hidfw::layout::Button button;
            button.SetPos((50 + xOriginEx) + 70 * i, yOriginEx + 80 + 330 * (idx % 2));
            button.SetSize(60, 25);
            button.SetText(buttonTextExclusive[i]);
            button.SetTextColor(nn::util::Color4u8::Black());
            button.ThroughCancel(true);                             // キャンセルを無効化します
            button.ThroughSelect(true);
            button.ThroughChoose(true);
            SixAxisGroup[idx].Add(new nns::hidfw::layout::Button(button));
        }
        SixAxisGroup[idx].FocusItem(0);
        SixAxisGroup[idx].Select(0);
    }

    ///////////////////////////////
    // HD振動
    ///////////////////////////////
    for (size_t idx = 0; idx < NpadIdNum; ++idx)
    {
        for (size_t i = 0; i < NN_ARRAY_SIZE(buttonTextExclusive); ++i)
        {
            nns::hidfw::layout::Button button;
            button.SetPos((50 + xOriginEx) + 70 * i, yOriginEx + 120 + 330 * (idx % 2));
            button.SetSize(60, 25);
            button.SetText(buttonTextExclusive[i]);
            button.SetTextColor(nn::util::Color4u8::Black());
            button.ThroughCancel(true);                             // キャンセルを無効化します
            button.ThroughSelect(true);
            button.ThroughChoose(true);
            VibGroup[idx].Add(new nns::hidfw::layout::Button(button));
        }
        VibGroup[idx].FocusItem(0);
        VibGroup[idx].Select(0);
    }
}

//======================================================================
// 更新処理
//======================================================================
void nnt::UpdateExclusiveFuncButtons(size_t index)
{
    IrGroup[index].Update();
    NfcGroup[index].Update();
    SixAxisGroup[index].Update();
    VibGroup[index].Update();

    ReInit(index);
}

//======================================================================
// 描画処理
// gDrawer.BeginDraw()～EndDraw()内でコールすることを前提としています
//======================================================================
void nnt::DrawExclusiveFuncButtons(size_t index)
{
    IrGroup[index].Draw();
    NfcGroup[index].Draw();
    SixAxisGroup[index].Draw();
    VibGroup[index].Draw();

    size_t drawPointY = index % 2;

    gTextWriter.SetCursor(xOriginEx, yOriginEx + 3 + (330 * drawPointY));
    gTextWriter.SetTextColor(nn::util::Color4u8::Black());
    gTextWriter.SetScale(1.0f, 1.0f);
    gTextWriter.Print("  IR");

    gTextWriter.SetCursor(xOriginEx, yOriginEx + 43 + (330 * drawPointY));
    gTextWriter.Print(" NFC");

    gTextWriter.SetCursor(xOriginEx, yOriginEx + 83 + (330 * drawPointY));
    gTextWriter.Print("6Axis");

    gTextWriter.SetCursor(xOriginEx, yOriginEx + 123 + (330 * drawPointY));
    gTextWriter.Print(" Vib");
}

//======================================================================
// 各排他機能の有効化
//======================================================================
void nnt::Active(size_t index)
{
    ///////////////////////////////////
    // IRセンサー
    ///////////////////////////////////
    int selIndex;
    IrGroup[index].GetSelectedItemIndex(&selIndex);
    if ((IrIsActive[index] == false) && (selIndex == 1))
    {
        nn::irsensor::RunMomentProcessor(IrHandles[index], IrConfig);
        IrIsActive[index] = true;
    }
    ///////////////////////////////////
    // NFC
    ///////////////////////////////////
    NfcGroup[index].GetSelectedItemIndex(&selIndex);
    if ((NfcIsActive[index] == false) && (selIndex == 1))
    {
        auto deviceState = nn::nfc::GetDeviceState(NfcHandles[selIndex]);
        if (
            deviceState == nn::nfc::DeviceState_Init ||
            deviceState == nn::nfc::DeviceState_Deactive
            )
        {
            if (gController.GetControllerList()[index]->isConnected())
            {
                if (nn::nfc::StartDetection(NfcHandles[index]).IsSuccess())
                {
                    NN_LOG("NFC[ID : %d] is Active\n", index);
                    NfcIsActive[index] = true;
                }
                else
                {
                    if (NfcErrorPrintCounter == 0)
                    {
                        NN_LOG("NFC[ID : %d] is Deactive. Please check Other Excusive Function State.\n", index);
                        NfcErrorPrintCounter++;
                    }
                    else if (NfcErrorPrintCounter >= 30)
                    {
                        NfcErrorPrintCounter = 0;
                    }
                    else
                    {
                        NfcErrorPrintCounter++;
                    }

                }
            }
            else
            {
                NN_LOG("[ID : %d] Controller is Not Connected\n", index);
            }
        }
    }

    ///////////////////////////////////
    // 6軸
    ///////////////////////////////////
    SixAxisGroup[index].GetSelectedItemIndex(&selIndex);
    if (selIndex == 1)
    {
        if (!SixIsActive[index])
        {
            for (int i = 0; i < SixAxisHandleCounts[index]; i++)
            {
                nn::hid::StartSixAxisSensor(SixAxishandles[index][i]);
                if (nn::hid::GetSixAxisSensorStates(&SixAxisStates[index], 1, SixAxishandles[index][i]) != 0)
                {
                    SixIsActive[index] = true;
                }
            }
        }
        else
        {
            if (SixAxisPrintCounter == 0)
            {
                for (int i = 0; i < SixAxisHandleCounts[index]; i++)
                {
                    nn::hid::GetSixAxisSensorStates(&SixAxisStates[index], 1, SixAxishandles[index][i]);
                    NN_LOG("accel(ID：%d) = (%f, %f, %f)\n", index, SixAxisStates[index].acceleration.x, SixAxisStates[index].acceleration.y, SixAxisStates[index].acceleration.z);
                }
                SixAxisPrintCounter++;
            }
            else if (SixAxisPrintCounter >= 60)
            {
                SixAxisPrintCounter = 0;
            }
            else
            {
                SixAxisPrintCounter++;
            }
        }
    }

    ///////////////////////////////////
    // HD振動
    ///////////////////////////////////
    VibGroup[index].GetSelectedItemIndex(&selIndex);
    if ((VibIsActive[index] == false) && (selIndex == 1))
    {
        nn::hid::VibrationDeviceInfo info;
        nn::hid::VibrationValue v1 = nn::hid::VibrationValue::Make(0.5f, 160.0f, 0.5f, 160.0f);
        for (int i = 0; i < vibrationDeviceCount[index]; i++)
        {
            nn::hid::GetVibrationDeviceInfo(&info, HdVibHandles[index][i]);
            nn::hid::SendVibrationValue(HdVibHandles[index][i], v1);
        }
        VibIsActive[index] = true;
    }
}

//======================================================================
// 各排他機能の無効化
//======================================================================
void nnt::Deactive(size_t index)
{
    ///////////////////////////////////
    // IRセンサー
    ///////////////////////////////////
    int selIndex;
    IrGroup[index].GetSelectedItemIndex(&selIndex);
    if ((IrIsActive[index] == true) && (selIndex == 0))
    {
        nn::irsensor::StopImageProcessor(IrHandles[index]);
        IrIsActive[index] = false;
    }

    ///////////////////////////////////
    // NFC
    ///////////////////////////////////
    NfcGroup[index].GetSelectedItemIndex(&selIndex);
    if ((NfcIsActive[index] == true) && (selIndex == 0))
    {
        nn::nfc::StopDetection(NfcHandles[index]);
        NfcIsActive[index] = false;
    }

    ///////////////////////////////////
    // 6軸
    ///////////////////////////////////
    SixAxisGroup[index].GetSelectedItemIndex(&selIndex);
    if ((SixIsActive[index] == true) && (selIndex == 0))
    {
        for (int i = 0; i < SixAxisHandleCounts[index]; i++)
        {
            nn::hid::StopSixAxisSensor(SixAxishandles[index][i]);
        }
        SixIsActive[index] = false;
    }

    ///////////////////////////////////
    // HD振動
    ///////////////////////////////////
    VibGroup[index].GetSelectedItemIndex(&selIndex);
    if ((VibIsActive[index] == true) && (selIndex == 0))
    {
        nn::hid::VibrationDeviceInfo info;
        nn::hid::VibrationValue v1 = nn::hid::VibrationValue::Make();
        for (int i = 0; i < vibrationDeviceCount[index]; i++)
        {
            nn::hid::GetVibrationDeviceInfo(&info, HdVibHandles[index][i]);
            nn::hid::SendVibrationValue(HdVibHandles[index][i], v1);
        }
        VibIsActive[index] = false;
    }
}

//======================================================================
// 再接続やStyle変更があった場合の再初期化
//======================================================================
void nnt::ReInit(size_t index)
{
    nn::hid::NpadIdType id = ConvertNpadIdFromIndex(index);
    nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(id);

    bool isConnected = gController.GetControllerList()[index]->isConnected();

    // Style変更や再接続があった場合はハンドルを失うので、再初期化
    if ((style != CurrentStyle[index]) || ((isConnected == true) && (isConnected != CurrentConnectState[index])))
    {
        // IRは接続なしでハンドルが取得できるので、なにもしない

        ///////////////////////////////////
        // NFC
        ///////////////////////////////////
        nn::nfc::ListDevices(HandleList, &NfcDeviceHandleCount, nn::nfc::DeviceCountMax);
        // NFC デバイスハンドルの更新を行います
        for (int i = 0; i < NfcDeviceHandleCount; i++)
        {
            nn::hid::NpadIdType t_id;
            if ((nn::nfc::GetNpadId(&t_id, HandleList[i]).IsSuccess()) && (t_id == id))
            {
                NfcHandles[index] = HandleList[i];
            }
        }

        ///////////////////////////////////
        // 6軸
        ///////////////////////////////////
        SixAxisHandleCounts[index] = nn::hid::GetSixAxisSensorHandles(SixAxishandles[index],
                                                                        nn::hid::NpadSixAxisSensorHandleCountMax,
                                                                        id,
                                                                        style);
        CurrentStyle[index] = style;

        for (int i = 0; i < SixAxisHandleCounts[index]; i++)
        {
            nn::hid::StartSixAxisSensor(SixAxishandles[index][i]);
            if (nn::hid::GetSixAxisSensorStates(&SixAxisStates[index], 1, SixAxishandles[index][i]) != 0)
            {
                SixIsActive[index] = true;
            }
        }

        ///////////////////////////////////
        // HD振動
        ///////////////////////////////////
        vibrationDeviceCount[index] = nn::hid::GetVibrationDeviceHandles(HdVibHandles[index], VibrationDeviceCountMax, id, style);
        for (int i = 0; i < vibrationDeviceCount[index]; i++)
        {
            nn::hid::InitializeVibrationDevice(HdVibHandles[index][i]);
        }
    }

    ///////////////////////////////////////
    // コントローラ接続状態とStyleの更新
    ///////////////////////////////////////
    CurrentStyle[index] = style;
    CurrentConnectState[index] = isConnected;
}
