﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/system/hid_SixAxisSensorCalibration.h>
#include <nn/hid/system/hid_UniquePad.h>

namespace
{
    template<typename T, ::std::size_t U>
    ::std::size_t GetArrayLength(const T (&staticArray)[U])
    {
        NN_UNUSED(staticArray);
        return U;
    }

    const ::nn::hid::NpadIdType NpadIds[] = {::nn::hid::NpadId::No1,
                                             ::nn::hid::NpadId::No2,
                                             ::nn::hid::NpadId::No3,
                                             ::nn::hid::NpadId::No4,
                                             ::nn::hid::NpadId::No5,
                                             ::nn::hid::NpadId::No6,
                                             ::nn::hid::NpadId::No7,
                                             ::nn::hid::NpadId::No8,
                                             ::nn::hid::NpadId::Handheld};

    const int NpadIdCountMax = static_cast<int>(GetArrayLength(NpadIds));


    void Print(const ::nn::hid::system::UniquePadId* pIds,
               const int& count)
    {
        for(int i = 0; i < count; i++)
        {
            int uniquePadControllerNumber;
            ::nn::hid::system::UniquePadInterface uniquePadInterface;
            ::nn::hid::system::UniquePadType uniquePadType;
            ::nn::hid::system::UniquePadSerialNumber uniquePadSerialNumber;

            auto resultControllerNumber = ::nn::hid::system::GetUniquePadControllerNumber(&uniquePadControllerNumber,
                                                                                          pIds[i]);
            auto resultInterface = ::nn::hid::system::GetUniquePadInterface(&uniquePadInterface,
                                                                            pIds[i]);
            uniquePadType = ::nn::hid::system::GetUniquePadType(pIds[i]);

            auto resultSerialNumber = ::nn::hid::system::GetUniquePadSerialNumber(&uniquePadSerialNumber, pIds[i]);


            if(resultControllerNumber.IsSuccess() &&
               resultInterface.IsSuccess() &&
               resultSerialNumber.IsSuccess())
            {
                NN_LOG("Id=0x%x, ControllerNumber=%d, Interface=%x, Type=%x\n",
                    pIds[i]._storage,
                    resultControllerNumber.IsSuccess() ? uniquePadControllerNumber : -1,
                    resultInterface.IsSuccess() ? uniquePadInterface : -1,
                    uniquePadType);
                NN_LOG("SerialNumber: ");
                for (auto& c : uniquePadSerialNumber._storage)
                {
                    NN_LOG("%c", c);
                }
                NN_LOG("\n");
            }
        }
    }

    void ExecuteUserCalibration(const ::nn::hid::system::UniquePadId* pIds,
                                const int& count)
    {
        for(int i = 0; i < count; i++)
        {
            ::nn::hid::system::UniqueSixAxisSensorHandle handles[::nn::hid::system::UniqueSixAxisSensorHandleCountMax];

            int handleCount = ::nn::hid::system::ListSixAxisSensorHandles(handles,
                                                                          pIds[i],
                                                                          ::nn::hid::system::UniqueSixAxisSensorHandleCountMax);

            for(int j = 0; j < handleCount; j++)
            {
                if(!::nn::hid::system::IsSixAxisSensorUserCalibrationSupported(handles[j]))
                {
                    continue;
                }

                auto result = ::nn::hid::system::ResetSixAxisSensorCalibrationValues(handles[j]);
                NN_LOG("ResetCalibration %s @ handle=%x\n", result.IsSuccess() ? "Success"
                                                                               : "Failure"
                                                          , handles[j]);

                result = ::nn::hid::system::StartSixAxisSensorUserCalibration(handles[j]);
                NN_LOG("UserCalibration %s @ handle=%x\n", result.IsSuccess() ? "Success"
                                                                              : "Failure"
                                                         , handles[j]);
            }
        }
    }

} //namespace

extern "C" void nnMain()
{
    NN_LOG("UniquePad Test Tool Start.\n");

    nn::hid::InitializeNpad();

    nn::hid::SetSupportedNpadIdType(NpadIds, sizeof(NpadIds) / sizeof(NpadIds[0]));

    // Set supported NpadStyle
    ::nn::hid::SetSupportedNpadStyleSet(::nn::hid::NpadStyleFullKey::Mask |
                                        ::nn::hid::NpadStyleJoyDual::Mask |
                                        ::nn::hid::NpadStyleJoyLeft::Mask |
                                        ::nn::hid::NpadStyleJoyRight::Mask);

    ::nn::os::MultiWaitType multiWait;
    ::nn::os::SystemEventType connectionEvent;
    ::nn::os::MultiWaitHolderType connectionHolder;

    ::nn::os::InitializeMultiWait(&multiWait);

    ::nn::hid::system::BindUniquePadConnectionEvent(&connectionEvent,
                                                    ::nn::os::EventClearMode_ManualClear);

    ::nn::os::InitializeMultiWaitHolder(&connectionHolder, &connectionEvent);
    ::nn::os::LinkMultiWaitHolder(&multiWait, &connectionHolder);

    bool runs = true;
    const int counts = 8;
    ::nn::hid::system::UniquePadId uniquePadIds[counts];
    int uniquePadIdCount = 0;

    while(NN_STATIC_CONDITION(runs))
    {
        auto pHolder = nn::os::WaitAny(&multiWait);

        if(pHolder == &connectionHolder)
        {
            ::nn::os::ClearSystemEvent(&connectionEvent);

            // UniquePadId の更新
            uniquePadIdCount = ::nn::hid::system::ListUniquePads(uniquePadIds,
                                                                 counts);
            NN_LOG("--------------\n");
            Print(uniquePadIds, uniquePadIdCount);

            // 接続直後の6軸センサー値は不安定になるため待つ
            NN_LOG("Wait 1sec\n");
            ::nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            ExecuteUserCalibration(uniquePadIds, uniquePadIdCount);
        }
    }

    NN_LOG("UniquePad Test Tool Done.\n");
}
