﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/util/util_Vector.h>

#include "ThrowMotionDetector.h"
#include "SixAxisSensorGestureProcessor.h"

namespace {

} // namespace

ThrowMotionDetector::ThrowMotionDetector() NN_NOEXCEPT
    : m_ThresholdThrowAirAccLengthSquared(0.02f)
    , m_ThresholdThrowEndpointsAccLengthSquared(3.f)
    , m_ThresholdTimeoutDeltaSamplingNumber(100)
    , m_IsThrowStarted(false)
    , m_ThrowStartedRawState()
{
    // 何もしない
}

ThrowMotionDetector::~ThrowMotionDetector() NN_NOEXCEPT
{
    // 何もしない
}

void ThrowMotionDetector::Reset() NN_NOEXCEPT
{
    m_Lifo.Clear();
}

void ThrowMotionDetector::Update(const SixAxisSensorRawState& rawState) NN_NOEXCEPT
{
    auto accLengthSquared = rawState.acc.LengthSquared();

    if (!m_IsThrowStarted &&
        accLengthSquared < m_ThresholdThrowAirAccLengthSquared)
    {
        // NN_LOG("%s: Throw started: %lld\n"
        //     , __FUNCTION__
        //     , rawState.samplingNumber - m_ThrowStartedRawState.samplingNumber);
        m_ThrowStartedRawState = rawState;
        m_IsThrowStarted = true;
    }
    else if (m_IsThrowStarted &&
             m_ThresholdTimeoutDeltaSamplingNumber < rawState.samplingNumber - m_ThrowStartedRawState.samplingNumber)
    {
        // NN_LOG("%s: Throw timeout\n", __FUNCTION__);
        m_IsThrowStarted = false;
    }
    else if(m_IsThrowStarted &&
            accLengthSquared > m_ThresholdThrowEndpointsAccLengthSquared)
    {
        // 着地
        m_IsThrowStarted = false;

        m_Lifo.Append(
            MotionState({
                m_SamplingNumber,
                rawState.samplingNumber,
                MotionType_Throw,
            })
        );

        // NN_LOG("%s: Throw finished\n", __FUNCTION__);

        m_SamplingNumber++;

    }
    else if(m_IsThrowStarted)
    {
        // 空中
        // NN_LOG("%f, %f, %f, %f\n"
        //     , accLengthSquared
        //     , rawState.acc.GetX()
        //     , rawState.acc.GetY()
        //     , rawState.acc.GetZ()
        // );
    }
}

int ThrowMotionDetector::GetMotions(MotionState* pOutStates, int count) NN_NOEXCEPT
{
    return m_Lifo.Read(pOutStates, count);
}
