﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "PalmaUtility.h"

namespace util{

    PalmaActivity innerActivity;                        //!< ツールが保持するActivity設定
    /* ------------------------------------------------------------ */
    // Class Palma
    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    // PUBLIC関数
    /* ------------------------------------------------------------ */
    Palma& Palma::GetInstance()
    {
        static Palma Instance;
        return Instance;
    }

    const char* Palma::ToString(nn::hid::PalmaOperationType type)
    {
        switch (type)
        {
            case nn::hid::PalmaOperationType_PlayActivity:
                return "PlayActivity";
            case nn::hid::PalmaOperationType_SetFrModeType:
                return "SetFrModeType";
            case nn::hid::PalmaOperationType_ReadStep:
                return "ReadStep";
            case nn::hid::PalmaOperationType_EnableStep:
                return "EnableStep";
            case nn::hid::PalmaOperationType_ResetStep:
                return "ResetStep";
            case nn::hid::PalmaOperationType_ReadApplicationSection:
                return "ReadApplicationSection";
            case nn::hid::PalmaOperationType_WriteApplicationSection:
                return "WriteApplicationSection";
            case nn::hid::PalmaOperationType_ReadUniqueCode:
                return "ReadUniqueCode";
            case nn::hid::PalmaOperationType_SetUniqueCodeInvalid:
                return "SetUniqueCodeInvalid";
            case nn::hid::PalmaOperationType_WriteActivityEntry:
                return "WriteActivityEntry";
            case nn::hid::PalmaOperationType_WriteRgbLedPatternEntry:
                return "WriteRgbLedPatternEntry";
            case nn::hid::PalmaOperationType_WriteWaveEntry:
                return "WriteWaveEntry";
            case nn::hid::PalmaOperationType_ReadDataBaseIdentificationVersion:
                return "ReadDataBaseIdentificationVersion";
            case nn::hid::PalmaOperationType_WriteDataBaseIdentificationVersion:
                return "WriteDataBaseIdentificationVersion";
            case nn::hid::PalmaOperationType_SuspendFeature:
                return "SuspendFeature";
            case nn::hid::PalmaOperationType_ReadPlayLog:
                return "ReadPlayLog";
            case nn::hid::PalmaOperationType_ResetPlayLog:
                return "ResetPlayLog";
            default:
                return "Unknown";
        }
    }

    const char* Palma::ToString(nn::hid::PalmaWaveSet waveSet)
    {
        switch (waveSet)
        {
            case nn::hid::PalmaWaveSet_Small:
                return "Small";
            case nn::hid::PalmaWaveSet_Medium:
                return "Medium";
            case nn::hid::PalmaWaveSet_Large:
                return "Large";
            default:
                return "Unknown";
        }
    }

    const char* Palma::ToString(nn::Result result)
    {
        if (result.IsSuccess())
        {
            return "Success";
        }
        else if (nn::hid::ResultPalmaNotConnected::Includes(result))
        {
            // Palma が接続されていません
            return "Device Not Connected";
        }
        else if (nn::hid::ResultPalmaInvalidHandle::Includes(result))
        {
            // 無効な Palma のハンドルが指定されました
            return "Invalid Handle";
        }
        else if (nn::hid::ResultPalmaNoOperationInfo::Includes(result))
        {
            // Palma の操作に対する実行結果がありません
            return "No OperationInfo";
        }
        else if (nn::hid::ResultPalmaBusy::Includes(result))
        {
            // Palma に対して他の操作が行われておりビジー状態です
            return "Busy";
        }
        else if (nn::hid::ResultPalmaUniqueCodeInvalid::Includes(result))
        {
            // Palma の UniqueCode が無効化されています
            return "UniqueCode is Invalid";
        }
        else if (nn::hid::ResultPalmaOperationNotSupported::Includes(result))
        {
            // Palma が 対応していない操作が実行されました
            return "Operation Not Supported";
        }
        else if (nn::hid::ResultPalmaInvalidFieldIndex::Includes(result))
        {
            // Palma の 無効なFieldIndexが指定されました
            return "Invalid Field Index";
        }
        else if (nn::hid::ResultPalmaOperationTimeout::Includes(result))
        {
            // Palma のコマンド通信がタイムアウトしました
            return "Timeout";
        }
        else
        {
            std::string str = "Unexpected result : " + std::to_string((result).GetInnerValueForDebug());
            return str.c_str();
        }
    }

    const char* Palma::ToString(nn::hid::PalmaFrModeType type)
    {
        switch (type)
        {
            case nn::hid::PalmaFrModeType_Off:
                return "OFF";
            case nn::hid::PalmaFrModeType_B01:
                return "B01";
            case nn::hid::PalmaFrModeType_B02:
                return "B02";
            case nn::hid::PalmaFrModeType_B03:
                return "B03";
            case nn::hid::PalmaFrModeType_Downloaded:
                return "Downloaded";
            default:
                return "Unknown";
        }
    }

    void Palma::GetResult(size_t index, const char* apiString, const char* resultString)
    {
        NN_LOG("%s : %s\n", apiString, resultString);
        memcpy(&m_controler[index].apiResult.string, &resultString, 32);
        m_controler[index].apiResult.viewCounter = 255;
    }

    void Palma::Enable(size_t index)
    {
        nn::hid::PalmaConnectionHandle handle;
        nn::hid::GetPalmaConnectionHandle(&handle, gNpad.ConvertNpadIdFromIndex(index));
        // すでに有効だったら何もしない
        if (m_controler[index].IsEnable == true)
        {
            if (handle._storage != m_controler[index].Handle._storage)
            {
                m_controler[index].IsEnable = false;
            }
            else
            {
                return;
            }
        }

        auto result = nn::hid::GetPalmaConnectionHandle(&m_controler[index].Handle, gNpad.ConvertNpadIdFromIndex(index));
        NN_LOG("[PALMA] >> GetPalmaConnectionHandle : %s\n", gPalma.ToString(result));

        result = nn::hid::InitializePalma(m_controler[index].Handle);
        NN_LOG("[PALMA] >> InitializePalma : %s\n", gPalma.ToString(result));

        result = nn::hid::BindPalmaOperationCompleteEvent(m_controler[index].Handle, &m_controler[index].OperationEvent, nn::os::EventClearMode_ManualClear);
        NN_LOG("[PALMA] >> BindPalmaOperationCompleteEvent : %s\n", gPalma.ToString(result));

        m_controler[index].IsEnable = true;
    }

    void Palma::EnableStep(size_t index)
    {
        auto result = nn::hid::EnablePalmaStep(m_controler[index].Handle, true);
        GetResult(index, "[PALMA] >> EnablePalmaStep", gPalma.ToString(result));
    }

    void Palma::DisableStep(size_t index)
    {
        auto result = nn::hid::EnablePalmaStep(m_controler[index].Handle, false);
        GetResult(index, "[PALMA] >> DisablePalmaStep", gPalma.ToString(result));
    }

    void Palma::StartScan()
    {
        nn::hid::EnableAnyPalmaConnection();
        NN_LOG("[TOOLEVENT] >> Scanning Palma Started.\n");
        gPalma.m_isScaning = true;
    }

    void Palma::StopScan()
    {
        nn::hid::DisableAnyPalmaConnection();
        NN_LOG("[TOOLEVENT] >> Scanning Palma Stopped.\n");
        gPalma.m_isScaning = false;
    }

    void Palma::SetPairing(size_t index)
    {
        nn::hid::PairPalma(m_controler[index].Handle);
    }

    void Palma::EnablePairedConnect()
    {
        nn::hid::EnablePairedPalmaConnection();
    }

    void Palma::DisablePairedConnect()
    {
        nn::hid::DisablePairedPalmaConnection();
    }

    void Palma::SetBoostMode(bool isSet)
    {
        nn::hid::SetPalmaBoostMode(isSet);
    }

    void Palma::ResetStep(size_t index)
    {
        auto result = nn::hid::ResetPalmaStep(m_controler[index].Handle);
        GetResult(index, "[PALMA] >> ResetPalmaStep", gPalma.ToString(result));
    }

    void Palma::SuspendFeatureSet(size_t index, nn::hid::PalmaFeatureSet featureSet)
    {
        auto result = nn::hid::SuspendPalmaFeature(m_controler[index].Handle, featureSet);
        GetResult(index, "[PALMA] >> SuspendPalmaFeature", gPalma.ToString(result));
        NN_LOG("[PALMA] >> FrMode : %s    RumbleFeedback : %s    Step : %s    MuteSwitch : %s\n",
                                                        featureSet.Test<nn::hid::PalmaFeature::FrMode>() ? "Suspend" : "ReStart",
                                                        featureSet.Test<nn::hid::PalmaFeature::RumbleFeedback>() ? "Suspend" : "ReStart",
                                                        featureSet.Test<nn::hid::PalmaFeature::Step>() ? "Suspend" : "ReStart",
                                                        featureSet.Test<nn::hid::PalmaFeature::MuteSwitch>() ? "Suspend" : "ReStart");
    }

    void Palma::ReadUniqueCode(size_t index)
    {
        auto result = nn::hid::ReadPalmaUniqueCode(m_controler[index].Handle);
        GetResult(index, "[PALMA] >> ReadPalmaUniqueCode", gPalma.ToString(result));
    }

    void Palma::DisableUniqueCode(size_t index)
    {
        auto result = nn::hid::SetPalmaUniqueCodeInvalid(m_controler[index].Handle);
        GetResult(index, "[PALMA] >> SetPalmaUniqueCodeInvalid", gPalma.ToString(result));
    }

    void Palma::ReadPlayLog(size_t index, uint16_t playLogIndex)
    {
        auto result = nn::hid::ReadPalmaPlayLog(m_controler[index].Handle, playLogIndex);
        GetResult(index, "[PALMA] >> ReadPalmaPlayLog", gPalma.ToString(result));
        NN_LOG("[PALMA] >> Index No : %d\n", playLogIndex);
    }

    void Palma::ResetPlayLog(size_t index, uint16_t playLogIndex)
    {
        auto result = nn::hid::ResetPalmaPlayLog(m_controler[index].Handle, playLogIndex);
        GetResult(index, "[PALMA] >> ResetPalmaPlayLog", gPalma.ToString(result));
        NN_LOG("[PALMA] >> Index No : %d\n", playLogIndex);
    }

    void Palma::GetDataBaseIdVer(size_t index)
    {
        auto result = nn::hid::GetPalmaDataBaseIdentificationVersion(m_controler[index].Handle);
        GetResult(index, "[PALMA] >> GetPalmaDataBaseIdVer", gPalma.ToString(result));
    }

    void Palma::SetDataBaseIdVer(size_t index, int idVer)
    {
        auto result = nn::hid::SetPalmaDataBaseIdentificationVersion(m_controler[index].Handle, idVer);
        GetResult(index, "[PALMA] >> SetPalmaDataBaseIdVer", gPalma.ToString(result));
        NN_LOG("[PALMA] >> IdVer No : %d\n", idVer);
    }

    void Palma::WritePalmaApplicationSection(size_t index, frm::AppSectionElement element)
    {
        nn::Result result;
        nn::fs::FileHandle file;
        int64_t fileSize;
        NN_ALIGNAS(nn::os::MemoryPageSize) nn::hid::PalmaApplicationSectionAccessBuffer workBuffer[nn::hid::PalmaApplicationSectionAccessUnitSizeMax];

        if (element.isFull == false)
        {
            std::string path = gFs.m_mount.type + gFs.m_mount.divChar + gFs.m_mount.path + "/ApplicationData/" + gPalma.m_Appdata.data[element.fileIndex];
            result = nn::fs::OpenFile(&file, path.c_str(), nn::fs::OpenMode_Read);
            if (nn::fs::ResultPathNotFound::Includes(result))
            {
                NN_LOG("[PALMA] >> File(%s) is not found \n", path.c_str());
                return;
            }
            NN_LOG("[PALMA] >> OpenFile is : %s\n", path.c_str());
            nn::fs::GetFileSize(&fileSize, file);

            if (fileSize > 256)
            {
                NN_LOG("[PALMA] >> FileSize Over 256 Bytes.\n");
                nn::fs::CloseFile(file);
                return;
            }
            NN_LOG("[PALMA] >> FileSize : %d\n",fileSize);

            nn::fs::ReadFile(file, 0, workBuffer, static_cast<size_t>(fileSize));
            nn::fs::CloseFile(file);

            result = nn::hid::WritePalmaApplicationSection(m_controler[index].Handle, element.address, fileSize, workBuffer[0]);
            GetResult(index, "[PALMA] >> WritePalmaApplicationSection", gPalma.ToString(result));
            NN_LOG("[PALMA] >> Address : %d    Size : %d\n", element.address, fileSize);
        }
        else
        {
            m_controler[index].IsAllAppData = true;
            AllWritePalmaApplicationSection(index);
        }
    }

    void Palma::AllWritePalmaApplicationSection(size_t index)
    {
        //util::WorkBufferFillTable.raw[0] = m_controler[index].AppDataAllFillCount;
        auto result = nn::hid::WritePalmaApplicationSection(m_controler[index].Handle,
                                                            m_controler[index].AppDataAllFillCount * 256,
                                                            256,
                                                            util::WorkBufferFillTable
                                                            );
        GetResult(index, "[PALMA] >> WritePalmaApplicationSection", gPalma.ToString(result));
        NN_LOG("[PALMA] >> Address : %d    Size : %d\n", m_controler[index].AppDataAllFillCount * 256, 256);
    }

    void Palma::ReadPalmaApplicationSection(size_t index, frm::AppSectionElement element)
    {
        if (element.isFull == false)
        {
            auto result = nn::hid::ReadPalmaApplicationSection(m_controler[index].Handle, element.address, element.readSize);
            GetResult(index, "[PALMA] >> ReadPalmaApplicationSection", gPalma.ToString(result));
            NN_LOG("[PALMA] >> Address : %d    Size : %d\n", element.address, element.readSize);
        }
        else
        {
            m_controler[index].IsAllAppData = true;
            AllReadPalmaApplicationSection(index);
        }
    }

    void Palma::AllReadPalmaApplicationSection(size_t index)
    {
        nn::Result result;

        result = nn::hid::ReadPalmaApplicationSection(m_controler[index].Handle, m_controler[index].AppDataAllFillCount * 256, 256);
        GetResult(index, "[PALMA] >> ReadPalmaApplicationSection", gPalma.ToString(result));
        NN_LOG("[PALMA] >> Address : %d    Size : %d\n", m_controler[index].AppDataAllFillCount * 256, 256);
    }

    void Palma::DumpPalmaApplicationSection(size_t index)
    {
        nn::Result result;
        nn::fs::FileHandle file;

        if (m_controler[index].AppDataAllFillCount == 0)
        {
            const size_t ApplicationHeapSize = 128 * 1024;

            ApplicationHeap applicationHeap;
            applicationHeap.Initialize(ApplicationHeapSize);

            nn::swkbd::ShowKeyboardArg showKeyboardArg;
            nn::swkbd::MakePreset(&(showKeyboardArg.keyboardConfig), nn::swkbd::Preset_Default);

            showKeyboardArg.keyboardConfig.isUseUtf8 = true;

            // 文字列の設定
            const char* guide_string = u8"保存するファイル名を指定してください";
            nn::swkbd::SetGuideTextUtf8(&showKeyboardArg.keyboardConfig, guide_string);

            // 共有メモリ用バッファの割り当て
            size_t in_heap_size = nn::swkbd::GetRequiredWorkBufferSize(false);
            void* swkbd_work_buffer = applicationHeap.Allocate(in_heap_size, nn::os::MemoryPageSize);
            showKeyboardArg.workBuf = swkbd_work_buffer;
            showKeyboardArg.workBufSize = in_heap_size;

            const char* initial_string = u8"data1.txt";
            nn::swkbd::SetInitialTextUtf8(&showKeyboardArg, initial_string);

            // 終了パラメータの設定
            size_t out_heap_size = nn::swkbd::GetRequiredStringBufferSize();
            nn::swkbd::String output_string;
            output_string.ptr = reinterpret_cast<char16_t*>(applicationHeap.Allocate(out_heap_size, nn::os::MemoryPageSize));
            output_string.bufSize = out_heap_size;

            result = nn::swkbd::ShowKeyboard(&output_string, showKeyboardArg);

            memcpy(&m_controler[index].AppDataDumpfileName, &output_string.ptr, 16);

            NN_LOG("[PALMA] >> fileName is %s\n", m_controler[index].AppDataDumpfileName);

            applicationHeap.Deallocate(output_string.ptr);
            applicationHeap.Deallocate(swkbd_work_buffer);

            applicationHeap.Finalize();
        }

        if (!result.IsSuccess())
        {
            NN_LOG("[PALMA] >> KeyBoard Input Canceled.\n");
            return;
        }

        std::string path = gFs.m_mount.type + gFs.m_mount.divChar + gFs.m_mount.path + "/ApplicationData/dump/" + m_controler[index].AppDataDumpfileName;
        result = nn::fs::OpenFile(&file, path.c_str(), nn::fs::OpenMode_AllowAppend | nn::fs::OpenMode_Write);
        if (nn::fs::ResultPathNotFound::Includes(result))
        {
            NN_LOG("[PALMA] >> Create New File.\n");
            nn::fs::CreateFile(path.c_str(), 0);
            result = nn::fs::OpenFile(&file, path.c_str(), nn::fs::OpenMode_AllowAppend | nn::fs::OpenMode_Write);
        }
        NN_LOG("[PALMA] >> OpenFile is : %s\n", path.c_str());

        result = nn::fs::WriteFile(file,
                                   m_controler[index].AppDataAllFillCount * 256,
                                   &m_controler[index].applicationSectionbuffer,
                                   static_cast<size_t>(m_controler[index].applicationSectionSize),
                                   nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag::WriteOptionFlag_Flush));

        nn::fs::CloseFile(file);

        NN_LOG("[PALMA] >> WriteFile[%d] : %s\n", m_controler[index].AppDataAllFillCount * 256, gPalma.ToString(result));

        nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(500));
    }

    void Palma::SetFrModeType(size_t index, int frMode)
    {
        auto result = nn::hid::SetPalmaFrModeType(m_controler[index].Handle, static_cast<nn::hid::PalmaFrModeType>(frMode));
        GetResult(index, "[PALMA] >> SetPalmaFrModeType", gPalma.ToString(result));
        NN_LOG("[PALMA] >> FrMode No : %d\n", frMode);
    }

    void Palma::ReadStep(size_t index)
    {
        auto result = nn::hid::ReadPalmaStep(m_controler[index].Handle);
        GetResult(index, "[PALMA] >> ReadPalmaStep", gPalma.ToString(result));
    }

    void Palma::Update()
    {
        for (auto index = 0; index < frm::NpadIdNum; index++)
        {
            if (m_controler[index].apiResult.viewCounter != 0)
            {
                m_controler[index].apiResult.viewCounter = m_controler[index].apiResult.viewCounter - 5;
            }

            if (m_controler[index].IsEnable)
            {
                m_controlerOld[index] = m_controler[index];

                if (nn::os::TryWaitSystemEvent(&m_controler[index].OperationEvent))
                {
                    nn::os::ClearSystemEvent(&m_controler[index].OperationEvent);
                    // PALMA の実行結果を取得する
                    nn::hid::PalmaOperationInfo palmaOperationInfo;
                    auto result = nn::hid::GetPalmaOperationInfo(&palmaOperationInfo, m_controler[index].Handle);
                    NN_LOG("[PALMA] >> GetPalmaOperationInfo : %s\n", gPalma.ToString(result));
                    NN_LOG("[PALMA] >> Type : %s    Result : %s\n", gPalma.ToString(palmaOperationInfo.type), gPalma.ToString(palmaOperationInfo.result));
                    if (nn::hid::ResultPalmaInvalidHandle::Includes(result))
                    {
                        // 切断などの要因で、PALMA のハンドルが無効になっている
                        NN_LOG("PlayPalmaActivity failed (InvalidHandle).\n");
                        // 対象の PALMA を無効にする
                        if (m_controler[index].IsEnable)
                        {
                            nn::os::DestroySystemEvent(&m_controler[index].OperationEvent);
                            m_controler[index].IsEnable = false;
                        }
                    }
                    if (result.IsSuccess())
                    {
                        switch (palmaOperationInfo.type)
                        {
                            case nn::hid::PalmaOperationType_PlayActivity:
                                if (m_reservedCheckInfo.isChecking)
                                {
                                    nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(1000));
                                    CheckReserved();
                                }
                                break;
                            case nn::hid::PalmaOperationType_ReadStep:
                                m_controler[index].step = palmaOperationInfo.individual.readStep.step;
                                break;
                            case nn::hid::PalmaOperationType_ReadUniqueCode:
                                memcpy(&m_controler[index].uniqueCode[0], &palmaOperationInfo.individual.readUniqueCode.uniqueCode[0], 16);
                                break;
                            case nn::hid::PalmaOperationType_ReadApplicationSection:
                                m_controler[index].applicationSectionbuffer = palmaOperationInfo.individual.readApplicationSection.buffer;
                                m_controler[index].applicationSectionSize = palmaOperationInfo.individual.readApplicationSection.size;
                                NN_LOG("[PALMA] >> Address : %d    Size : %d\n", palmaOperationInfo.individual.readApplicationSection.address, palmaOperationInfo.individual.readApplicationSection.size);
                                gPalma.DumpPalmaApplicationSection(index);
                                if (m_controler[index].IsAllAppData)
                                {
                                    m_controler[index].AppDataAllFillCount++;
                                    if (m_controler[index].AppDataAllFillCount < 16)
                                    {
                                        AllReadPalmaApplicationSection(index);
                                    }
                                    else
                                    {
                                        m_controler[index].IsAllAppData = false;
                                        m_controler[index].AppDataAllFillCount = 0;
                                    }
                                }
                                break;
                            case nn::hid::PalmaOperationType_WriteApplicationSection:
                                NN_LOG("[PALMA] >> Address : %d    Size : %d\n", palmaOperationInfo.individual.writeApplicationSection.address, palmaOperationInfo.individual.writeApplicationSection.size);
                                if (m_controler[index].IsAllAppData)
                                {
                                    m_controler[index].AppDataAllFillCount++;
                                    if (m_controler[index].AppDataAllFillCount < 16)
                                    {
                                        AllWritePalmaApplicationSection(index);
                                    }
                                    else
                                    {
                                        m_controler[index].IsAllAppData = false;
                                        m_controler[index].AppDataAllFillCount = 0;
                                    }
                                }
                                break;
                            case nn::hid::PalmaOperationType_WriteRgbLedPatternEntry:
                                m_controler[index].IsWriting = false;
                                break;
                            case nn::hid::PalmaOperationType_WriteWaveEntry:
                                m_controler[index].IsWriting = false;
                                break;
                            case nn::hid::PalmaOperationType_ReadDataBaseIdentificationVersion:
                                m_controler[index].databaseIdVersion = palmaOperationInfo.individual.readDataBaseIdentificationVersion.version;
                                break;
                            case nn::hid::PalmaOperationType_SuspendFeature:
                                m_controler[index].feature = palmaOperationInfo.individual.suspendPalmaFeature.suspendFeatureSet;
                                break;
                            case nn::hid::PalmaOperationType_ReadPlayLog:
                                memcpy(&m_controler[index].playLog[0], &palmaOperationInfo.individual.readPalmaPlayLog.raw[0], palmaOperationInfo.individual.readPalmaPlayLog.size);
                                int playLogDecimal;
                                memcpy(&playLogDecimal, &m_controler[index].playLog, palmaOperationInfo.individual.readPalmaPlayLog.size);
                                NN_LOG("[PALMA] >> PlayLog : %d(%d [Byte])\n", playLogDecimal, palmaOperationInfo.individual.readPalmaPlayLog.size);
                                break;
                            default:
                                break;
                        }
                    }
                }
            }
        }
    } // NOLINT(impl/function_size)

    PalmaController Palma::GetController(size_t index)
    {
        return m_controler[index];
    }

    nn::Result Palma::WriteActivity(size_t index, frm::ActivityElement activity)
    {
        innerActivity.activityEntry = activity.Entry;

        auto result = nn::hid::WritePalmaActivityEntry(m_controler[index].Handle, activity.Index, &innerActivity.activityEntry);
        GetResult(index, "[PALMA] >> WritePalmaActivityEntry", gPalma.ToString(result));
        NN_LOG("[PALMA] >> Index No : %d    WaveIndex No : %d    RgbLedIndex No : %d\n", activity.Index, activity.Entry.waveIndex, activity.Entry.rgbLedPatternIndex);

        if (nn::hid::ResultPalmaInvalidHandle::Includes(result))
        {
            // 切断などの要因で、PALMA のハンドルが無効になっている
            NN_LOG("WritePalmaActivity failed (InvalidHandle).\n");
            // 対象の PALMA を無効にする
            if (m_controler[index].IsEnable)
            {
                nn::os::DestroySystemEvent(&m_controler[index].OperationEvent);
                m_controler[index].IsEnable = false;
            }
        }
        return result;
    }

    nn::Result Palma::PlayActivity(size_t index, frm::ActivityElement activity)
    {
        auto result = nn::hid::PlayPalmaActivity(m_controler[index].Handle, activity.Index);
        GetResult(index, "[PALMA] >> PlayPalmaActivity", gPalma.ToString(result));
        NN_LOG("[PALMA] >> Index No : %d\n", activity.Index);

        if (nn::hid::ResultPalmaInvalidHandle::Includes(result))
        {
            // 切断などの要因で、PALMA のハンドルが無効になっている
            NN_LOG("PlayPalmaActivity failed (InvalidHandle).\n");
            // 対象の PALMA を無効にする
            if (m_controler[index].IsEnable)
            {
                nn::os::DestroySystemEvent(&m_controler[index].OperationEvent);
                m_controler[index].IsEnable = false;
            }
        }
        return result;
    }

    void Palma::CheckReserved(size_t index, int checkArea)
    {
        if (checkArea == 0)
        {
            m_reservedCheckInfo.startArray = 0;
            m_reservedCheckInfo.endArray = ReservedIndexCount / 3 - 1;
        }
        else if (checkArea == 1)
        {
            m_reservedCheckInfo.startArray = ReservedIndexCount / 3;
            m_reservedCheckInfo.endArray = (ReservedIndexCount / 3) * 2 - 1;
        }
        else if (checkArea == 2)
        {
            m_reservedCheckInfo.startArray = (ReservedIndexCount / 3) * 2;
            m_reservedCheckInfo.endArray = ReservedIndexCount - 1;
        }
        m_reservedCheckInfo.checkIndex = index;
        m_reservedCheckInfo.isChecking = true;

        m_reservedCheckInfo.array = m_reservedCheckInfo.startArray;

        frm::ActivityElement activity;
        activity.Index = ReservedIndex[m_reservedCheckInfo.array];
        PlayActivity(m_reservedCheckInfo.checkIndex, activity);
    }

    void Palma::CheckReserved()
    {
        m_reservedCheckInfo.array++;
        if (m_reservedCheckInfo.array <= m_reservedCheckInfo.endArray)
        {
            frm::ActivityElement activity;
            activity.Index = ReservedIndex[m_reservedCheckInfo.array];
            PlayActivity(m_reservedCheckInfo.checkIndex, activity);
        }
        else
        {
            m_reservedCheckInfo.array       = 0;
            m_reservedCheckInfo.startArray  = 0;
            m_reservedCheckInfo.endArray    = 0;
            m_reservedCheckInfo.checkIndex  = 0;
            m_reservedCheckInfo.isChecking  = false;
        }
    }

    void Palma::WriteWave(size_t pathIndex, size_t handleIndex, frm::WriteElement activity, int fileIndex)
    {
        nn::Result result;
        nn::fs::FileHandle file;
        int64_t fileSize;

        if (m_controler[handleIndex].IsWriting)
        {
            NN_LOG("[PALMA] >> This Command Is Ignored \n");
            return;
        }

        m_controler[handleIndex].IsWriting = true;

        std::string path = gFs.m_mount.type + gFs.m_mount.divChar + gFs.m_mount.path + "/Wave/" + gPalma.m_ActivityFile.wave[pathIndex];
        result = nn::fs::OpenFile(&file, path.c_str(), nn::fs::OpenMode_Read);
        if (nn::fs::ResultPathNotFound::Includes(result))
        {
            // ファイルが存在しない
            NN_LOG("[PALMA] >> File(%s) is not found \n", path.c_str());
            return;
        }
        NN_LOG("[PALMA] >> OpenFile is : %s\n", path.c_str());

        nn::fs::GetFileSize(&fileSize, file);
        auto alignedBufferSize = (fileSize / nn::os::MemoryPageSize + 1) * nn::os::MemoryPageSize;
        NN_ABORT_UNLESS_LESS_EQUAL(alignedBufferSize, sizeof(gFs.m_WorkBuffer));

        nn::fs::ReadFile(file, 0, gFs.m_WorkBuffer, static_cast<size_t>(fileSize));

        nn::fs::CloseFile(file);

        result = nn::hid::WritePalmaWaveEntry(m_controler[handleIndex].Handle,
                                              activity.Entry.waveSet,
                                              activity.Entry.waveIndex,
                                              gFs.m_WorkBuffer,
                                              alignedBufferSize,
                                              fileSize
                                              );
        GetResult(handleIndex, "[PALMA] >> WritePalmaWaveEntry", gPalma.ToString(result));

        if (nn::hid::ResultPalmaInvalidHandle::Includes(result))
        {
            // 切断などの要因で、PALMA のハンドルが無効になっている
            NN_LOG("PlayPalmaActivity failed (InvalidHandle).\n");

            // 対象の PALMA を無効にする
            if (m_controler[handleIndex].IsEnable)
            {
                // PALMA がほかの BLE 操作を処理中
                NN_LOG("PlayPalmaActivity failed (Busy).\n");
            }
        }
    }

    void Palma::WriteRgbLed(size_t pathIndex, size_t handleIndex, frm::WriteElement activity, int fileIndex)
    {
        nn::Result result;
        nn::fs::FileHandle file;
        int64_t fileSize;

        if (m_controler[handleIndex].IsWriting)
        {
            NN_LOG("[PALMA] >> This Command Is Ignored \n");
            return;
        }

        m_controler[handleIndex].IsWriting = true;

        std::string path = gFs.m_mount.type + gFs.m_mount.divChar + gFs.m_mount.path + "/RgbLed/" + gPalma.m_ActivityFile.rgbLed[pathIndex];
        result = nn::fs::OpenFile(&file, path.c_str(), nn::fs::OpenMode_Read);
        if (nn::fs::ResultPathNotFound::Includes(result))
        {
            NN_LOG("[PALMA] >> File(%s) is not found \n", path.c_str());
            return;
        }
        NN_LOG("[PALMA] >> OpenFile is : %s\n", path.c_str());

        nn::fs::GetFileSize(&fileSize, file);

        nn::fs::ReadFile(file, 0, gFs.m_WorkBuffer, static_cast<size_t>(fileSize));

        nn::fs::CloseFile(file);

        result = nn::hid::WritePalmaRgbLedPatternEntry(m_controler[handleIndex].Handle,
                                                       activity.Entry.rgbLedPatternIndex,
                                                       gFs.m_WorkBuffer,
                                                       fileSize
        );
        GetResult(handleIndex, "[PALMA] >> WritePalmaRgbLedPatternEntry", gPalma.ToString(result));

        if (nn::hid::ResultPalmaInvalidHandle::Includes(result))
        {
            // 切断などの要因で、PALMA のハンドルが無効になっている
            NN_LOG("PlayPalmaActivity failed (InvalidHandle).\n");

            // 対象の PALMA を無効にする
            if (m_controler[handleIndex].IsEnable)
            {
                // PALMA がほかの BLE 操作を処理中
                NN_LOG("PlayPalmaActivity failed (Busy).\n");
            }
        }
    }
}
