﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @brief
    Npadのアサイン切替機能
 */

#include "NpadAssignControl.h"

namespace {
    const char* FunctionText[] =
    {
        "Swap",
        "Merge",
        "DivideL",
        "DivideR",
        "SetSingle",
        "SetDual",
        "Disconnect",
        "SupportID",
        "SupportStyle",
    };
    const int FunctionTextSize = NN_ARRAY_SIZE(FunctionText);

    enum SelectState
    {
        SelectState_SelFunc = 0,
        SelectState_SelFirstNpad,
        SelectState_SelSecondNpad,
        SelectState_SelSupportNpadId,
        SelectState_SelSupportNpadStyle,
        SelectStateCount,
    };

    int IdState = 0;
    nn::hid::NpadIdType npadIds[9];
    int count = 0;

    int StyleState = 0;
    nn::hid::NpadStyleSet style;

    struct AssignmentState
    {
        SelectState selstate;
        size_t func;
        size_t firstNpad;
        size_t secondNpad;
        bool SupportNpadIdList[9];
        bool SupportNpadStyleList[6];
        bool isSelected;
    };

    AssignmentState state;

    int drawSmoothCount;

    uint8_t m_TwinkleCounter = 0;

    bool up = false;
    bool down = false;
    bool fix = false;
    bool cancel = false;

    struct DrawPotition
    {
        float         X = 1280.f;
        const float   Y = 240.f;
        const float   Width = 300.f;
        const float   Height = 300.f;
    };

    DrawPotition m_DrawPotition;

    int m_SwipeCounter = 0;
    int m_SwipeCounterState = 0;

    float m_DeltaX = 0.f;
}

/////////////////////////////////
// Style変換
/////////////////////////////////
const char* ToString(size_t index)
{
    switch (index)
    {
        case 0:
            return "JoyLeft";
        case 1:
            return "JoyRight";
        case 2:
            return "JoyDual";
        case 3:
            return "FullKey";
        case 4:
            return "HandHeld";
        case 5:
            return "Palma";
        default:
            return "Undefined";
    }
}

void CountSmooth()
{
    if (up == true || down == true)
    {
        drawSmoothCount = 5;
    }

    drawSmoothCount = (drawSmoothCount == 0) ? 0 : drawSmoothCount < 0 ? drawSmoothCount + 1 : drawSmoothCount - 1;
}

void UpdateTwinkleCounter()
{
    if (m_TwinkleCounter < 250)
    {
        m_TwinkleCounter = m_TwinkleCounter + 5;
    }
    else if (m_TwinkleCounter >= 250)
    {
        m_TwinkleCounter = 100;
    }
}

void UpdateSwipeCounter(int functionMenu)
{
    m_SwipeCounterState = functionMenu == 1 ? 1 : 2;

    if (m_SwipeCounterState == 1)
    {
        if (m_SwipeCounter < 10)
        {
            m_SwipeCounter++;
        }
        // 0→Widthに向かう感じ
        m_DeltaX = m_DrawPotition.Width - m_DrawPotition.Width / (static_cast<float>(m_SwipeCounter) * 30.f);
    }
    else if (m_SwipeCounterState == 2)
    {
        if (m_SwipeCounter > 0)
        {
            m_SwipeCounter--;
        }
        else
        {
            m_SwipeCounterState = 0;
            m_DeltaX = 0.f;
        }
        // Width→0に向かう感じ
        m_DeltaX = pow(static_cast<double>(m_SwipeCounter), 6.0) * (m_DrawPotition.Width / pow(10.0, 6.0));
    }
}

/////////////////////////////////////////////
// サポートするNpadIdのダンプ
/////////////////////////////////////////////
void DumpNpadId()
{
    for (auto index = 0; index < frm::NpadIdNum; index++)
    {
        if (state.SupportNpadIdList[index])
        {
            npadIds[count] = gNpad.ConvertNpadIdFromIndex(index);
            count++;
        }
    }
}

//////////////////////////////////////////////
// サポートするNpadIdの設定
//////////////////////////////////////////////
void SelectedSupportNpadId()
{
    if (up == true || down == true)
    {
        state.SupportNpadIdList[IdState] = (state.SupportNpadIdList[IdState] ? false : true);
    }
    if (fix == true)
    {
        if (IdState >= 8)
        {
            IdState = 0;
            state.isSelected = true;
            state.selstate = SelectState_SelFunc;
            DumpNpadId();
        }
        else
        {
            IdState++;
        }
    }
    if (cancel == true)
    {
        if (IdState == 0)
        {
            state.selstate = SelectState_SelFunc;
        }
        else
        {
            IdState--;
        }
    }
}

//////////////////////////////////////////////
// サポートするNpadStyleのダンプ
//////////////////////////////////////////////
void DumpNpadStyle()
{
    for (auto i = 0; i < 6; i++)
    {
        if (state.SupportNpadStyleList[i])
        {
            switch (i)
            {
                case 0:
                    style |= nn::hid::NpadStyleJoyLeft::Mask;
                    break;
                case 1:
                    style |= nn::hid::NpadStyleJoyRight::Mask;
                    break;
                case 2:
                    style |= nn::hid::NpadStyleJoyDual::Mask;
                    break;
                case 3:
                    style |= nn::hid::NpadStyleFullKey::Mask;
                    break;
                case 4:
                    style |= nn::hid::NpadStyleHandheld::Mask;
                    break;
                case 5:
                    style |= nn::hid::NpadStylePalma::Mask;
                    break;
                default:
                    // do nothing
                    break;
            }
        }
    }
}

//////////////////////////////////////////////
// サポートするNpadStyleの設定
//////////////////////////////////////////////
void SelectedSupportNpadStyle()
{
    if (up == true || down == true)
    {
        state.SupportNpadStyleList[StyleState] = (state.SupportNpadStyleList[StyleState] ? false : true);
    }
    if (fix == true)
    {
        if (StyleState >= 5)
        {
            StyleState = 0;
            state.isSelected = true;
            state.selstate = SelectState_SelFunc;
            DumpNpadStyle();
        }
        else
        {
            StyleState++;
        }
    }
    if (cancel == true)
    {
        if (StyleState == 0)
        {
            state.selstate = SelectState_SelFunc;
        }
        else
        {
            StyleState--;
        }
    }
}

/////////////////////////////////
// 選択状況に応じたAPIの実行
/////////////////////////////////
void DoNpadAssign()
{
    if (state.isSelected)
    {
        auto firstId = gNpad.ConvertNpadIdFromIndex(state.firstNpad);
        auto secondId = gNpad.ConvertNpadIdFromIndex(state.secondNpad);

        switch (state.func)
        {
            case 0:         //!< Swap
                nn::hid::SwapNpadAssignment(firstId, secondId);
                break;
            case 1:         //!< Merge
                nn::hid::MergeSingleJoyAsDualJoy(firstId, secondId);
                break;
            case 2:         //!< DivideL
                nn::hid::SetNpadJoyAssignmentModeSingle(firstId, nn::hid::NpadJoyDeviceType_Left);
                break;
            case 3:         //!< DivideR
                nn::hid::SetNpadJoyAssignmentModeSingle(firstId, nn::hid::NpadJoyDeviceType_Right);
                break;
            case 4:         //!< SetSingle
                nn::hid::SetNpadJoyAssignmentModeSingle(firstId);
                break;
            case 5:         //!< SetDual
                nn::hid::SetNpadJoyAssignmentModeDual(firstId);
                break;
            case 6:         //!< Disconnect
                nn::hid::DisconnectNpad(firstId);
                break;
            case 7:         //!< SupportId
                nn::hid::SetSupportedNpadIdType(npadIds, count);
                break;
            case 8:         //!< SupportStyle
                nn::hid::SetSupportedNpadStyleSet(style);
                break;
            default:
                break;
        }
    }
}

/////////////////////////////
// アサイン状況の選択
/////////////////////////////
void NpadAssignmentControl(frm::NpadState* npadState)
{
    up = false;
    down = false;
    fix = false;
    cancel = false;
    state.isSelected = false;

    count = 0;

    for (auto i = 0; i < 9; i++)
    {
            npadIds[i] = 0;
    }
    style.Reset();

    // 接続中のコントローラのボタンの押下げ状況を確認します
    for (auto index = 0; index < frm::NpadIdNum; index++)
    {
        if (npadState[index].trigger.Test<nn::hid::NpadButton::StickLDown>()
        ||  npadState[index].trigger.Test<nn::hid::NpadButton::StickRDown>()
        ){
            down = true;
        }
        if (npadState[index].trigger.Test<nn::hid::NpadButton::StickLUp>()
        ||  npadState[index].trigger.Test<nn::hid::NpadButton::StickRUp>()
        ){
            up = true;
        }
        if (npadState[index].trigger.Test<nn::hid::NpadButton::StickL>()
        || npadState[index].trigger.Test<nn::hid::NpadButton::StickR>()
        ||  npadState[index].trigger.Test<nn::hid::NpadButton::L>()
        ||  npadState[index].trigger.Test<nn::hid::NpadButton::R>()
        ){
            fix = true;
        }
        if (npadState[index].trigger.Test<nn::hid::NpadPalmaButton::Palma>()
        ||  npadState[index].trigger.Test<nn::hid::NpadButton::ZL>()
        ||  npadState[index].trigger.Test<nn::hid::NpadButton::ZR>()
        ){
            cancel = true;
        }
    }

    switch (state.selstate)
    {
        case SelectState_SelFunc:
            if (down == true)
            {
                state.func = (state.func == FunctionTextSize - 1) ? 0 : state.func + 1;
            }
            if (up == true)
            {
                state.func = (state.func == 0) ? state.func = state.func = FunctionTextSize - 1 : state.func - 1;
            }
            // SupportID、SupportStyle以外の場合
            if (fix == true && state.func != 7 && state.func != 8)
            {
                state.selstate = SelectState_SelFirstNpad;
            }
            // SupportIDの場合
            else if (fix == true && state.func == 7)
            {
                state.selstate = SelectState_SelSupportNpadId;
            }
            // SupportStyleの場合
            else if (fix == true && state.func == 8)
            {
                state.selstate = SelectState_SelSupportNpadStyle;
            }
            else
            {
                // do nothing
            }

            if (cancel == true)
            {
                // do nothing
            }
            break;
        case SelectState_SelFirstNpad:
            if (down == true)
            {
                state.firstNpad = ((state.firstNpad < 8) ? (state.firstNpad + 1) : 0);
            }
            if (up == true)
            {
                state.firstNpad = ((state.firstNpad > 0) ? (state.firstNpad - 1) : 8);
            }
            if (fix == true)
            {
                // // DivideL、DivideR、SetSingle、SetDual、Disconnectの場合
                if (state.func == 2
                 || state.func == 3
                 || state.func == 4
                 || state.func == 5
                 || state.func == 6)
                {
                    state.isSelected = true;
                    state.selstate = SelectState_SelFunc;
                }
                else
                {
                    state.selstate = SelectState_SelSecondNpad;
                }
            }
            if (cancel == true)
            {
                state.selstate = SelectState_SelFunc;
            }
            break;
        case SelectState_SelSecondNpad:
            if (down == true)
            {
                state.secondNpad = ((state.secondNpad < 8) ? (state.secondNpad + 1) : 0);
            }
            if (up == true)
            {
                state.secondNpad = ((state.secondNpad > 0) ? (state.secondNpad - 1) : 8);
            }
            if (fix == true)
            {
                state.isSelected = true;
                state.selstate = SelectState_SelFunc;
            }
            if (cancel == true)
            {
                state.selstate = SelectState_SelFirstNpad;
            }
            break;
        case SelectState_SelSupportNpadId:
            SelectedSupportNpadId();
            break;
        case SelectState_SelSupportNpadStyle:
            SelectedSupportNpadStyle();
            break;
        default:
            break;
    }
    DoNpadAssign();
} // NOLINT(impl/function_size)



/////////////////////////////////
// 選択状況に応じた描画
/////////////////////////////////
void DrawNpadAssignMenu(GraphicsSystem* pGraphicsSystem,
                        nn::gfx::util::DebugFontTextWriter* pTextWriter,
                        int functionMenu)
{
    CountSmooth();
    UpdateTwinkleCounter();

    UpdateSwipeCounter(functionMenu);

    m_DrawPotition.X = 1280.f - m_DeltaX - 5.f;

    nn::util::Color4u8 color = { 50,  255, 255, m_TwinkleCounter };

    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

    int tempDrawSmoothCount = drawSmoothCount;

    // 枠の作成
    pPrimitiveRenderer->SetColor(Color::Gray);
    pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(),
                                                        m_DrawPotition.X,
                                                        m_DrawPotition.Y,
                                                        m_DrawPotition.Width,
                                                        m_DrawPotition.Height);

    pPrimitiveRenderer->SetLineWidth(1.f);
    pPrimitiveRenderer->SetColor(Color::Black);
    pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(),
                                                        m_DrawPotition.X,
                                                        m_DrawPotition.Y,
                                                        m_DrawPotition.Width,
                                                        m_DrawPotition.Height);

    pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(),
                                                        m_DrawPotition.X + 3,
                                                        m_DrawPotition.Y + 3,
                                                        m_DrawPotition.Width - 6,
                                                        m_DrawPotition.Height - 6);

    pTextWriter->SetScale(1.0, 1.0);

    int DeltaX = 10;
    int DeltaY = 10;

    // TOP項目の作成
    pTextWriter->SetTextColor(Color::Black);
    pTextWriter->SetCursor(m_DrawPotition.X + DeltaX, m_DrawPotition.Y + DeltaY - 5);
    pTextWriter->Print("[Npad Assign Selector]");

    DeltaX = 25;
    DeltaY = 30;

    switch (state.selstate)
    {
        case SelectState_SelSecondNpad:
            pTextWriter->SetScale(1.f, 1.f * (1 + tempDrawSmoothCount * 0.1));
            pTextWriter->SetCursor(m_DrawPotition.X + DeltaX * 2, m_DrawPotition.Y + DeltaY * 3);
            pTextWriter->Print("%s", gNpad.ToString(gNpad.ConvertNpadIdFromIndex(state.secondNpad)));
            tempDrawSmoothCount = 0;
            // through down
        case SelectState_SelFirstNpad:
            pTextWriter->SetScale(1.f, 1.f * (1 + tempDrawSmoothCount * 0.1));
            pTextWriter->SetCursor(m_DrawPotition.X + DeltaX * 2, m_DrawPotition.Y + DeltaY * 2);
            pTextWriter->Print("%s", gNpad.ToString(gNpad.ConvertNpadIdFromIndex(state.firstNpad)));
            tempDrawSmoothCount = 0;
            // through down
        case SelectState_SelFunc:
            pTextWriter->SetScale(1.f, 1.f * (1 + tempDrawSmoothCount * 0.1));
            pTextWriter->SetCursor(m_DrawPotition.X + DeltaX * 1, m_DrawPotition.Y + DeltaY * 1);
            pTextWriter->Print("%s", FunctionText[state.func]);
            tempDrawSmoothCount = 0;
            break;
        case SelectState_SelSupportNpadId:
            for (auto index = 0; index < 9; index++)
            {
                if (IdState >= index)
                {
                    if (IdState == index)
                    {
                        pTextWriter->SetScale(1.f, 1.f * (1 + tempDrawSmoothCount * 0.1));
                    }
                    else
                    {
                        pTextWriter->SetScale(1.f, 1.f);
                    }
                    pTextWriter->SetCursor(m_DrawPotition.X + DeltaX * 2, m_DrawPotition.Y + DeltaY * 2 + ((DeltaY - 5) * index));
                    pTextWriter->Print("%s：%s", gNpad.ToString(gNpad.ConvertNpadIdFromIndex(index)), (state.SupportNpadIdList[index] ? "ON" : "OFF"));
                }
            }
            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetCursor(m_DrawPotition.X + DeltaX * 1, m_DrawPotition.Y + DeltaY * 1);
            pTextWriter->Print("%s", FunctionText[state.func]);
            break;
        case SelectState_SelSupportNpadStyle:
            for (auto index = 0; index < 6; index++)
            {
                if (StyleState >= index)
                {
                    if (StyleState == index)
                    {
                        pTextWriter->SetScale(1.f, 1.f * (1 + tempDrawSmoothCount * 0.1));
                    }
                    else
                    {
                        pTextWriter->SetScale(1.f, 1.f);
                    }
                    pTextWriter->SetCursor(m_DrawPotition.X + DeltaX * 2, m_DrawPotition.Y + DeltaY * 2 + (DeltaY * index));
                    pTextWriter->Print("%s：%s", ToString(index), (state.SupportNpadStyleList[index] ? "ON" : "OFF"));
                }
            }
            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetCursor(m_DrawPotition.X + DeltaX * 1, m_DrawPotition.Y + DeltaY * 1);
            pTextWriter->Print("%s", FunctionText[state.func]);
            break;
        default:
            break;
    }
}
