﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <cstdlib>
#include <string>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/hid/hid_PalmaTypes.h>
#include "Npad.h"
#include "Gesture.h"

namespace frm {

    const int SoundIndexMax = 224;
    const int RgbLedIndexMax = 512;
    const int ActivityIndexMax = 512;
    const int PlayLogIndexMax = 256;
    const int AppSectionSizeMax = 256;

    enum CursolStatus
    {
        CursolStatus_Top = 0,
        CursolStatus_FixedNpad,
        CursolStatus_FixedActivity,
        CursolStatus_FixedNpadFunction,
        CursolStatus_FixedNpadStartScan,
        CursolStatus_FixedNpadWriteActivity,
        CursolStatus_FixedNpadPlayActivity,
        CursolStatus_FixedNpadCheckReserved,
        CursolStatus_FixedNpadReadPlayLog,
        CursolStatus_FixedNpadResetPlayLog,
        CursolStatus_FixedNpadReadAppSection,
        CursolStatus_FixedNpadWriteAppSection,
        CursolStatus_FixedNpadWriteDataBaseIdVer,
        CursolStatus_FixedNpadSetFrModeType,
        CursolStatus_FixedNpadSuspendFeatureSet,
        CursolStatus_FixedNpadSetBoostMode,
        CursolStatus_FixedActivitySound,
        CursolStatus_FixedActivityRgbLed,
        CursolStatus_WriteSound,
        CursolStatus_WriteRgbLed,
        CursolStatus_DoNpadFunction,
    };

    static const char* NpadMenu[] = {
        "Start Scan",
        "Stop Scan",
        "Set Pair",
        "Reset Pair(ForDebug)",
        "Enable PairedConnect",
        "Disable PairedConnect",
        "Play Activity",
        "Play(Check Reserved)",
        "Write Activity",
        "Enable Step",
        "Read Step",
        "Disable Step",
        "Reset Step",
        "Set FrModeType",
        "Suspend FeatureSet",
        "Read UniqueCode",
        "Diable UniqueCode",
        "Read PlayLog",
        "Reset PlayLog",
        "Read AppSection",
        "Write AppSection",
        "Get DataBaseIdVer.",
        "Write DataBaseIdVer.",
        "*********",        // Set BoostMode
        "Play IrSensor",
        "Play Vibration",
    };
    const int NpadMenuNum = NN_ARRAY_SIZE(NpadMenu);

    enum NpadFunctionSelect
    {
        NpadFunctionSelect_StartScan = 0,
        NpadFunctionSelect_StopScan,
        NpadFunctionSelect_SetPairing,
        NpadFunctionSelect_ResetPairing,
        NpadFunctionSelect_EnablePairedConnect,
        NpadFunctionSelect_DisablePairedConnect,
        NpadFunctionSelect_PlayActivity,
        NpadFunctionSelect_CheckReserved,
        NpadFunctionSelect_WriteActivity,
        NpadFunctionSelect_EnableStep,
        NpadFunctionSelect_ReadStep,
        NpadFunctionSelect_DisableStep,
        NpadFunctionSelect_ResetStep,
        NpadFunctionSelect_SetFrModeType,
        NpadFunctionSelect_SuspendFeatureSet,
        NpadFunctionSelect_ReadUniqueCode,
        NpadFunctionSelect_DiableUniqueCode,
        NpadFunctionSelect_ReadPlayLog,
        NpadFunctionSelect_ResetPlayLog,
        NpadFunctionSelect_ReadAppSection,
        NpadFunctionSelect_WriteAppSection,
        NpadFunctionSelect_GetDataBaseIdVer,
        NpadFunctionSelect_WriteDataBaseIdVer,
        NpadFunctionSelect_SetBoostMode,
        NpadFunctionSelect_PlayIrSensor,
        NpadFunctionSelect_PlayVibration,
        NpadFunctionSelect_NoOperate,
    };

    enum WriteFunctionSelect
    {
        ActivityFunctionSelect_ActivityFilePathSelect = 0,
        ActivityFunctionSelect_WriteSound,
        ActivityFunctionSelect_WriteRgbLed,
        ActivityFunctionSelect_LoadFile,
        ActivityFunctionSelect_NoOperate,
    };

    struct AppSectionElement
    {
        int             position;
        int             fileIndex;
        int             address;
        int             readSize;
        bool            isFull = false;
    };

    // Npadに対し、Activityの再生、Activityエントリの書き込み場所を指定する為の構造体
    struct ActivityElement
    {
        nn::hid::PalmaActivityEntry     Entry;
        nn::hid::PalmaActivityIndex     Index;
    };

    // 指定したNpadに対し、SoundないしRgbLed書き込みを指定する為の構造体
    struct WriteElement
    {
        nn::hid::PalmaActivityIndex     npadIndex;
        nn::hid::PalmaActivityEntry     Entry;
    };

    struct CursolPosition
    {
        bool                            topForcus;
        int                             npadNo;
        int                             npadFunction[frm::NpadIdNum];
        int                             scanDeviseType;
        int                             activity;
        int                             sound;
        int                             rgbLed;
        int                             dataBaseIdVer[frm::NpadIdNum];
        AppSectionElement               appSection[frm::NpadIdNum];
        WriteElement                    writeElement;
        int                             writePosition;
        ActivityElement                 activityElement[frm::NpadIdNum];
        int                             activityPosition[frm::NpadIdNum];
        int                             featureSetPosition[frm::NpadIdNum];
        nn::hid::PalmaFeatureSet        featureSet[frm::NpadIdNum];
        int                             frMode[frm::NpadIdNum];
        uint16_t                        playLogIndex[frm::NpadIdNum];
        int                             reservedCheckIndex[frm::NpadIdNum];
        bool                            isAutoConnectMode[frm::NpadIdNum];
        bool                            isBoostMode[frm::NpadIdNum];

        CursolStatus                    CursolStatus;

        WriteFunctionSelect             writeFunctionFixed;
        NpadFunctionSelect              npadFunctionFixed[frm::NpadIdNum];
    };

    class Cursol
    {
    private:
        struct ButtonSelect
        {
            bool up;
            bool down;
            bool left;
            bool right;
            bool fix;
            bool cancel;
        };

        CursolPosition m_cursol;

        void Top(ButtonSelect buttonSelect, frm::GestureState gestureState);

        void FixedActivity(ButtonSelect buttonSelect);

        void FixedActivitySound(ButtonSelect buttonSelect, int cursolCount);

        void WriteSound(ButtonSelect buttonSelect);

        void FixedActivityRgbLed(ButtonSelect buttonSelect, int cursolCount);

        void WriteRgbLed(ButtonSelect buttonSelect);

        void FixedNpad(ButtonSelect buttonSelect, frm::GestureState gestureState);

        void FixedNpadFunction(ButtonSelect buttonSelect);

        void FixedNpadStartScan(ButtonSelect buttonSelect);

        void FixedNpadSetFrModeType(ButtonSelect buttonSelect);

        void FixedNpadSuspendFeatureSet(ButtonSelect buttonSelect);

        void FixedNpadWriteDataBaseIdVer(ButtonSelect buttonSelect);

        void FixedNpadReadPlayLog(ButtonSelect buttonSelect);

        void FixedNpadResetPlayLog(ButtonSelect buttonSelect);

        void FixedNpadReadAppSection(ButtonSelect buttonSelect);

        void FixedNpadWriteAppSection(ButtonSelect buttonSelect, int appSectionCount);

        void FixedNpadSetBoostMode(ButtonSelect buttonSelect);

        void FixedNpadWriteActivity(ButtonSelect buttonSelect);

        void FixedNpadPlayActivity(ButtonSelect buttonSelect);

        void FixedNpadCheckReserved(ButtonSelect buttonSelect);

        void DoNpadFunction(void);

    public:

        //----------------------------------------------------------------
        // @brief   インスタンス取得
        //----------------------------------------------------------------
        static Cursol& GetInstance() NN_NOEXCEPT;

        void Initialize();

        void Update(frm::NpadState * npadState, frm::GestureState gestureState, int waveCursolCount, int rgbLedCursolCount, int appSectionCount);

        CursolPosition GetCursol(void);
    };
}

#ifndef gCursol
#define gCursol (frm::Cursol::GetInstance())
#endif
