﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "BleUtility.h"

namespace util{

    typedef void(*Callback)();

    BleEvent            event;                          //!< BLE関連のシステムイベント群
    BleMultiWaitHolder  multiWaitHolder;                //!< BLE関連の多重待ちイベントをホールドする変数

    /* ------------------------------------------------------------ */
    // クラス外関数
    /* ------------------------------------------------------------ */
    void ScanHandler()
    {
        nn::os::ClearSystemEvent(&event.scan);

        if (gBle.m_IsScanningSmartDevice)
        {
            nn::bluetooth::BleScanResult scanResult;
            // スキャンできたスマートデバイスを1ずつ取得
            auto scanResultCount = nn::bluetooth::GetBleScanResult(&scanResult, 1);
            if (scanResultCount > 0)
            {
                for (auto i = 0; i < scanResultCount; i++)
                {
                    if (nn::bluetooth::ResultBleConnectionFull().Includes(nn::bluetooth::ConnectToGattServer(scanResult.address)))
                    {
                        NN_LOG("[BLE] >> Ble Connection Full\n");
                    }
                }
            }
        }
    }

    void ConnectionHandler()
    {
        nn::os::ClearSystemEvent(&event.connection);

        // 接続イベント情報の取得
        gBle.m_bleState.connectionInfo.dataNum = nn::bluetooth::GetBleConnectionInfoList(gBle.m_bleState.connectionInfo.data, NN_ARRAY_SIZE(gBle.m_bleState.connectionInfo.data));
        NN_LOG("[BLE] >> ConnectionInfoCount = %d\n", gBle.m_bleState.connectionInfo.dataNum);
    }

    void ServiceDiscoveryHandler()
    {
        nn::os::ClearSystemEvent(&event.serviceDiscovery);
        for (int i = 0; i < gBle.m_bleState.connectionInfo.dataNum; ++i)
        {
            // Gattサーバー情報の取得
            gBle.m_bleState.connectionInfo.Serv[i].dataNum =
                        nn::bluetooth::GetGattServices(gBle.m_bleState.connectionInfo.Serv[i].data,
                                                        16,
                                                        gBle.m_bleState.connectionInfo.data[i].connectionHandle);

            NN_LOG("[BLE] >> GetGattServices Count = %d\n", gBle.m_bleState.connectionInfo.Serv[i].dataNum);

            // GattOperationの登録
            for (int count = 0; count < gBle.m_bleState.connectionInfo.Serv[i].dataNum; ++count)
            {
                gBle.RegisterGattOperation(&gBle.m_bleState.connectionInfo.Serv[i].data[count]);
            }

            for (int j = 0; j < gBle.m_bleState.connectionInfo.Serv[i].dataNum; ++j)
            {
                gBle.m_bleState.connectionInfo.Serv[i].Char[j].dataNum = gBle.m_bleState.connectionInfo.Serv[i].data[j].GetCharacteristics(
                                                                            gBle.m_bleState.connectionInfo.Serv[i].Char[j].data,
                                                                            16);

                NN_LOG("[BLE] >> GattService[%d][%d] = 0x%04x    Characteristic Count = %d\n",
                        i,
                        j,
                        gBle.m_bleState.connectionInfo.Serv[i].data[j].GetUuid().uu.uuid16,
                        gBle.m_bleState.connectionInfo.Serv[i].Char[j].dataNum);
            }

            nn::bluetooth::GattAttributeUuid fwVerUuid;
            fwVerUuid.length = nn::bluetooth::GattAttributeUuidLength_16;
            fwVerUuid.uu.uuid16 = 0x2A28;   //!< Software Revision String

            for (int j = 0; j < gBle.m_bleState.connectionInfo.Serv[i].dataNum; ++j)
            {
                for (auto k = 0; k < gBle.m_bleState.connectionInfo.Serv[i].Char[j].dataNum; k++)
                {
                    if (gBle.m_bleState.connectionInfo.Serv[i].Char[j].data[k].GetUuid() == fwVerUuid)
                    {
                        nn::bluetooth::ReadGattCharacteristic(gBle.m_bleState.connectionInfo.Serv[i].Char[j].data[k]);
                    }
                }
            }
        }
    }

    void GattOperationHandler()
    {
        nn::os::ClearSystemEvent(&event.gattOperation);

        // GATT Operation に関しては、即結果を取得する
        nn::bluetooth::BleClientGattOperationInfo info;
        nn::bluetooth::GetGattOperationResult(&info);

        if (info.status == nn::bluetooth::BleGattOperationStatus_Success)
        {
            nn::bluetooth::GattAttributeUuid fwVerUuid;
            fwVerUuid.length = nn::bluetooth::GattAttributeUuidLength_16;
            fwVerUuid.uu.uuid16 = 0x2A28;   //!< Software Revision String

            if (info.operation == nn::bluetooth::GattOperationType_ReadCharacteristic && info.charcteristicUuid == fwVerUuid)
            {
                auto index = info.connectionHandle;
                for (int i = 0; i < gBle.m_bleState.connectionInfo.dataNum; ++i)
                {
                    if (index == gBle.m_bleState.connectionInfo.data[i].connectionHandle)
                    {
                        gBle.m_bleState.connectionInfo.Serv[i].fwVerLength = info.length;
                        if (info.length <= 32)
                        {
                            memcpy(&gBle.m_bleState.connectionInfo.Serv[i].fwVer, info.data, gBle.m_bleState.connectionInfo.Serv[i].fwVerLength);
                        }
                        else
                        {
                            NN_LOG("[BLE] >> FwVersion Length Over 32 Bytes\n");
                        }
                    }
                }
            }
        }
    }

    void MultiWaitThreadFunction(void* arg)
    {
        NN_UNUSED(arg);

        for (;;)
        {
            nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&multiWaitHolder.header);
            auto calllback = reinterpret_cast<Callback>(nn::os::GetMultiWaitHolderUserData(holder));

            if (calllback)
            {
                // コールバックの呼び出し
                calllback();
            }
        }
    }

    /* ------------------------------------------------------------ */
    // PUBLIC関数
    /* ------------------------------------------------------------ */
    Ble& Ble::GetInstance()
    {
        static Ble Instance;
        return Instance;
    }

    void Ble::Initialize()
    {
        nn::bluetooth::InitializeBle();

        // BLE 関連のシステムイベントの取得
        nn::bluetooth::AcquireBleScanEvent(&event.scan);
        nn::bluetooth::AcquireBleConnectionStateChangedEvent(&event.connection);
        nn::bluetooth::AcquireBleServiceDiscoveryEvent(&event.serviceDiscovery);
        nn::bluetooth::AcquireBleGattOperationEvent(&event.gattOperation);

        // 多重待ちオブジェクトの初期化とリンク
        nn::os::InitializeMultiWaitHolder(&multiWaitHolder.scan, &event.scan);
        nn::os::SetMultiWaitHolderUserData(&multiWaitHolder.scan, reinterpret_cast<uintptr_t>(ScanHandler));
        nn::os::InitializeMultiWaitHolder(&multiWaitHolder.connection, &event.connection);
        nn::os::SetMultiWaitHolderUserData(&multiWaitHolder.connection, reinterpret_cast<uintptr_t>(ConnectionHandler));
        nn::os::InitializeMultiWaitHolder(&multiWaitHolder.serviceDiscovery, &event.serviceDiscovery);
        nn::os::SetMultiWaitHolderUserData(&multiWaitHolder.serviceDiscovery, reinterpret_cast<uintptr_t>(ServiceDiscoveryHandler));
        nn::os::InitializeMultiWaitHolder(&multiWaitHolder.gattOperation, &event.gattOperation);
        nn::os::SetMultiWaitHolderUserData(&multiWaitHolder.gattOperation, reinterpret_cast<uintptr_t>(GattOperationHandler));

        nn::os::InitializeMultiWait(&multiWaitHolder.header);
        nn::os::LinkMultiWaitHolder(&multiWaitHolder.header, &multiWaitHolder.scan);
        nn::os::LinkMultiWaitHolder(&multiWaitHolder.header, &multiWaitHolder.connection);
        nn::os::LinkMultiWaitHolder(&multiWaitHolder.header, &multiWaitHolder.serviceDiscovery);
        nn::os::LinkMultiWaitHolder(&multiWaitHolder.header, &multiWaitHolder.gattOperation);

        // 多重待ちスレッドを生成、開始
        nn::os::CreateThread(&multiWaitHolder.thread, MultiWaitThreadFunction, nullptr, multiWaitHolder.threadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
        nn::os::StartThread(&multiWaitHolder.thread);

        nn::bluetooth::GetBleScanParameter(&gBle.m_bleState.palmaAdvertisePacketParameter, nn::bluetooth::BleScanParameterId_Palma);
    }

    void Ble::RegisterGattOperation(nn::bluetooth::GattService* service)
    {
        // GattOperationの登録既に登録されているかを走査する
        bool isAlreadyRegistered = false;
        for (std::vector<nn::bluetooth::GattAttributeUuid>::iterator itr = registeredUuid.begin(); itr != registeredUuid.end(); ++itr)
        {
            if ((*itr).uu.uuid16 == service->GetUuid().uu.uuid16)
            {
                isAlreadyRegistered = true;
            }
        }
        if (!isAlreadyRegistered)
        {
            int timeoutCount = 0;
            // 不正な終了が入った時に登録状態が残存するので、登録予定のサービスを一旦クリアする
            while (true)
            {
                if (nn::bluetooth::UnregisterGattOperationNotification(service->GetUuid()).IsSuccess())
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                    break;
                }
                else
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                    timeoutCount++;
                    if (timeoutCount >= 2)
                    {
                        break;
                    }
                }
            }
            timeoutCount = 0;
            // 再登録
            while (true)
            {
                if (nn::bluetooth::RegisterGattOperationNotification(service->GetUuid()).IsSuccess())
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
                    break;
                }
                else
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
                    timeoutCount++;
                    if (timeoutCount >= 2)
                    {
                        break;
                    }
                }
            }
            registeredUuid.push_back(service->GetUuid());
        }
    }

    void Ble::StartBleScanSmartDevice()
    {
        auto result = nn::bluetooth::StartBleScanSmartDevice(util::ADV_SERVICE_UUID);
        if (result.IsFailure())
        {
            NN_LOG("[BLE] >> Failed to start BLE Scan Smart Device\n");
        }
        else
        {
            m_IsScanningSmartDevice = true;
        }
    }

    void Ble::StopBleScanSmartDevice()
    {
        auto result = nn::bluetooth::StopBleScanSmartDevice();
        if (result.IsFailure())
        {
            NN_LOG("[BLE] >> Failed to stop BLE Scan Smart Device\n");
        }
        m_IsScanningSmartDevice = false;
    }

    void Ble::finalize()
    {
        for (auto i = 0; i < gBle.m_bleState.connectionInfo.dataNum; i++)
        {
            nn::bluetooth::DisconnectFromGattServer(gBle.m_bleState.connectionInfo.data[i].connectionHandle);
        }
        // スレッドの終了を待つ
        nn::os::WaitThread(&multiWaitHolder.thread);
        nn::os::DestroyThread(&multiWaitHolder.thread);

        // 多重待ちオブジェクトのアンリンクと破棄
        nn::os::UnlinkAllMultiWaitHolder(&multiWaitHolder.header);
        nn::os::FinalizeMultiWait(&multiWaitHolder.header);

        // システムイベントの破棄
        nn::os::DestroySystemEvent(&event.scan);
        nn::os::DestroySystemEvent(&event.connection);
        nn::os::DestroySystemEvent(&event.serviceDiscovery);
        nn::os::DestroySystemEvent(&event.gattOperation);
    }

    // Debug用
    void Ble::ResetPairing()
    {
        for (auto i = 0; i < gBle.m_bleState.connectionInfo.dataNum; i++)
        {
            nn::bluetooth::UnpairGattServer(gBle.m_bleState.connectionInfo.data[i].connectionHandle, gBle.m_bleState.palmaAdvertisePacketParameter);
        }
    }
}
