﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <mutex>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/image/image_JpegDecoder.h>

#include "../Simple2D.h"
#include "Simple2D_Util.h"

namespace s2d { namespace detail {

namespace
{

const size_t BitmapFileHeaderSize = 14;

// Bitmap ファイルヘッダ
struct BitmapFileHeader
{
    uint16_t type;
    uint32_t size;
    uint16_t reserved1;
    uint16_t reserved2;
    uint32_t offBits;
};

// Bitmap 情報ヘッダ
struct BitmapInfoHeader
{
    uint32_t  size;
    int32_t   width;
    int32_t   height;
    uint16_t  planes;
    uint16_t  bitCount;
    uint32_t  compression;
    uint32_t  sizeImage;
    int32_t   xPelsPerMeter;
    int32_t   yPelsPerMeter;
    uint32_t  clrUsed;
    uint32_t  clrImportant;
};

// カラーテーブルの要素
struct RgbQuad
{
    uint8_t blue;
    uint8_t green;
    uint8_t red;
    uint8_t reserved;
};

// Bitmap 情報
struct BitmapInfo
{
    BitmapInfoHeader header;
    RgbQuad          colors[1];
};

// ファイルの中身を一括で読み込む
nn::Result ReadFile(void** ppOutBuffer, size_t* pOutSize, const char* filename) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(ppOutBuffer);
    NN_ASSERT_NOT_NULL(pOutSize);
    NN_ASSERT_NOT_NULL(filename);

    nn::fs::FileHandle handle;
    NN_RESULT_DO(nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t fileSize;
    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

    auto* readBuffer = new char[fileSize];
    NN_ABORT_UNLESS_NOT_NULL(readBuffer);

    bool needsDelete = true;
    NN_UTIL_SCOPE_EXIT
    {
        if (needsDelete)
        {
            delete[] readBuffer;
        }
    };

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, readBuffer, static_cast<size_t>(fileSize)));

    *ppOutBuffer = readBuffer;
    *pOutSize = static_cast<size_t>(fileSize);
    needsDelete = false;

    NN_RESULT_SUCCESS;
}

// Bitmap のデコード
ResultCode DecodeBitmap(
    Size* pImageSize,
    void** ppOutBuffer,
    const char* pBitmapData,
    size_t bitmapSize) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pImageSize);
    NN_ASSERT_NOT_NULL(ppOutBuffer);
    NN_ASSERT_NOT_NULL(pBitmapData);
    NN_ASSERT_GREATER(bitmapSize, 0);

    BitmapFileHeader fileHeader;
    std::memcpy(&fileHeader.type, pBitmapData, sizeof(fileHeader.type));
    std::memcpy(&fileHeader.size, pBitmapData + 2, sizeof(fileHeader.size));
    std::memcpy(&fileHeader.reserved1, pBitmapData + 6, sizeof(fileHeader.reserved1));
    std::memcpy(&fileHeader.reserved2, pBitmapData + 8, sizeof(fileHeader.reserved2));
    std::memcpy(&fileHeader.offBits, pBitmapData + 10, sizeof(fileHeader.offBits));

    BitmapInfoHeader infoHeader;
    std::memcpy(&infoHeader, pBitmapData + BitmapFileHeaderSize, sizeof(infoHeader));

    // Bitmap 以外は弾く
    if (std::memcmp(&fileHeader.type, "BM", 2) != 0)
    {
        NN_LOG("fileHeader.type = 0x%04X\n", fileHeader.type);
        return ResultCode::DecodeFailed;
    }

    // とりあえず 24bit/32bit 以外は弾く
    if (infoHeader.bitCount != 24 &&
        infoHeader.bitCount != 32)
    {
        NN_LOG("infoHeader.bitCount = 0x%02X\n", infoHeader.bitCount);
        return ResultCode::DecodeFailed;
    }

    // 圧縮形式は弾く
    if (infoHeader.compression != 0)
    {
        NN_LOG("infoHeader.compression = 0x%02X\n", infoHeader.compression);
        return ResultCode::DecodeFailed;
    }

    // ピクセルデータのコピー
    size_t pixelDataSize = infoHeader.width * infoHeader.height * 4;
    auto* pixelData = new uint8_t[pixelDataSize];
    auto* pSrcData  = pBitmapData + fileHeader.offBits;

    int pixelSize = infoHeader.bitCount / 8;
    int dstPitch  = infoHeader.width * 4;
    int srcPitch  = infoHeader.width * pixelSize;
    if (srcPitch % 4 != 0)
    {
        // 24bpp の場合はパディングを考慮
        srcPitch = (srcPitch / 4 + 1) * 4;
    }

    for (int i = 0; i < infoHeader.height; i++)
    {
        for (int j = 0; j < infoHeader.width; j++)
        {
            // BGRA -> RGBA
            int src = i * srcPitch + j * pixelSize;
            int dst = (infoHeader.height - i - 1) * dstPitch + j * 4;
            pixelData[dst]     = pSrcData[src + 2];
            pixelData[dst + 1] = pSrcData[src + 1];
            pixelData[dst + 2] = pSrcData[src];
            pixelData[dst + 3] = (pixelSize == 4 ? pSrcData[src + 3] : 255);
        }
    }

    *ppOutBuffer = pixelData;
    pImageSize->width  = static_cast<decltype(Size::width)>(infoHeader.width);
    pImageSize->height = static_cast<decltype(Size::height)>(infoHeader.height);

    return ResultCode::Ok;
}

// JPEG のデコード
ResultCode DecodeJpeg(
    Size* pImageSize,
    void** ppOutBuffer,
    const char* pJpegData,
    size_t jpegSize) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pImageSize);
    NN_ASSERT_NOT_NULL(ppOutBuffer);
    NN_ASSERT_NOT_NULL(pJpegData);
    NN_ASSERT_GREATER(jpegSize, 0);

    nn::image::JpegDecoder decoder;
    decoder.SetImageData(pJpegData, jpegSize);

    {
        auto status = decoder.Analyze();
        if (status != nn::image::JpegStatus_Ok)
        {
            return ResultCode::DecodeFailed;
        }
    }

    auto workSize = decoder.GetAnalyzedWorkBufferSize();
    auto* workBuffer = new char[workSize];
    NN_UTIL_SCOPE_EXIT
    {
        delete[] workBuffer;
    };

    const int alignment = 1;
    auto dimension = decoder.GetAnalyzedDimension();
    auto stride    = 4 * alignment * ((dimension.width + alignment - 1) / alignment);
    auto dataSize  = stride * dimension.height;
    auto* decodeBuffer = new char[dataSize];

    {
        auto status = decoder.Decode(decodeBuffer, dataSize, alignment, workBuffer, workSize);
        if (status != nn::image::JpegStatus_Ok)
        {
            delete[] decodeBuffer;
            return ResultCode::DecodeFailed;
        }
    }

    *ppOutBuffer = decodeBuffer;
    pImageSize->width  = static_cast<decltype(Size::width)>(dimension.width);
    pImageSize->height = static_cast<decltype(Size::height)>(dimension.height);

    return ResultCode::Ok;
}

}  // anonymous

int RoundUpPow2(int value) NN_NOEXCEPT
{
    int dest = 1;
    while (value > dest)
    {
        dest *= 2;
    }

    return dest;
}

ResultCode LoadBitmap(Size* pImageSize, void** ppOutBuffer, const char* filename) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pImageSize);
    NN_ASSERT_NOT_NULL(ppOutBuffer);
    NN_ASSERT_NOT_NULL(filename);

    char* pReadBuffer;
    size_t fileSize;
    if (ReadFile(reinterpret_cast<void**>(&pReadBuffer), &fileSize, filename).IsFailure())
    {
        return ResultCode::ReadFailed;
    }

    NN_UTIL_SCOPE_EXIT
    {
        delete[] pReadBuffer;
    };

    return DecodeBitmap(pImageSize, ppOutBuffer, pReadBuffer, fileSize);
}

ResultCode LoadJpeg(Size* pImageSize, void** ppOutBuffer, const char* filename) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pImageSize);
    NN_ASSERT_NOT_NULL(ppOutBuffer);
    NN_ASSERT_NOT_NULL(filename);

    char* pReadBuffer;
    size_t fileSize;
    if (ReadFile(reinterpret_cast<void**>(&pReadBuffer), &fileSize, filename).IsFailure())
    {
        return ResultCode::ReadFailed;
    }

    NN_UTIL_SCOPE_EXIT
    {
        delete[] pReadBuffer;
    };

    return DecodeJpeg(pImageSize, ppOutBuffer, pReadBuffer, fileSize);
}

}}  // s2d::detail
