﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Npad.h>

#include "MenuItem.h"
#include "MenuTop.h"
#include "MenuAssignment.h"
#include "MenuDeviceControl.h"
#include "MenuAdditionalSettings.h"
#include "MenuControllerSupport.h"
#include "NpadStyleDefinition.h"

// スタイル毎の有効/無効設定
class MenuItemStyleEnabled : public MenuItemBoolean
{
private:
    const nn::hid::NpadStyleSet m_Style;

public:
    MenuItemStyleEnabled(const std::string& name, const nn::hid::NpadStyleSet& style) NN_NOEXCEPT :
        MenuItemBoolean(name),
        m_Style(style)
    {
        // 何もしない
    }

    nn::hid::NpadStyleSet ReturnStyleIfEnabled() NN_NOEXCEPT
    {
        if (this->IsEnabled())
        {
            return m_Style;
        }
        else
        {
            nn::hid::NpadStyleSet style;
            return style.Reset();
        }
    }

    void UpdateValueWithCurrentStyle(const nn::hid::NpadStyleSet& supportedStyles) NN_NOEXCEPT
    {
        m_Value = ((supportedStyles & m_Style).IsAnyOn()) ? 0 : 1;
    }
};

// Npad で有効なスタイルに関する設定
class MenuItemNpadSupportedStyles final : public MenuItemHolder
{
public:
    MenuItemNpadSupportedStyles() NN_NOEXCEPT :
        MenuItemHolder("Supported NpadStyles", true)
    {
        for (auto& style : NpadStyles)
        {
            AddMenuItem(new MenuItemStyleEnabled(style.styleString, style.style));
        }
    }
protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::NpadStyleSet supportedStyles;
        supportedStyles.Reset();

        for (std::vector<MenuItem*>::iterator it = m_pMenuItems.begin();
             it != m_pMenuItems.end();
             ++it)
        {
            supportedStyles |= reinterpret_cast<MenuItemStyleEnabled*>(*it)->ReturnStyleIfEnabled();
        }
        nn::hid::SetSupportedNpadStyleSet(supportedStyles);
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::NpadStyleSet supportedStyles = nn::hid::GetSupportedNpadStyleSet();
        for (std::vector<MenuItem*>::iterator it = m_pMenuItems.begin();
             it != m_pMenuItems.end();
             ++it)
        {
            reinterpret_cast<MenuItemStyleEnabled*>(*it)->UpdateValueWithCurrentStyle(supportedStyles);
        }
    }
};

class IMenuItemSupportedNpadId : public MenuItem
{
public:
    explicit IMenuItemSupportedNpadId(const std::string& name) NN_NOEXCEPT :
        MenuItem(name, true)
    {
        // 何もしない
    };

    virtual int GetSupportedNpadIds(nn::hid::NpadIdType* pIds, int size) NN_NOEXCEPT = 0;
};

// NpadId:Handheld の有効無効
class MenuItemNpadIdHandheld : public IMenuItemSupportedNpadId
{
private:
    bool m_IsHandheldEnabled;

public:
    MenuItemNpadIdHandheld() NN_NOEXCEPT :
        IMenuItemSupportedNpadId("Handheld")
    {
        this->AddSelectable("\uE14B");
        this->AddSelectable("\uE14C");
        // デフォルトは Enabled
        m_IsHandheldEnabled = true;
    }

    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        if (this->IsEnabled())
        {
            nn::hid::system::EnableAssigningSingleOnSlSrPress();
        }
        else
        {
            nn::hid::system::DisableAssigningSingleOnSlSrPress();
        }
    }

    virtual int GetSupportedNpadIds(nn::hid::NpadIdType* pIds, int size) NN_NOEXCEPT NN_OVERRIDE
    {
        m_IsHandheldEnabled = this->IsEnabled();
        if (this->IsEnabled())
        {
            pIds[0] = nn::hid::NpadId::Handheld;
            return 1;
        }
        return 0;
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        m_Value = m_IsHandheldEnabled ? 0 : 1;
    }
private:
    bool IsEnabled() NN_NOEXCEPT
    {
        return (m_Value == 0);
    }

};

// 有効にする NpadId の組み合わせの定義
class MenuItemNpadIdPattern final : public IMenuItemSupportedNpadId
{
private:
    struct NpadIdString
    {
        std::vector<nn::hid::NpadIdType> ids;
        std::string string;
    };
    std::vector<NpadIdString> m_IdPattern;
    int m_CurrentValue;

public:
    MenuItemNpadIdPattern() NN_NOEXCEPT :
        IMenuItemSupportedNpadId("No*")
    {
        m_IdPattern.push_back({ {},
                                "None" });
        m_IdPattern.push_back({ { nn::hid::NpadId::No1 },
                                "No1" });
        m_IdPattern.push_back({ { nn::hid::NpadId::No1,
                                  nn::hid::NpadId::No2 },
                                "No1-2" });
        m_IdPattern.push_back({ { nn::hid::NpadId::No1,
                                  nn::hid::NpadId::No2,
                                  nn::hid::NpadId::No3, },
                                "No1-3" });
        m_IdPattern.push_back({ { nn::hid::NpadId::No1,
                                  nn::hid::NpadId::No2,
                                  nn::hid::NpadId::No3,
                                  nn::hid::NpadId::No4, },
                                "No1-4" });
        m_IdPattern.push_back({ { nn::hid::NpadId::No1,
                                  nn::hid::NpadId::No2,
                                  nn::hid::NpadId::No3,
                                  nn::hid::NpadId::No4,
                                  nn::hid::NpadId::No5, },
                                "No1-5" });
        m_IdPattern.push_back({ { nn::hid::NpadId::No1,
                                  nn::hid::NpadId::No2,
                                  nn::hid::NpadId::No3,
                                  nn::hid::NpadId::No4,
                                  nn::hid::NpadId::No5,
                                  nn::hid::NpadId::No6, },
                                "No1-6" });
        m_IdPattern.push_back({ { nn::hid::NpadId::No1,
                                  nn::hid::NpadId::No2,
                                  nn::hid::NpadId::No3,
                                  nn::hid::NpadId::No4,
                                  nn::hid::NpadId::No5,
                                  nn::hid::NpadId::No6,
                                  nn::hid::NpadId::No7, },
                                "No1-7" });
        m_IdPattern.push_back({ { nn::hid::NpadId::No1,
                                  nn::hid::NpadId::No2,
                                  nn::hid::NpadId::No3,
                                  nn::hid::NpadId::No4,
                                  nn::hid::NpadId::No5,
                                  nn::hid::NpadId::No6,
                                  nn::hid::NpadId::No7,
                                  nn::hid::NpadId::No8, },
                                "No1-8" });
        for (std::vector<NpadIdString>::iterator it = m_IdPattern.begin();
             it != m_IdPattern.end();
             ++it)
        {
            AddSelectable((it)->string);
        }
        m_CurrentValue = 8;
    }

    virtual int GetSupportedNpadIds(nn::hid::NpadIdType* pIds, int size) NN_NOEXCEPT NN_OVERRIDE
    {
        auto idString = m_IdPattern.at(m_Value);
        for (int i = 0; i < idString.ids.size() && i < size; ++i)
        {
            pIds[i] = idString.ids.at(i);
        }
        m_CurrentValue = m_Value;
        return std::min(static_cast<int>(idString.ids.size()), size);
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        m_Value = m_CurrentValue;
    }
};

// Npad の有効な Id に関する設定
class MenuItemNpadSupportedIds final : public MenuItemHolder
{
public:
    MenuItemNpadSupportedIds() NN_NOEXCEPT :
        MenuItemHolder("Supported NpadIds", true)
    {
        AddMenuItem(new MenuItemNpadIdHandheld());    // Handheld
        AddMenuItem(new MenuItemNpadIdPattern());     // No1-8
    }
protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::NpadIdType supportedIds[NpadIdCountMax];
        int count = 0;
        for (std::vector<MenuItem*>::iterator it = m_pMenuItems.begin();
             it != m_pMenuItems.end();
             ++it)
        {
            count += reinterpret_cast<IMenuItemSupportedNpadId*>(*it)->GetSupportedNpadIds(&supportedIds[count],
                NN_ARRAY_SIZE(supportedIds) - count);
        }
        nn::hid::SetSupportedNpadIdType(supportedIds, count);
    }
};

// Joy-Con の持ち方の設定
class MenuItemHoldType final : public MenuItem
{
public:
    MenuItemHoldType() NN_NOEXCEPT :
        MenuItem("HoldType", true)
    {
        AddSelectable("Vertical");
        AddSelectable("Horizontal");
    }

    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_Value == 0)
        {
            nn::hid::SetNpadJoyHoldType(nn::hid::NpadJoyHoldType_Vertical);
        }
        else
        {
            nn::hid::SetNpadJoyHoldType(nn::hid::NpadJoyHoldType_Horizontal);
        }
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        switch (nn::hid::GetNpadJoyHoldType())
        {
        case nn::hid::NpadJoyHoldType_Vertical:
            m_Value = 0;
            break;
        case nn::hid::NpadJoyHoldType_Horizontal:
            m_Value = 1;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
};

// SlSr 割り当てモード
class MenuItemSingleOnSlSr : public MenuItemBoolean
{
private:
    bool m_Enabled;

public:
    MenuItemSingleOnSlSr() NN_NOEXCEPT :
        MenuItemBoolean("Single On SlSr"),
        m_Enabled(false) // デフォルト無効
    {
        // 何もしない
    }

    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        m_Enabled = this->IsEnabled();
        if (m_Enabled)
        {
            nn::hid::system::EnableAssigningSingleOnSlSrPress();
        }
        else
        {
            nn::hid::system::DisableAssigningSingleOnSlSrPress();
        }
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        this->UpdateValue(m_Enabled);
    }
};

// LR 割り当てモード
class MenuItemLrMode : public MenuItemBoolean
{
private:
    bool m_Enabled;

public:
    MenuItemLrMode() NN_NOEXCEPT :
        MenuItemBoolean("LR Mode"),
        m_Enabled(false) // デフォルト無効
    {
        // 何もしない
    }

    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        m_Enabled = this->IsEnabled();
        if (m_Enabled)
        {
            nn::hid::StartLrAssignmentMode();
        }
        else
        {
            nn::hid::StopLrAssignmentMode();
        }
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        this->UpdateValue(m_Enabled);
    }
};

// システム共通設定
class MenuItemSystemCommonPolicy final : public MenuItem
{
private:
    // 確定した値
    int m_FixedValue;

public:
    MenuItemSystemCommonPolicy() NN_NOEXCEPT :
        MenuItem("SystemCommon", true),
        m_FixedValue(0)
    {
        AddSelectable("Disabled");
        AddSelectable("Normal");
        AddSelectable("Full");
    }

    virtual void NextValue(bool recursive) NN_NOEXCEPT NN_OVERRIDE
    {
        // 一度セットされたら値は変更できない
        if (m_FixedValue != 0)
        {
            return;
        }

        MenuItem::NextValue(recursive);
    }

    virtual void PrevValue(bool recursive) NN_NOEXCEPT NN_OVERRIDE
    {
        // 一度セットされたら値は変更できない
        if (m_FixedValue != 0)
        {
            return;
        }

        MenuItem::PrevValue(recursive);
    }

    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_Value == 1)
        {
            nn::hid::system::ApplyNpadSystemCommonPolicy();
        }
        else if(m_Value == 2)
        {
            nn::hid::system::ApplyNpadSystemCommonPolicyFull();
        }
        m_FixedValue = m_Value;
    }
};

// 携帯機操作のモード
class MenuItemHandheldMode final : public MenuItem
{
public:
    MenuItemHandheldMode() NN_NOEXCEPT :
        MenuItem("Handheld Mode", true)
    {
        AddSelectable("Dual");
        AddSelectable("Single");
        AddSelectable("None");
    }

    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        switch(m_Value)
        {
        case 0: // Dual
            nn::hid::SetNpadHandheldActivationMode(nn::hid::NpadHandheldActivationMode_Dual);
            break;
        case 1: // Single
            nn::hid::SetNpadHandheldActivationMode(nn::hid::NpadHandheldActivationMode_Single);
            break;
        case 2: // None
            nn::hid::SetNpadHandheldActivationMode(nn::hid::NpadHandheldActivationMode_None);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        switch (nn::hid::GetNpadHandheldActivationMode())
        {
        case nn::hid::NpadHandheldActivationMode_Dual:
            m_Value = 0;
            break;
        case nn::hid::NpadHandheldActivationMode_Single:
            m_Value = 1;
            break;
        case nn::hid::NpadHandheldActivationMode_None:
            m_Value = 2;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
};

// Npad のその他の設定
class MenuItemNpadOthers final : public MenuItemHolder
{
public:
    MenuItemNpadOthers() NN_NOEXCEPT :
        MenuItemHolder("Other Npad Settings", true)
    {
        AddMenuItem(new MenuItemHoldType());           // 持ち方設定
        AddMenuItem(new MenuItemSingleOnSlSr());       // SlSr 接続割り当てモード設定
        AddMenuItem(new MenuItemLrMode());             // LR 割り当てモード設定
        AddMenuItem(new MenuItemSystemCommonPolicy()); // 携帯コントローラーの動作モード
        AddMenuItem(new MenuItemHandheldMode());       // 携帯コントローラーの動作モード
    }
protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        for (std::vector<MenuItem*>::iterator it = m_pMenuItems.begin();
            it != m_pMenuItems.end();
            ++it)
        {
            (*it)->Proceed();
        }
    }
};

// TestTool のトップメニュー
MenuItemTestToolTop::MenuItemTestToolTop() NN_NOEXCEPT :
    MenuItemHolder("", false)
{
    AddMenuItem(new MenuItemNpadSupportedStyles()); // NpadStyle の有効設定
    AddMenuItem(new MenuItemNpadSupportedIds());    // NpadId の有効設定
    AddMenuItem(new MenuItemNpadOthers());          // Npad のその他の設定
    AddMenuItem(new MenuItemAssignmentTop());    // コントローラーの割り当て設定
    AddMenuItem(new MenuItemDeviceControl());    // デバイス設定
    AddMenuItem(new MenuItemControllerSupport());    // コンサポ呼び出し
    AddMenuItem(new MenuItemAdditionalSettings());    // 追加の設定
}

