﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <vector>
#include <nn/util/util_MathTypes.h>

#include "NpadIdDefine.h"
#include "ScreenStatics.h"
#include "s2d/Simple2D.h"

class MenuItem
{
    NN_DISALLOW_COPY(MenuItem);
    NN_DISALLOW_MOVE(MenuItem);

public:
    const bool IsFunctional;
    const std::string Name;

protected:
    int m_Value;
    std::vector<std::string> m_Selectables;

private:
    bool m_IsSelected;

public:
    explicit MenuItem(const std::string& name, const bool& isFunctional) NN_NOEXCEPT;
    virtual ~MenuItem() NN_NOEXCEPT { /* 何もしない */ };

    virtual void Print(s2d::Point2D & position, int depth) NN_NOEXCEPT;
    virtual bool IsChildMenuOpened() NN_NOEXCEPT { return false; };

    virtual void NextValue(bool recursive) NN_NOEXCEPT;
    virtual void PrevValue(bool recursive) NN_NOEXCEPT;

    virtual void NextItem() NN_NOEXCEPT { /* 何もしない */ };
    virtual void PrevItem() NN_NOEXCEPT{ /* 何もしない */ };
    virtual bool Select() NN_NOEXCEPT { return true; };
    virtual bool Cancel() NN_NOEXCEPT { return true; };
    virtual void ClearSelected() NN_NOEXCEPT { m_IsSelected = false; };
    virtual void Update() NN_NOEXCEPT { /* 何もしない */ };

    virtual void GetItemIndex(int* pOutIndex, int* pOutDepth) NN_NOEXCEPT
    {
        *pOutIndex = 0;
        *pOutDepth = 0;
    };
    virtual int GetSize() NN_NOEXCEPT { return 1; };

    //!< Menu 毎の実行関数
    virtual void Proceed() NN_NOEXCEPT { /* 何もしない */ };

protected:
    void AddSelectable(std::string selectable) NN_NOEXCEPT;
};

class MenuItemHolder : public MenuItem
{
    NN_DISALLOW_COPY(MenuItemHolder);
    NN_DISALLOW_MOVE(MenuItemHolder);

private:
    const bool ExpandInNewMenu;
    const int SelectionCountMax;

protected:
    std::vector<MenuItem*> m_pMenuItems;
    int m_ItemIndex;
    int m_FunctionalItemCount;
    int m_SelectedCount;
    int m_ValuesSelected[NpadIdCountMax];

    enum Focus
    {
        Focus_Out,
        Focus_On,
        Focus_Child,
    };
    Focus m_Focus;

public:
    explicit MenuItemHolder(const std::string& name, const bool& isFunctional) NN_NOEXCEPT;
    explicit MenuItemHolder(const std::string& name, const bool& isFunctional, const bool& expandInNewMenu) NN_NOEXCEPT;
    explicit MenuItemHolder(const std::string& name, const bool& isFunctional, const bool& expandInNewMenu, const int& selectionCountMax) NN_NOEXCEPT;
    virtual ~MenuItemHolder() NN_NOEXCEPT NN_OVERRIDE { /* 何もしない*/ };

    virtual void Print(s2d::Point2D & position, int depth) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool IsChildMenuOpened() NN_NOEXCEPT NN_OVERRIDE;

    virtual void NextValue(bool recursive) NN_NOEXCEPT NN_OVERRIDE;
    virtual void PrevValue(bool recursive) NN_NOEXCEPT NN_OVERRIDE;
    virtual void NextItem() NN_NOEXCEPT NN_OVERRIDE;
    virtual void PrevItem() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool Select() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool Cancel() NN_NOEXCEPT NN_OVERRIDE;
    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    virtual void GetItemIndex(int* pOutIndex, int* pOutDepth) NN_NOEXCEPT NN_OVERRIDE;
    virtual int GetSize() NN_NOEXCEPT NN_OVERRIDE;
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE { /* 何もしない */ };

protected:
    void AddMenuItem(MenuItem* pMenuItem) NN_NOEXCEPT;

private:
    bool IsValueSelected(int value) NN_NOEXCEPT;
};

class MenuItemBoolean : public MenuItem
{
public:
    explicit MenuItemBoolean(const std::string& name) NN_NOEXCEPT;
    virtual ~MenuItemBoolean() NN_NOEXCEPT { /* 何もしない */ };

protected:
    bool IsEnabled() NN_NOEXCEPT;
    void UpdateValue(bool enabled) NN_NOEXCEPT;
};

// NpadId を選択するためのメニュー
class MenuItemNpadIdsSelectable : public MenuItemHolder
{
private:
    const bool IsHandheldSupported;

public:
    explicit MenuItemNpadIdsSelectable(const std::string& name, const bool& isHandheldSupported, const int& selectionCountMax) NN_NOEXCEPT;
    virtual ~MenuItemNpadIdsSelectable() NN_NOEXCEPT { /* 何もしない */ };

protected:
    int GetValue(nn::hid::NpadIdType* pOutIds, int size) NN_NOEXCEPT;

private:
    nn::hid::NpadIdType GetNpadIdTypeFromValue(int value) NN_NOEXCEPT;
};

// NpadId を選んで次に進むためのメニュー
template<typename TMenuItem>
class MenuItemNpadIds : public MenuItemHolder
{
public:
    explicit MenuItemNpadIds(const std::string& name, const bool& isHandheldSupported) NN_NOEXCEPT :
        MenuItemHolder(name, true, true, 0)
    {
        AddMenuItem(new TMenuItem("NpadId::No1", nn::hid::NpadId::No1));
        AddMenuItem(new TMenuItem("NpadId::No2", nn::hid::NpadId::No2));
        AddMenuItem(new TMenuItem("NpadId::No3", nn::hid::NpadId::No3));
        AddMenuItem(new TMenuItem("NpadId::No4", nn::hid::NpadId::No4));
        AddMenuItem(new TMenuItem("NpadId::No5", nn::hid::NpadId::No5));
        AddMenuItem(new TMenuItem("NpadId::No6", nn::hid::NpadId::No6));
        AddMenuItem(new TMenuItem("NpadId::No7", nn::hid::NpadId::No7));
        AddMenuItem(new TMenuItem("NpadId::No8", nn::hid::NpadId::No8));
        if (isHandheldSupported)
        {
            AddMenuItem(new TMenuItem("NpadId::Handheld", nn::hid::NpadId::Handheld));
        }
    }
};

// Enable/Disable を選択するシンプルなメニュー
class MenuItemEnableDisable : public MenuItemHolder
{
public:
    explicit MenuItemEnableDisable(const std::string& name) NN_NOEXCEPT :
        MenuItemHolder(name, true, true)
    {
        AddMenuItem(new MenuItem("Enable", true));
        AddMenuItem(new MenuItem("Disable", true));
    }

protected:
    bool IsEnableSelected() NN_NOEXCEPT
    {
        return m_ItemIndex == 0;
    }
};
