﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
* @examplesource{HidGesture_Main.cpp,PageSampleHidGesture}
*
* @brief
*  nn::hid ライブラリのジェスチャ API の使い方を示すサンプルプログラム
*/

/**
* @page PageSampleHidGesture ジェスチャの取得
* @tableofcontents
*
* @brief
* nn::hid ライブラリを用いてジェスチャを取得するサンプルプログラムの解説です。
*
* @section PageSampleHidGesture_SectionBrief 概要
* ジェスチャの状態を表示します。
*
* @section PageSampleHidGesture_SectionFileStructure ファイル構成
* 本サンプルプログラムは
* @link ../../../Samples/Sources/Applications/HidGesture
* Samples/Sources/Applications/HidGesture @endlink 以下にあります。
*
* @section PageSampleHidGesture_SectionNecessaryEnvironment 必要な環境
* Windows 環境においては、タッチまたはマウス操作が入力として扱われます。
*
* @section PageSampleHidGesture_SectionHowToOperate 操作方法
* サンプルプログラムを実行するとウィンドウが立ち上がります。
* ウインドウ上には、左半分に矩形状に数字が、右半分には操作ログが表示され、
* 矩形にはピンチや回転といった連続的な操作の結果が、
* 操作ログにはタップやスワイプと言った単発的な操作の結果が反映されます。
* 操作ログからはコンテキスト番号を確認することもできます。
* また、タッチ位置にはカーソルが描画されます。
*
* @section PageSampleHidGesture_SectionPrecaution 注意事項
* コンソールウィンドウには何も表示されません。
*
* @section PageSampleHidGesture_SectionHowToExecute 実行手順
* サンプルプログラムをビルドし、実行してください。
*
* @section PageSampleHidGesture_SectionDetail 解説
* サンプルプログラムの処理の流れは以下の通りです。
*
* - ジェスチャ API を Initialize する。
* - 利用可能なジェスチャを全て取得する。
* - eventNumber から新規に発生したジェスチャを選り分ける。
* - 新規に発生したジェスチャを画面表示に反映させる。
*/

#include "frm_Touch.h"

namespace frm{
namespace hid{

    /* Singleton パターン */
    TouchScreen& TouchScreen::GetInstance() NN_NOEXCEPT
    {
        static TouchScreen Instance;
        return Instance;
    }

    uint8_t frameCounter = 80;
    bool f_flash = false;

    bool t_firstCal = false;

    TapState t_St;  // タッチ状態

    int32_t originY;

        void TouchScreen::Initialize() NN_NOEXCEPT
        {
            nn::hid::InitializeGesture();
        }

        const std::list<nn::hid::GestureState>& TouchScreen::GetEvents() const NN_NOEXCEPT
        {
            return m_Events;
        }

        const std::vector<nn::hid::GesturePoint>& TouchScreen::GetPoints() const NN_NOEXCEPT
        {
            return m_Points;
        }

        float TouchScreen::GetScale() NN_NOEXCEPT
        {
            return m_Scale;
        }

        float TouchScreen::GetAngle() NN_NOEXCEPT
        {
            return m_Angle;
        }

        void TouchScreen::Update() NN_NOEXCEPT
        {
            const int count = nn::hid::GetGestureStates(m_States, nn::hid::GestureStateCountMax);

            for (int i = count - 1; i >= 0; --i)
            {
                if (m_EventNumber < m_States[i].eventNumber)
                {
                    this->Precess(m_States[i]);
                }
            }

            m_EventNumber = m_States[0].eventNumber;
        }

        void TouchScreen::Precess(const nn::hid::GestureState& state) NN_NOEXCEPT
        {
            m_Points.clear();

            for (int i = 0; i < state.pointCount; ++i)
            {
                m_Points.push_back(state.points[i]);
            }

            switch (state.GetGestureType())
            {
                case nn::hid::GestureType_Idle:
                case nn::hid::GestureType_Complete:
                case nn::hid::GestureType_Cancel:
                case nn::hid::GestureType_Touch:
                case nn::hid::GestureType_Press:
                case nn::hid::GestureType_Tap:
                case nn::hid::GestureType_Swipe:
                    m_Events.push_back(state);
                    break;

                case nn::hid::GestureType_Pinch:
                    m_Scale *= state.scale;
                    m_Scale = m_Scale > 0.5 ? m_Scale : 0.5f;
                    m_Scale = m_Scale < 2.0 ? m_Scale : 2.0f;
                    break;

                case nn::hid::GestureType_Rotate:
                    m_Angle += state.rotationAngle;
                    m_Angle = m_Angle > -360 ? m_Angle : m_Angle + 360;
                    m_Angle = m_Angle <  360 ? m_Angle : m_Angle - 360;
                    break;

                default:
                    break;
            }

            for (size_t i = m_Events.size(); i > 30; --i)
            {
                m_Events.pop_front();
            }
        }

    void TouchScreen::WriteCursor(
        nn::gfx::util::DebugFontTextWriter* pTextWriter,
        int32_t x, int32_t y) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pTextWriter);

        const float barOffsetX = pTextWriter->CalculateStringWidth("|") / 2;
        const float barOffsetY = pTextWriter->CalculateStringHeight("|") / 2;
        const float minusOffsetX = pTextWriter->CalculateStringWidth("-") / 2;
        const float minusOffsetY = pTextWriter->CalculateStringHeight("-") / 2;
        const float plusOffsetX = pTextWriter->CalculateStringWidth("+") / 2;
        const float plusOffsetY = pTextWriter->CalculateStringHeight("+") / 2;

        pTextWriter->SetCursor(x - barOffsetX - 30, y - barOffsetY);
        pTextWriter->Print("|");
        pTextWriter->SetCursor(x - barOffsetX + 30, y - barOffsetY);
        pTextWriter->Print("|");
        pTextWriter->SetCursor(x - minusOffsetX, y - minusOffsetY - 30);
        pTextWriter->Print("-");
        pTextWriter->SetCursor(x - minusOffsetX, y - minusOffsetY + 30);
        pTextWriter->Print("-");
        pTextWriter->SetCursor(x - plusOffsetX - 30, y - plusOffsetY - 30);
        pTextWriter->Print("+");
        pTextWriter->SetCursor(x - plusOffsetX - 30, y - plusOffsetY + 30);
        pTextWriter->Print("+");
        pTextWriter->SetCursor(x - plusOffsetX + 30, y - plusOffsetY - 30);
        pTextWriter->Print("+");
        pTextWriter->SetCursor(x - plusOffsetX + 30, y - plusOffsetY + 30);
        pTextWriter->Print("+");
    }

    void TouchScreen::WritePoints(
        nn::gfx::util::DebugFontTextWriter* pTextWriter,
        const std::vector<nn::hid::GesturePoint>& points) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pTextWriter);

        pTextWriter->SetTextColor(Color::Green);

        for (size_t i = 0; i < points.size(); ++i)
        {
            WriteCursor(pTextWriter, points[i].x, points[i].y);
        }
    }

    const char* TouchScreen::GetGestureDirection(nn::hid::GestureDirection direction) NN_NOEXCEPT
    {
        switch (direction)
        {
            case nn::hid::GestureDirection_None:
                return "None";
            case nn::hid::GestureDirection_Left:
                return "Left";
            case nn::hid::GestureDirection_Up:
                return "Up";
            case nn::hid::GestureDirection_Right:
                return "Right";
            case nn::hid::GestureDirection_Down:
                return "Down";
            default: NN_UNEXPECTED_DEFAULT;
        }
    }

    int TouchScreen::CalRelativeAxis(nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
    {
        int relY = 0;

        if (t_firstCal == false)
        {
            originY = m_States[0].y;
            t_firstCal = true;
        }

        if (m_States[0].GetGestureType() != nn::hid::GestureType_Idle && m_States[0].y != 0)
        {
            relY = m_States[0].y - originY;
        }
        else
        {
            t_firstCal = false;
        }

        return relY;
    }

    void TouchScreen::makeTouchButton(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                         GraphicsSystem* pGraphicsSystem,
                         const char* title,
                         TapState* t_St,
                         float orgX,
                         float orgY,
                         float width,
                         float height
                         ) NN_NOEXCEPT
    {
        // レンダラの生成
        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        ////////////////////////////////////////////
        // タッチ状態の取得
        ////////////////////////////////////////////
        t_St->btnOld = t_St->btn;
        if (m_States[0].x >= orgX && m_States[0].x <= orgX + width && m_States[0].y >= orgY && m_States[0].y <= orgY + height) {
            t_St->btn = true;
        }
        else
        {
            t_St->btn = false;
        }

        if (t_St->btnOld != t_St->btn)
        {
            if (t_St->btnOld == false)
            {
                t_St->trgOffOn = true;
                t_St->trgOnOff = false;
            }
            else
            {
                t_St->trgOffOn = false;
                t_St->trgOnOff = true;
            }
        }
        else
        {
            t_St->trgOffOn = false;
            t_St->trgOnOff = false;
        }

        pTextWriter->SetScale(1.4, 1.4);
        pTextWriter->SetCursor(orgX + 20, orgY + (height / 2) - 15);

        // 線の幅を指定
        pPrimitiveRenderer->SetLineWidth(1.f);
        // 枠を描画する
        pPrimitiveRenderer->SetColor(HalfColor::White);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), orgX, orgY, width, height);

        pGraphicsSystem->GetPrimitiveRenderer().Draw2DLine(&pGraphicsSystem->GetCommandBuffer(), orgX, orgY, orgX + 4, orgY + 4);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DLine(&pGraphicsSystem->GetCommandBuffer(), orgX + width, orgY, orgX + width - 4, orgY + 4);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DLine(&pGraphicsSystem->GetCommandBuffer(), orgX, orgY + height, orgX + 4, orgY + height - 4);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DLine(&pGraphicsSystem->GetCommandBuffer(), orgX + width, orgY + height, orgX + width - 4, orgY + height - 4);

        pPrimitiveRenderer->SetColor(Color::White);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), orgX + 4, orgY + 4, width - 8, height - 8);

        if (m_States[0].x >= orgX && m_States[0].x <= orgX + width && m_States[0].y >= orgY && m_States[0].y <= orgY + height) {
            // 描画カラーの指定
            pPrimitiveRenderer->SetColor(ClearColor::White);
            // タイトルカラーの指定
            pTextWriter->SetTextColor(Color::Black);
        }
        else
        {
            // 描画カラーの指定
            pPrimitiveRenderer->SetColor(ClearColor::Black);
            // タイトルカラーの指定
            pTextWriter->SetTextColor(Color::White);
        }
        // ボタン背景を描画する
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), orgX, orgY, width, height);
        // ボタンタイトルを描画する
        pTextWriter->Print("%s\n", title);
    }

    void TouchScreen::makeTouchButtonFrame(GraphicsSystem* pGraphicsSystem, float orgX, float orgY, float width, float height) NN_NOEXCEPT
    {
        // レンダラの生成
        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        // 線の幅を指定
        pPrimitiveRenderer->SetLineWidth(4.f);
        // 枠を描画する
        nn::util::Color4u8 clr = { 50, 255, 255, frameCounter };
        pPrimitiveRenderer->SetColor(clr);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), orgX, orgY, width, height);
        if (frameCounter == 255)
        {
            f_flash = true;
        }
        else if (frameCounter == 80)
        {
            f_flash = false;
        }
        else
        {
            // 何もしない
        }
        if (f_flash == true)
        {
            frameCounter = frameCounter - 5;
        }
        else
        {
            frameCounter = frameCounter + 5;
        }
    }
}
} // namespace
