﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/hid/hid_Xpad.h>
#include <nn/hid/debug/hid_Xpad.h>
#include <nn/os.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nntest.h>

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_CLANG)
#pragma clang diagnostic ignored "-Wdeprecated-declarations"
#endif

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_GCC)
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
#endif

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_VC)
#pragma warning(disable:4996)
#endif

#include "../Common/testHid_Xpad.h"

class XpadInitializationSuiteWithParam : public ::testing::TestWithParam<int>
{
protected:
    static ::nn::hid::BasicXpadId g_BasicXpadIds[::nn::hid::XpadIdCountMax];

    static ::nn::hid::BasicXpadState g_BasicXpadState;
};

INSTANTIATE_TEST_CASE_P(XpadInitializationSuite,
                        XpadInitializationSuiteWithParam,
                        ::testing::Range(0, 4));

//!< Xpad の初期化状態は GetXpadState() の呼び出しに反映されるか
TEST_P(XpadInitializationSuiteWithParam, NoInitializationTest1)
{
    const int index = GetParam();

    const int count = ::nn::hid::GetXpadIds(g_BasicXpadIds,
                                            ::nn::hid::XpadIdCountMax);
    ASSERT_GE(::nn::hid::XpadIdCountMax, count);
    ASSERT_GT(count, index);

    for (int i = 0; i < count; ++i)
    {
        if (i != index)
        {
            ::nn::hid::InitializeXpad(g_BasicXpadIds[i]);
        }
    }

    // Xpad が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::GetXpadState(&g_BasicXpadState, g_BasicXpadIds[index]), "");

    ::nn::hid::InitializeXpad(g_BasicXpadIds[index]);

    ::nn::hid::InitializeXpad(g_BasicXpadIds[index]);

    ::nn::hid::debug::FinalizeXpad(g_BasicXpadIds[index]);

    // Xpad が終了しきっていないので成功する。
    ::nn::hid::GetXpadState(&g_BasicXpadState, g_BasicXpadIds[index]);

    for (int i = 0; i < count; ++i)
    {
        ::nn::hid::debug::FinalizeXpad(g_BasicXpadIds[i]);
    }

    // Xpad の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::GetXpadState(&g_BasicXpadState, g_BasicXpadIds[index]), "");
}

//!< Xpad の初期化状態は GetXpadStates() の呼び出しに反映されるか
TEST_P(XpadInitializationSuiteWithParam, NoInitializationTest2)
{
    const int index = GetParam();

    const int count = ::nn::hid::GetXpadIds(g_BasicXpadIds,
                                            ::nn::hid::XpadIdCountMax);
    ASSERT_GE(::nn::hid::XpadIdCountMax, count);
    ASSERT_GT(count, index);

    for (int i = 0; i < count; ++i)
    {
        if (i != index)
        {
            ::nn::hid::InitializeXpad(g_BasicXpadIds[i]);
        }
    }

    // Xpad が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::GetXpadStates(
            &g_BasicXpadState, 1, g_BasicXpadIds[index]), "");

    ::nn::hid::InitializeXpad(g_BasicXpadIds[index]);

    ::nn::hid::InitializeXpad(g_BasicXpadIds[index]);

    ::nn::hid::debug::FinalizeXpad(g_BasicXpadIds[index]);

    // Xpad が終了しきっていないので成功する。
    ::nn::hid::GetXpadStates(&g_BasicXpadState, 1, g_BasicXpadIds[index]);

    for (int i = 0; i < count; ++i)
    {
        ::nn::hid::debug::FinalizeXpad(g_BasicXpadIds[i]);
    }

    // Xpad の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::GetXpadStates(
            &g_BasicXpadState, 1, g_BasicXpadIds[index]), "");
}

//!< Xpad の初期化状態は SetXpadAutoPilotState() の呼び出しに反映されるか
TEST_P(XpadInitializationSuiteWithParam, NoInitializationTest3)
{
    const int index = GetParam();

    const int count = ::nn::hid::GetXpadIds(g_BasicXpadIds,
                                            ::nn::hid::XpadIdCountMax);
    ASSERT_GE(::nn::hid::XpadIdCountMax, count);
    ASSERT_GT(count, index);

    for (int i = 0; i < count; ++i)
    {
        if (i != index)
        {
            ::nn::hid::InitializeXpad(g_BasicXpadIds[i]);
        }
    }

    ::nn::hid::debug::BasicXpadAutoPilotState autoPilotState = {};

    // Xpad が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::SetXpadAutoPilotState(
            g_BasicXpadIds[index], autoPilotState), "");

    ::nn::hid::InitializeXpad(g_BasicXpadIds[index]);

    ::nn::hid::InitializeXpad(g_BasicXpadIds[index]);

    ::nn::hid::debug::FinalizeXpad(g_BasicXpadIds[index]);

    // Xpad が終了しきっていないので成功する。
    ::nn::hid::debug::SetXpadAutoPilotState(g_BasicXpadIds[index],
                                            autoPilotState);

    ::nnt::hid::FinalizeXpadForAutoPilot(g_BasicXpadIds[index]);

    for (int i = 0; i < count; ++i)
    {
        if (i != index)
        {
            ::nn::hid::debug::FinalizeXpad(g_BasicXpadIds[i]);
        }
    }

    // Xpad の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::SetXpadAutoPilotState(
            g_BasicXpadIds[index], autoPilotState), "");
}

//!< Xpad の初期化状態は UnsetXpadAutoPilotState() の呼び出しに反映されるか
TEST_P(XpadInitializationSuiteWithParam, NoInitializationTest4)
{
    const int index = GetParam();

    const int count = ::nn::hid::GetXpadIds(g_BasicXpadIds,
                                            ::nn::hid::XpadIdCountMax);
    ASSERT_GE(::nn::hid::XpadIdCountMax, count);
    ASSERT_GT(count, index);

    for (int i = 0; i < count; ++i)
    {
        if (i != index)
        {
            ::nn::hid::InitializeXpad(g_BasicXpadIds[i]);
        }
    }

    // Xpad が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::UnsetXpadAutoPilotState(g_BasicXpadIds[index]), "");

    ::nn::hid::InitializeXpad(g_BasicXpadIds[index]);

    ::nn::hid::InitializeXpad(g_BasicXpadIds[index]);

    ::nn::hid::debug::FinalizeXpad(g_BasicXpadIds[index]);

    // Xpad が終了しきっていないので成功する。
    ::nn::hid::debug::UnsetXpadAutoPilotState(g_BasicXpadIds[index]);

    for (int i = 0; i < count; ++i)
    {
        ::nn::hid::debug::FinalizeXpad(g_BasicXpadIds[i]);
    }

    // Xpad の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::UnsetXpadAutoPilotState(g_BasicXpadIds[index]), "");
}

//!< Xpad の初期化処理は最低一つの入力状態の取得を保証するか
TEST_P(XpadInitializationSuiteWithParam, StateReadingTest1)
{
    const int index = GetParam();

    const int count = ::nn::hid::GetXpadIds(g_BasicXpadIds,
                                            ::nn::hid::XpadIdCountMax);
    ASSERT_GE(::nn::hid::XpadIdCountMax, count);
    ASSERT_GT(count, index);

    ::nn::hid::InitializeXpad(g_BasicXpadIds[index]);

    // 初期化直後の入力状態の取得に成功する。
    ::nn::hid::GetXpadState(&g_BasicXpadState, g_BasicXpadIds[index]);
    EXPECT_LE(0, g_BasicXpadState.samplingNumber);
    EXPECT_FALSE(g_BasicXpadState.attributes.Test<
                     ::nn::hid::BasicXpadAttribute::IsConnected>());
    EXPECT_TRUE(g_BasicXpadState.buttons.IsAllOff());
    EXPECT_EQ(0, g_BasicXpadState.analogStickR.x);
    EXPECT_EQ(0, g_BasicXpadState.analogStickR.y);
    EXPECT_EQ(0, g_BasicXpadState.analogStickL.x);
    EXPECT_EQ(0, g_BasicXpadState.analogStickL.y);

    ::nn::hid::debug::FinalizeXpad(g_BasicXpadIds[index]);
}

//!< Xpad の初期化処理はサンプリングを正しく有効化するか
TEST_P(XpadInitializationSuiteWithParam, StateReadingTest2)
{
    const int index = GetParam();

    const int count = ::nn::hid::GetXpadIds(g_BasicXpadIds,
                                            ::nn::hid::XpadIdCountMax);
    ASSERT_GE(::nn::hid::XpadIdCountMax, count);
    ASSERT_GT(count, index);

    ::nn::hid::InitializeXpad(g_BasicXpadIds[index]);

    ::nn::hid::InitializeXpad(g_BasicXpadIds[index]);

    ::nn::hid::GetXpadState(&g_BasicXpadState, g_BasicXpadIds[index]);

    int64_t samplingNumber = g_BasicXpadState.samplingNumber;

    ::nn::hid::debug::FinalizeXpad(g_BasicXpadIds[index]);

    // Xpad が終了しきっていないのでサンプリングは継続する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::XpadStateCountMax *
            ::nnt::hid::GetXpadSamplingInterval().GetMilliSeconds()));

    ::nn::hid::GetXpadState(&g_BasicXpadState, g_BasicXpadIds[index]);

    EXPECT_LT(samplingNumber + ::nn::hid::XpadStateCountMax / 4,
              g_BasicXpadState.samplingNumber);

    ::nn::hid::debug::FinalizeXpad(g_BasicXpadIds[index]);
#if defined(NN_BUILD_CONFIG_OS_WIN)
    samplingNumber = g_BasicXpadState.samplingNumber;

    // Xpad が終了しているのでサンプリングは停止する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::XpadStateCountMax *
            ::nnt::hid::GetXpadSamplingInterval().GetMilliSeconds()));

    ::nn::hid::InitializeXpad(g_BasicXpadIds[index]);

    ::nn::hid::GetXpadState(&g_BasicXpadState, g_BasicXpadIds[index]);

    EXPECT_GT(samplingNumber + ::nn::hid::XpadStateCountMax / 4,
              g_BasicXpadState.samplingNumber);

    ::nn::hid::debug::FinalizeXpad(g_BasicXpadIds[index]);
#endif
}

::nn::hid::BasicXpadId XpadInitializationSuiteWithParam::g_BasicXpadIds[
    ::nn::hid::XpadIdCountMax];

::nn::hid::BasicXpadState XpadInitializationSuiteWithParam::g_BasicXpadState;
