﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <nn/nn_Common.h>
#include <nn/hid.h>
#include <nn/hid/hid_PowerState.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/debug/hid_Xpad.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_CLANG)
#pragma clang diagnostic ignored "-Wdeprecated-declarations"
#endif

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_GCC)
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
#endif

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_VC)
#pragma warning(disable:4996)
#endif

#include "../Common/testHid_Xpad.h"

namespace
{

::nn::hid::BasicXpadId g_BasicXpadIds[::nn::hid::XpadIdCountMax];

::nn::hid::BasicXpadState g_BasicXpadStates1[::nn::hid::XpadStateCountMax];
::nn::hid::BasicXpadState g_BasicXpadStates2[::nn::hid::XpadStateCountMax];

//!< Xpad の自動操作状態をファジングします。
void FuzzXpadAutoPilotState(::nn::hid::debug::BasicXpadAutoPilotState* pState,
                            int seed) NN_NOEXCEPT;

//!< Xpad の入力状態が入力無し状態であることを期待します。
void ExpectDefaultXpadState(const ::nn::hid::BasicXpadState& state) NN_NOEXCEPT;

//!< Xpad の入力状態がファジングされた入力状態であることを期待します。
void ExpectFuzzedXpadState(const ::nn::hid::BasicXpadState& state) NN_NOEXCEPT;

//!< Xpad の入力状態が同じドライバの状態から決定されたことを期待します。
void ExpectSameOriginXpadStates(const ::nn::hid::BasicXpadState& lhs,
                                const ::nn::hid::BasicXpadState& rhs
                                ) NN_NOEXCEPT;

//!< Xpad の入力状態が隣り合っていることを期待します。
void ExpectAdjacentXpadStates(const ::nn::hid::BasicXpadState& lhs,
                              const ::nn::hid::BasicXpadState& rhs
                              ) NN_NOEXCEPT;

}

class XpadAutoPilotSuiteWithParam : public ::testing::TestWithParam<int>
{
};

INSTANTIATE_TEST_CASE_P(XpadAutoPilotSuite,
                        XpadAutoPilotSuiteWithParam,
                        ::testing::Range(0, 4));

//!< Xpad の初期化処理は自動操作状態を正しく管理するか
TEST_P(XpadAutoPilotSuiteWithParam, InitializationTest1)
{
    const int xpadIdIndex = GetParam();

    const int xpadIdCount = ::nn::hid::GetXpadIds(g_BasicXpadIds,
                                                  ::nn::hid::XpadIdCountMax);
    ASSERT_GE(::nn::hid::XpadIdCountMax, xpadIdCount);
    ASSERT_GT(xpadIdCount, xpadIdIndex);

    for (int i = 0; i < xpadIdCount; ++i)
    {
        if (i != xpadIdIndex)
        {
            ::nn::hid::InitializeXpad(g_BasicXpadIds[i]);
        }
    }

    const ::nn::hid::BasicXpadId& xpadId = g_BasicXpadIds[xpadIdIndex];

    ::nn::hid::debug::BasicXpadAutoPilotState autoPilotState;
    FuzzXpadAutoPilotState(&autoPilotState, 1);

    ::nn::hid::InitializeXpad(xpadId);

    ::nn::hid::debug::SetXpadAutoPilotState(xpadId, autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::XpadStateCountMax / 2 *
            ::nnt::hid::GetXpadSamplingInterval().GetMilliSeconds()));

    ::nn::hid::BasicXpadState& state = g_BasicXpadStates1[0];

    // 自動操作が反映された入力状態が取得される。
    ::nn::hid::GetXpadState(&state, xpadId);
    {
        SCOPED_TRACE("");
        ExpectFuzzedXpadState(state);
    }

    ::nn::hid::InitializeXpad(xpadId);

    for (int i = 0; i < xpadIdCount; ++i)
    {
        ::nn::hid::debug::FinalizeXpad(g_BasicXpadIds[i]);
    }

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::XpadStateCountMax / 2 *
            ::nnt::hid::GetXpadSamplingInterval().GetMilliSeconds()));

    // まだユーザが存在するので自動操作設定は解除されない。
    ::nn::hid::GetXpadState(&state, xpadId);
    {
        SCOPED_TRACE("");
        ExpectFuzzedXpadState(state);
    }

    ::nnt::hid::FinalizeXpadForAutoPilot(xpadId);

    ::nn::hid::InitializeXpad(xpadId);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::XpadStateCountMax / 2 *
            ::nnt::hid::GetXpadSamplingInterval().GetMilliSeconds()));

    // Xpad が完全に開放されたことで自動操作設定が解除される。
    ::nn::hid::GetXpadState(&state, xpadId);
    {
        SCOPED_TRACE("");
        ExpectDefaultXpadState(state);
    }

    ::nnt::hid::FinalizeXpadForAutoPilot(xpadId);
}

//!< Xpad の入力状態は不正な自動操作状態が設定されても仕様を保証するか
TEST_P(XpadAutoPilotSuiteWithParam, InvalidArgumentTest1)
{
    const int xpadIdIndex = GetParam();

    const int xpadIdCount = ::nn::hid::GetXpadIds(g_BasicXpadIds,
                                                  ::nn::hid::XpadIdCountMax);
    ASSERT_GE(::nn::hid::XpadIdCountMax, xpadIdCount);
    ASSERT_GT(xpadIdCount, xpadIdIndex);

    const ::nn::hid::BasicXpadId& xpadId = g_BasicXpadIds[xpadIdIndex];

    ::nn::hid::debug::BasicXpadAutoPilotState autoPilotState = {};

    // 不正な電源状態
    autoPilotState.powerState = -1;

    // 禁則処理に違反
    autoPilotState.buttons.Set<::nn::hid::BasicXpadButton::Left>(true);
    autoPilotState.buttons.Set<::nn::hid::BasicXpadButton::Up>(true);
    autoPilotState.buttons.Set<::nn::hid::BasicXpadButton::Right>(true);
    autoPilotState.buttons.Set<::nn::hid::BasicXpadButton::Down>(true);

    // 変域を超過
    autoPilotState.analogStickR.x = ::nn::hid::AnalogStickMax;
    autoPilotState.analogStickR.y = ::nn::hid::AnalogStickMax;
    autoPilotState.analogStickL.x = -::nn::hid::AnalogStickMax;
    autoPilotState.analogStickL.y = -::nn::hid::AnalogStickMax;

    ::nn::hid::InitializeXpad(xpadId);

    ::nn::hid::debug::SetXpadAutoPilotState(xpadId, autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::XpadStateCountMax / 2 *
            ::nnt::hid::GetXpadSamplingInterval().GetMilliSeconds()));

    ::nn::hid::BasicXpadState& state = g_BasicXpadStates1[0];
    ::nn::hid::GetXpadState(&state, xpadId);

    ::nnt::hid::FinalizeXpadForAutoPilot(xpadId);

    // 接続状態にある
    EXPECT_TRUE(state.attributes
                     .Test<::nn::hid::BasicXpadAttribute::IsConnected>());

    // 禁則処理が保証されている
    EXPECT_TRUE(state.buttons.Test<::nn::hid::BasicXpadButton::Left>());
    EXPECT_TRUE(state.buttons.Test<::nn::hid::BasicXpadButton::Up>());
    EXPECT_FALSE(state.buttons.Test<::nn::hid::BasicXpadButton::Right>());
    EXPECT_FALSE(state.buttons.Test<::nn::hid::BasicXpadButton::Down>());

    const ::nn::hid::AnalogStickState analogSticks[] = {
        state.analogStickR,
        state.analogStickL
    };

    for (int i = 0; i < 2; ++i)
    {
        SCOPED_TRACE(i == 0 ? "state.analogStickR" : "state.analogStickL");
        // 変域が保証されている
        const int64_t x = analogSticks[i].x;
        const int64_t y = analogSticks[i].y;
        EXPECT_LE(static_cast<int32_t>(::std::sqrt(x * x + y * y)),
                  ::nn::hid::AnalogStickMax);
    }
}

//!< Xpad の入力状態は自動操作状態の設定を正しく反映するか
TEST_P(XpadAutoPilotSuiteWithParam, StateReadingTest1)
{
    const int xpadIdIndex = GetParam();

    const int xpadIdCount =
        ::nn::hid::GetXpadIds(g_BasicXpadIds, ::nn::hid::XpadIdCountMax);
    ASSERT_GE(::nn::hid::XpadIdCountMax, xpadIdCount);
    ASSERT_GT(xpadIdCount, xpadIdIndex);

    const ::nn::hid::BasicXpadId& xpadId = g_BasicXpadIds[xpadIdIndex];

    ::nn::hid::InitializeXpad(xpadId);

    ::nn::hid::GetXpadState(&g_BasicXpadStates1[0], xpadId);
    {
        SCOPED_TRACE("");
        ExpectDefaultXpadState(g_BasicXpadStates1[0]);
    }

    ::nn::hid::debug::BasicXpadAutoPilotState autoPilotState;

    for (int i = 0; i <= ::nn::hid::XpadStateCountMax / 3; ++i)
    {
        FuzzXpadAutoPilotState(&autoPilotState, i);
        ::nn::hid::debug::SetXpadAutoPilotState(xpadId, autoPilotState);
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                2 * ::nnt::hid::GetXpadSamplingInterval().GetMilliSeconds()));
    }

    int count = ::nn::hid::GetXpadStates(
        g_BasicXpadStates1, ::nn::hid::XpadStateCountMax, xpadId);
    EXPECT_LE(1, count);

    bool isActive = true;

    // 自動操作状態に応じて Xpad の入力状態は変化する。
    for (int i = 0; i < count; ++i)
    {
        const ::nn::hid::BasicXpadState& lhs = g_BasicXpadStates1[i];

        EXPECT_LE(0, lhs.samplingNumber);

        if (lhs.buttons.IsAllOff())
        {
            isActive = false;
        }
        else
        {
            SCOPED_TRACE("");
            EXPECT_TRUE(isActive);
            ExpectFuzzedXpadState(lhs);
        }

        if (i + 1 < count)
        {
            SCOPED_TRACE("");
            ExpectAdjacentXpadStates(lhs, g_BasicXpadStates1[i + 1]);
        }
    }

    ::nnt::hid::FinalizeXpadForAutoPilot(xpadId);
}

//!< Xpad の入力状態は自動操作状態の解除を正しく反映するか
TEST_P(XpadAutoPilotSuiteWithParam, StateReadingTest2)
{
    const int xpadIdIndex = GetParam();

    const int xpadIdCount = ::nn::hid::GetXpadIds(g_BasicXpadIds,
                                                  ::nn::hid::XpadIdCountMax);
    ASSERT_GE(::nn::hid::XpadIdCountMax, xpadIdCount);
    ASSERT_GT(xpadIdCount, xpadIdIndex);

    const ::nn::hid::BasicXpadId& xpadId = g_BasicXpadIds[xpadIdIndex];

    ::nn::hid::InitializeXpad(xpadId);

    ::nn::hid::debug::BasicXpadAutoPilotState autoPilotState;
    FuzzXpadAutoPilotState(&autoPilotState, 10);

    ::nn::hid::debug::SetXpadAutoPilotState(xpadId, autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::XpadStateCountMax *
            ::nnt::hid::GetXpadSamplingInterval().GetMilliSeconds()));

    // 自動操作状態を解除
    ::nn::hid::debug::UnsetXpadAutoPilotState(xpadId);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::XpadStateCountMax / 2 *
            ::nnt::hid::GetXpadSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetXpadStates(g_BasicXpadStates1,
                                         ::nn::hid::XpadStateCountMax,
                                         xpadId);
    EXPECT_EQ(::nn::hid::XpadStateCountMax, count);

    bool hasChanged = false;

    ::nn::hid::BasicXpadState state = {};

    // 入力状態列状で自動操作状態は連続的に切り替わる。
    for (int i = 0; i < count; ++i)
    {
        if (i + 1 < count)
        {
            const ::nn::hid::BasicXpadState& lhs = g_BasicXpadStates1[i];
            const ::nn::hid::BasicXpadState& rhs = g_BasicXpadStates1[i + 1];
            SCOPED_TRACE("");
            EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
        }

        if (g_BasicXpadStates1[i].analogStickR.x != state.analogStickR.x)
        {
            EXPECT_FALSE(hasChanged);
            hasChanged = true;
            state.attributes
                 .Set<::nn::hid::BasicXpadAttribute::IsConnected>(true);
            state.buttons = autoPilotState.buttons;
            state.analogStickR = autoPilotState.analogStickR;
            state.analogStickL = autoPilotState.analogStickL;
        }

        {
            SCOPED_TRACE("");
            ExpectSameOriginXpadStates(state, g_BasicXpadStates1[i]);
        }
    }

    EXPECT_TRUE(hasChanged);

    ::nnt::hid::FinalizeXpadForAutoPilot(xpadId);
}

//!< LIFO の境界を跨いで自動操作状態の設定を正しく反映するか
TEST_P(XpadAutoPilotSuiteWithParam, StateReadingTest3)
{
    const int xpadIdIndex = GetParam();

    const int xpadIdCount = ::nn::hid::GetXpadIds(g_BasicXpadIds,
                                                  ::nn::hid::XpadIdCountMax);
    ASSERT_GE(::nn::hid::XpadIdCountMax, xpadIdCount);
    ASSERT_GT(xpadIdCount, xpadIdIndex);

    const ::nn::hid::BasicXpadId& xpadId = g_BasicXpadIds[xpadIdIndex];

    ::nn::hid::InitializeXpad(xpadId);

    ::nn::hid::debug::BasicXpadAutoPilotState autoPilotState;

    int lastCount = 0;

    // LIFO の境界を 2 回跨ぐまでサンプリングを継続
    for (int i = 0; i < ::nn::hid::XpadStateCountMax * 3; ++i)
    {
        FuzzXpadAutoPilotState(&autoPilotState, i);
        ::nn::hid::debug::SetXpadAutoPilotState(xpadId, autoPilotState);
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                ::nnt::hid::GetXpadSamplingInterval().GetMilliSeconds()));

        // 今回のサンプリング結果を格納するバッファ
        ::nn::hid::BasicXpadState *ptr1 = (i % 2 == 1) ? &g_BasicXpadStates1[0]
                                                       : &g_BasicXpadStates2[0];

        // 前回のサンプリング結果が格納されているバッファ
        ::nn::hid::BasicXpadState *ptr2 = (i % 2 == 1) ? &g_BasicXpadStates2[0]
                                                       : &g_BasicXpadStates1[0];

        // 取得される入力状態の数は単調増加する。
        int count = ::nn::hid::GetXpadStates(ptr1,
                                             ::nn::hid::XpadStateCountMax,
                                             xpadId);
        EXPECT_LE(lastCount, count);

        bool isActive = true;

        for (int j = 0; j < count; ++j)
        {
            if (ptr1[j].buttons.IsAllOff())
            {
                isActive = false;
                EXPECT_LE(0, ptr1[i].samplingNumber);
            }
            else
            {
                SCOPED_TRACE("");
                EXPECT_TRUE(isActive);
                ExpectFuzzedXpadState(ptr1[j]);
            }

            if (j + 1 < count)
            {
                SCOPED_TRACE("");
                ExpectAdjacentXpadStates(ptr1[j], ptr1[j + 1]);
            }

            // 前回のサンプリング結果と重複する箇所には同じ値が入る。
            if (lastCount > 0 &&
                ptr1[j].samplingNumber == ptr2[0].samplingNumber)
            {
                for (int k = 0; k < lastCount - j; ++k)
                {
                    SCOPED_TRACE("");
                    EXPECT_EQ(ptr2[k].samplingNumber,
                              ptr1[j + k].samplingNumber);
                    ExpectSameOriginXpadStates(ptr2[k], ptr1[j + k]);
                }

                break;
            }
        }

        lastCount = count;
    }

    EXPECT_EQ(::nn::hid::XpadStateCountMax, lastCount);

    ::nnt::hid::FinalizeXpadForAutoPilot(xpadId);
}

namespace
{

void FuzzXpadAutoPilotState(::nn::hid::debug::BasicXpadAutoPilotState* pState,
                            int seed) NN_NOEXCEPT
{
    pState->powerState = ::nn::hid::PowerState_OnBattery;
    pState->buttons.Reset();
    pState->buttons.Set(seed % (::nn::hid::BasicXpadButton::Down::Index + 1),
                        true);
    pState->analogStickR.x = seed * 3;
    pState->analogStickR.y = seed * 5;
    pState->analogStickL.x = seed * 7;
    pState->analogStickL.y = seed * 11;
}

void ExpectDefaultXpadState(const ::nn::hid::BasicXpadState& state) NN_NOEXCEPT
{
    EXPECT_LE(0, state.samplingNumber);
    EXPECT_FALSE(state.attributes
                      .Test<::nn::hid::BasicXpadAttribute::IsConnected>());
    EXPECT_TRUE(state.buttons.IsAllOff());
    EXPECT_EQ(0, state.analogStickR.x);
    EXPECT_EQ(0, state.analogStickR.y);
    EXPECT_EQ(0, state.analogStickL.x);
    EXPECT_EQ(0, state.analogStickL.y);
}

void ExpectFuzzedXpadState(const ::nn::hid::BasicXpadState& state) NN_NOEXCEPT
{
    EXPECT_LE(0, state.samplingNumber);
    EXPECT_TRUE(state.attributes
                     .Test<::nn::hid::BasicXpadAttribute::IsConnected>());
    EXPECT_EQ(0, state.analogStickR.x % 3);
    EXPECT_EQ(0, state.analogStickR.y % 5);
    EXPECT_EQ(0, state.analogStickL.x % 7);
    EXPECT_EQ(0, state.analogStickL.y % 11);
    const int seed = state.analogStickR.x / 3;
    ::nn::hid::BasicXpadButtonSet buttons = {};
    buttons.Set(seed % (::nn::hid::BasicXpadButton::Down::Index + 1), true);
    EXPECT_EQ(buttons, state.buttons);
    EXPECT_EQ(seed, state.analogStickR.x / 3);
    EXPECT_EQ(seed, state.analogStickR.y / 5);
    EXPECT_EQ(seed, state.analogStickL.x / 7);
    EXPECT_EQ(seed, state.analogStickL.y / 11);
}

void ExpectSameOriginXpadStates(const ::nn::hid::BasicXpadState& lhs,
                                const ::nn::hid::BasicXpadState& rhs
                                ) NN_NOEXCEPT
{
    EXPECT_EQ(lhs.attributes, rhs.attributes);
    EXPECT_EQ(lhs.buttons, rhs.buttons);
    EXPECT_EQ(lhs.analogStickR.x, rhs.analogStickR.x);
    EXPECT_EQ(lhs.analogStickR.y, rhs.analogStickR.y);
    EXPECT_EQ(lhs.analogStickL.x, rhs.analogStickL.x);
    EXPECT_EQ(lhs.analogStickL.y, rhs.analogStickL.y);
}

void ExpectAdjacentXpadStates(const ::nn::hid::BasicXpadState& lhs,
                              const ::nn::hid::BasicXpadState& rhs
                              ) NN_NOEXCEPT
{
    EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
    EXPECT_GE(lhs.analogStickR.x, rhs.analogStickR.x);
    EXPECT_GE(lhs.analogStickR.y, rhs.analogStickR.y);
    EXPECT_GE(lhs.analogStickL.x, rhs.analogStickL.x);
    EXPECT_GE(lhs.analogStickL.y, rhs.analogStickL.y);
}

}
