﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_NpadSystem.h>
#include <nn/hid/system/hid_NpadSystemExt.h>
#include <nn/hid/hid_Vibration.h>
#include <nn/hid/hid_VibrationGcErm.h>

#include "../Npad/testHid_NpadCommon.h"

namespace {

    const int VibrationDeviceCountMax = 2;

    enum TestPattern
    {
        TestPattern_Argment,
        TestPattern_NoArgment,
    };

    //デバイス情報の確認
    void CheckDeviceInfo(nn::hid::VibrationDeviceInfo* pDeviceInfo, nn::hid::NpadStyleSet style)
    {
        if(style == nn::hid::NpadStyleGc::Mask)
        {
            // Position は None
            EXPECT_EQ(pDeviceInfo->position, nn::hid::VibrationDevicePosition_None);
            // StyleGc の場合は GcErm になる
            EXPECT_EQ(pDeviceInfo->deviceType, nn::hid::VibrationDeviceType_GcErm);
        }
        else if (style == nn::hid::NpadStyleJoyLeft::Mask)
        {
            EXPECT_EQ(pDeviceInfo->position, nn::hid::VibrationDevicePosition_Left);
            EXPECT_EQ(pDeviceInfo->deviceType, nn::hid::VibrationDeviceType_LinearResonantActuator);
        }
        else if (style == nn::hid::NpadStyleJoyRight::Mask)
        {
            EXPECT_EQ(pDeviceInfo->position, nn::hid::VibrationDevicePosition_Right);
            EXPECT_EQ(pDeviceInfo->deviceType, nn::hid::VibrationDeviceType_LinearResonantActuator);
        }
        else
        {
            // 現時点では振動子の位置は必ず左右どちらかになる
            EXPECT_GE(pDeviceInfo->position, nn::hid::VibrationDevicePosition_Left);
            EXPECT_LE(pDeviceInfo->position, nn::hid::VibrationDevicePosition_Right);

            // 現時点では振動子の種類は必ずLRAになる
            EXPECT_EQ(pDeviceInfo->deviceType, nn::hid::VibrationDeviceType_LinearResonantActuator);
        }
    }

    void CheckVibrationTarget(nn::hid::VibrationTarget* pTarget, nn::hid::NpadStyleSet style)
    {
        //現在の振動値の確認
        //nn::hid::VibrationNode::Update()で、VibrationTargetの保持している振動値が更新されます。
        //update()前と、update()後のGetActualVibrationの値をそれぞれチェックしています。
        for (int i = 0; i < 2; i++)
        {
            nn::hid::VibrationValue value;
            pTarget->GetActualVibration(&value);
            //接続していないので値は必ず0になる。
            EXPECT_FLOAT_EQ(value.amplitudeHigh, 0.0f);
            EXPECT_FLOAT_EQ(value.amplitudeLow, 0.0f);
            EXPECT_FLOAT_EQ(value.frequencyHigh, static_cast<float>(nn::hid::VibrationFrequencyHighDefault));
            EXPECT_FLOAT_EQ(value.frequencyLow, static_cast<float>(nn::hid::VibrationFrequencyLowDefault));

            nn::hid::VibrationNode::Update();
        }

        //デバイス情報の確認
        nn::hid::VibrationDeviceInfo info;
        pTarget->GetVibrationDeviceInfo(&info);
        CheckDeviceInfo(&info, style);

        //ハンドル解放
        pTarget->UnsetVibrationDeviceHandle();
        EXPECT_EQ(pTarget->IsVibrationDeviceHandleSet(), false);
    }

    void VibrationTargetTest(nn::hid::VibrationDeviceHandle handle, nn::hid::NpadStyleSet style, TestPattern pattern)
    {
        nn::hid::VibrationTarget* target;

        if (pattern == TestPattern_Argment)
        {
            //引数ありコンストラクタ
            target = new nn::hid::VibrationTarget(handle);
        }
        else if (pattern == TestPattern_NoArgment)
        {
            //引数なしコンストラクタ
            target = new nn::hid::VibrationTarget();
            EXPECT_EQ(target->IsVibrationDeviceHandleSet(), false);

            //ハンドル設定
            target->SetVibrationDeviceHandle(handle);
        }
        else
        {
            target = new nn::hid::VibrationTarget();
            FAIL();//そんな定義はない
        }

        EXPECT_EQ(target->IsVibrationDeviceHandleSet(), true);

        //ハンドル取得
        nn::hid::VibrationDeviceHandle deviceHandle = target->GetVibrationDeviceHandle();
        EXPECT_EQ(deviceHandle._storage, handle._storage);

        CheckVibrationTarget(target, style);

        delete target;
    }

    void VibrationDeviceHandleTest(const nn::hid::NpadIdType& id, nn::hid::NpadStyleSet style, int vibrationDeviceCountMax, TestPattern pattern)
    {
        nn::hid::InitializeNpad();

        // 使用する操作形態を設定する
        nn::hid::SetSupportedNpadStyleSet(style);

        // 使用する Npad を設定する
        nn::hid::SetSupportedNpadIdType(testhid::NpadIds, NN_ARRAY_SIZE(testhid::NpadIds));

        // 振動子のハンドルを取得する
        nn::hid::VibrationDeviceHandle handles[VibrationDeviceCountMax];

        int m_VibrationDeviceCount = nn::hid::GetVibrationDeviceHandles(
            handles, vibrationDeviceCountMax, id, style);

        // 0 <= m_VibrationDeviceCount <= VibrationDeviceCountMax
        EXPECT_GE(m_VibrationDeviceCount, 0);
        EXPECT_LE(m_VibrationDeviceCount, VibrationDeviceCountMax);

        for (int i = 0; i < m_VibrationDeviceCount; i++)
        {
            // 振動子を初期化する
            nn::hid::InitializeVibrationDevice(handles[i]);

            //正常系テスト
            VibrationTargetTest(handles[i], style, pattern);
        }
    }
}

//!< VibrationTargetのテスト 引数ありコンストラクタ（正常系）
TEST(VibrationTargetSuite, NormalTest1)
{
    int tests[] = { 1,2 };

    for (int id = 0; id < testhid::NpadIdCountMax; id++)
    {
        for (int style = 0; style < testhid::NpadStyleCountMax; style++)
        {
            for (auto number : tests)
            {
                VibrationDeviceHandleTest(testhid::NpadIds[id], testhid::NpadStyles[style], number, TestPattern_Argment);
            }
        }
    }
}

//!< VibrationTargetのテスト　引数なしコンストラクタ（正常系）
TEST(VibrationTargetSuite, NormalTest2)
{
    int tests[] = { 1,2 };

    for (int id = 0; id < testhid::NpadIdCountMax; id++)
    {
        for (int style = 0; style < testhid::NpadStyleCountMax; style++)
        {
            for (auto number : tests)
            {
                VibrationDeviceHandleTest(testhid::NpadIds[id], testhid::NpadStyles[style], number, TestPattern_NoArgment);
            }
        }
    }
}

//!< VibrationTargetにVibrationDeviceHandleを設定しない場合のテスト（正常系）
TEST(VibrationTargetSuite, NormalTest3)
{
    nn::hid::VibrationTarget target;
    EXPECT_EQ(target.IsVibrationDeviceHandleSet(), false);

    //ハンドル解放
    target.UnsetVibrationDeviceHandle();
    EXPECT_EQ(target.IsVibrationDeviceHandleSet(), false);

    //ハンドル取得
    nn::hid::VibrationDeviceHandle handle = target.GetVibrationDeviceHandle();
    EXPECT_EQ(handle._storage, 0);

    //現在の振動値の確認
    nn::hid::VibrationValue value;
    target.GetActualVibration(&value);
    EXPECT_FLOAT_EQ(value.amplitudeHigh, 0.0f);
    EXPECT_FLOAT_EQ(value.amplitudeLow, 0.0f);
    EXPECT_FLOAT_EQ(value.frequencyHigh, static_cast<float>(nn::hid::VibrationFrequencyHighDefault));
    EXPECT_FLOAT_EQ(value.frequencyLow, static_cast<float>(nn::hid::VibrationFrequencyLowDefault));

    //デバイス情報の確認
    nn::hid::VibrationDeviceInfo info;
    target.GetVibrationDeviceInfo(&info);
    EXPECT_EQ(info.deviceType, nn::hid::VibrationDeviceType_Unknown);
    EXPECT_EQ(info.position, nn::hid::VibrationDevicePosition_None);
}
