﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nntest.h>

#include <nn/nn_Log.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/hid/system/hid_Result.system.h>

#include <nn/result/result_HandlingUtility.h>

namespace
{
    template<typename T, ::std::size_t U>
    ::std::size_t GetArrayLength(const T (&staticArray)[U])
    {
        NN_UNUSED(staticArray);
        return U;
    }

    const ::nn::hid::NpadIdType NpadIds[] = {::nn::hid::NpadId::No1,
                                             ::nn::hid::NpadId::No2,
                                             ::nn::hid::NpadId::No3,
                                             ::nn::hid::NpadId::No4,
                                             ::nn::hid::NpadId::No5,
                                             ::nn::hid::NpadId::No6,
                                             ::nn::hid::NpadId::No7,
                                             ::nn::hid::NpadId::No8,
                                             ::nn::hid::NpadId::Handheld};

    const int NpadIdCountMax = static_cast<int>(GetArrayLength(NpadIds));

    const ::nn::hid::NpadStyleSet SupportedStyles = (::nn::hid::NpadStyleFullKey::Mask |
                                                     ::nn::hid::NpadStyleJoyDual::Mask |
                                                     ::nn::hid::NpadStyleJoyLeft::Mask |
                                                     ::nn::hid::NpadStyleJoyRight::Mask);

    const int ListCount = 2; // セッション数の上限に関する対処

    void Initialize()
    {
        nn::hid::InitializeNpad();

        nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);

        ::nn::hid::SetSupportedNpadStyleSet(SupportedStyles);
    }
}

// UniquePad を正しく列挙するか
TEST(UniquePadInitialization, DISABLED_ListUniquePadsTest1)
{

    Initialize();

    ::nn::hid::system::UniquePadId uniquePadIds[::nn::hid::system::UniquePadIdCountMax + 1];

    // 引数の事前条件チェックが有効か
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::system::ListUniquePads(nullptr, 1), "");
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::system::ListUniquePads(uniquePadIds, -1), "");
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::system::ListUniquePads(uniquePadIds, ::nn::hid::system::UniquePadIdCountMax + 1), "");

    for(auto& id : uniquePadIds)
    {
        id._storage = 0xDEADBEEFDEADBEEF;
    }

    const int OutCount = ::nn::hid::system::ListUniquePads(uniquePadIds,
                                                           ListCount);

    // 返り値の値域が仕様通りか
    EXPECT_TRUE(0 <= OutCount && OutCount <= ::nn::hid::system::UniquePadIdCountMax);

    // outCount 以降のバッファの状態は変えない
    for(int i = OutCount; i < ::nn::hid::system::UniquePadIdCountMax; i++)
    {
        EXPECT_TRUE(uniquePadIds[i]._storage == 0xDEADBEEFDEADBEEF);
    }
}

// GetUniquePadControllerNumber は正しく UniquePad の状態を取得するか
TEST(UniquePadInitialization, DISABLED_GetUniquePadControllerNumberTest1)
{
    Initialize();

    ::nn::hid::system::UniquePadId uniquePadIds[::nn::hid::system::UniquePadIdCountMax];

    const int OutCount = ::nn::hid::system::ListUniquePads(uniquePadIds,
                                                           ListCount);
    EXPECT_TRUE(0 <= OutCount && OutCount <= ::nn::hid::system::UniquePadIdCountMax);

    // 引数の事前条件チェックが有効か
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::system::GetUniquePadControllerNumber(nullptr, uniquePadIds[0]), "");

    for(int i = 0; i < ListCount; i++)
    {
        int uniquePadControllerNumber = -1;
        const auto& result = ::nn::hid::system::GetUniquePadControllerNumber(&uniquePadControllerNumber,
                                                                             uniquePadIds[i]);

        if(ResultUniquePadDisconnected::Includes(result) ||
           ResultUniquePadNotAvailable::Includes(result))
        {
            // バッファの状態を変えない
            EXPECT_TRUE(uniquePadControllerNumber == -1);
        }

        if(result.IsSuccess())
        {
            // コントローラ番号が規定の値域以内
            EXPECT_TRUE(0 <= uniquePadControllerNumber && uniquePadControllerNumber < 8);
            // NN_LOG("ID=0x%x, ControllerNumber = 0x%x\n", uniquePadIds[i],
            //                                              uniquePadControllerNumber);
        }
    }
}

// GetUniquePadInterface は正しく UniquePad の状態を取得するか
TEST(UniquePadInitialization, DISABLED_GetUniquePadInterfaceTest1)
{
    Initialize();

    ::nn::hid::system::UniquePadId uniquePadIds[::nn::hid::system::UniquePadIdCountMax];

    const int OutCount = ::nn::hid::system::ListUniquePads(uniquePadIds,
                                                           ListCount);
    EXPECT_TRUE(0 <= OutCount && OutCount <= ::nn::hid::system::UniquePadIdCountMax);

    // 引数の事前条件チェックが有効か
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::system::GetUniquePadInterface(nullptr, uniquePadIds[0]), "");


    for(int i = 0; i < ListCount; i++)
    {
        ::nn::hid::system::UniquePadInterface uniquePadInterface;
        const auto& result = ::nn::hid::system::GetUniquePadInterface(&uniquePadInterface,
                                                                      uniquePadIds[i]);

        if(ResultUniquePadDisconnected::Includes(result))
        {
            // 何もしない
        }

        if(result.IsSuccess())
        {
            EXPECT_TRUE(uniquePadInterface == ::nn::hid::system::UniquePadInterface_Bluetooth ||
                        uniquePadInterface == ::nn::hid::system::UniquePadInterface_Rail      ||
                        uniquePadInterface == ::nn::hid::system::UniquePadInterface_Usb       ||
                        uniquePadInterface == ::nn::hid::system::UniquePadInterface_Embedded);

            // NN_LOG("ID=0x%x, Interface = 0x%x\n", uniquePadIds[i],
            //                                       uniquePadInterface);
        }
    }
}

// GetUniquePadType は正しく UniquePad の状態を取得するか
TEST(UniquePadInitialization, DISABLED_GetUniquePadTypeTest1)
{
    Initialize();

    ::nn::hid::system::UniquePadId uniquePadIds[::nn::hid::system::UniquePadIdCountMax];

    const int OutCount = ::nn::hid::system::ListUniquePads(uniquePadIds,
                                                           ListCount);
    EXPECT_TRUE(0 <= OutCount && OutCount <= ::nn::hid::system::UniquePadIdCountMax);

    for(int i = 0; i < ListCount; i++)
    {
        ::nn::hid::system::UniquePadInterface uniquePadInterface;
        const auto& result = ::nn::hid::system::GetUniquePadInterface(&uniquePadInterface,
                                                                      uniquePadIds[i]);
        if(result.IsSuccess())
        {
            ::nn::hid::system::UniquePadType uniquePadType;
            uniquePadType = ::nn::hid::system::GetUniquePadType(uniquePadIds[i]);

            EXPECT_TRUE(uniquePadType == ::nn::hid::system::UniquePadType_Embedded          ||
                        uniquePadType == ::nn::hid::system::UniquePadType_FullKeyController ||
                        uniquePadType == ::nn::hid::system::UniquePadType_LeftController    ||
                        uniquePadType == ::nn::hid::system::UniquePadType_RightController);
            // NN_LOG("ID=0x%x, Type = 0x%x\n", uniquePadIds[i],
            //                                  uniquePadType);
        }
    }
}
