﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_Mouse.h>
#include <nn/hid/debug/hid_Mouse.h>
#include <nn/os.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nntest.h>

#include "../Common/testHid_Mouse.h"

namespace
{
::nn::hid::MouseState g_State;
} // namespace

//!< Mouse の初期化状態は GetMouseState() の呼び出しに反映されるか
TEST(MouseInitializationSuite, NoInitializationTest1)
{
    // Mouse が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::GetMouseState(&g_State), "");

    ::nn::hid::InitializeMouse();

    ::nn::hid::InitializeMouse();

    ::nn::hid::debug::FinalizeMouse();

    // Mouse が終了しきっていないので成功する。
    ::nn::hid::GetMouseState(&g_State);

    ::nn::hid::debug::FinalizeMouse();

    // Mouse の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::GetMouseState(&g_State), "");
}

//!< Mouse の初期化状態は GetMouseStates() の呼び出しに反映されるか
TEST(MouseInitializationSuite, NoInitializationTest2)
{
    // Mouse が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::GetMouseStates(&g_State, 1), "");

    ::nn::hid::InitializeMouse();

    ::nn::hid::InitializeMouse();

    ::nn::hid::debug::FinalizeMouse();

    // Mouse が終了しきっていないので成功する。
    ::nn::hid::GetMouseState(&g_State);

    ::nn::hid::debug::FinalizeMouse();

    // Mouse の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::GetMouseStates(&g_State, 1), "");
}

//!< Mouse の初期化状態は SetMouseAutoPilotState() の呼び出しに反映されるか
TEST(MouseInitializationSuite, NoInitializationTest3)
{
    ::nn::hid::debug::MouseAutoPilotState autoPilotState = {};

    // Mouse が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState), "");

    ::nn::hid::InitializeMouse();

    ::nn::hid::InitializeMouse();

    ::nn::hid::debug::FinalizeMouse();

    // Mouse が終了しきっていないので成功する。
    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nnt::hid::FinalizeMouseForAutoPilot();

    // Mouse の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState), "");
}

//!< Mouse の初期化状態は UnsetMouseAutoPilotState() の呼び出しに反映されるか
TEST(MouseInitializationSuite, NoInitializationTest4)
{
    // Mouse が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::debug::UnsetMouseAutoPilotState(), "");

    ::nn::hid::InitializeMouse();

    ::nn::hid::InitializeMouse();

    ::nn::hid::debug::FinalizeMouse();

    // Mouse が終了しきっていないので成功する。
    ::nn::hid::debug::UnsetMouseAutoPilotState();

    ::nn::hid::debug::FinalizeMouse();

    // Mouse の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::debug::UnsetMouseAutoPilotState(), "");
}

//!< Mouse の初期化処理は最低一つの入力状態の取得を保証するか
TEST(MouseInitializationSuite, StateReadingTest1)
{
    ::nn::hid::InitializeMouse();

    // 初期化直後の入力状態の取得に成功する。
    ::nn::hid::GetMouseState(&g_State);
    EXPECT_LE(0, g_State.samplingNumber);
    EXPECT_EQ(0, g_State.x);
    EXPECT_EQ(0, g_State.y);
    EXPECT_EQ(0, g_State.deltaX);
    EXPECT_EQ(0, g_State.deltaY);
    EXPECT_EQ(0, g_State.wheelDelta);
    EXPECT_TRUE(g_State.buttons.IsAllOff());
    EXPECT_TRUE(
        g_State.attributes.Test<::nn::hid::MouseAttribute::Transferable>());

    ::nn::hid::debug::FinalizeMouse();
}

//!< Mouse の初期化処理はサンプリングを正しく有効化するか
TEST(MouseInitializationSuite, StateReadingTest2)
{
    ::nn::hid::InitializeMouse();

    ::nn::hid::InitializeMouse();

    ::nn::hid::GetMouseState(&g_State);

    int64_t samplingNumber = g_State.samplingNumber;

    ::nn::hid::debug::FinalizeMouse();

    // Mouse が終了しきっていないのでサンプリングは継続する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    ::nn::hid::GetMouseState(&g_State);

    EXPECT_LT(samplingNumber + ::nn::hid::MouseStateCountMax / 4,
              g_State.samplingNumber);

    ::nn::hid::debug::FinalizeMouse();
#if defined(NN_BUILD_CONFIG_OS_WIN)
    samplingNumber = g_State.samplingNumber;

    // Mouse が終了しているのでサンプリングは停止する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    ::nn::hid::InitializeMouse();

    ::nn::hid::GetMouseState(&g_State);

    EXPECT_GT(samplingNumber + ::nn::hid::MouseStateCountMax / 4,
              g_State.samplingNumber);

    ::nn::hid::debug::FinalizeMouse();
#endif
}
