﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/debug/hid_Keyboard.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include "../Common/testHid_Keyboard.h"

namespace
{

::nn::hid::KeyboardState g_States[::nn::hid::KeyboardStateCountMax];

//!< 指定の優先度のスレッドから入力状態の読み出しが可能かテストします。
void ThreadingTest(int priority);

} // namespace

//!< 入力状態の読み出し時にバッファは正しく利用されるか
TEST(KeyboardReadingStateSuite, ReadingStateTest1)
{
    for (::nn::hid::KeyboardState& state : g_States)
    {
        state = ::nn::hid::KeyboardState();
        state.samplingNumber = 0xDEADBEEF;
        state.keys.Set(1, true);
        state.modifiers.Set(2, true);
        state.attributes.Set(3, true);
    }

    ::nn::hid::InitializeKeyboard();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::KeyboardStateCountMax / 2 *
            ::nnt::hid::GetKeyboardSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetKeyboardStates(g_States,
                                             ::nn::hid::KeyboardStateCountMax);

    EXPECT_GE(::nn::hid::KeyboardStateCountMax, count);

    for (int i = count; i < ::nn::hid::KeyboardStateCountMax; ++i)
    {
        // 入力状態の書き込まれなかった領域は値を維持する。
        EXPECT_EQ(0xDEADBEEF, g_States[i].samplingNumber);
        auto keys = ::nn::hid::KeyboardKeySet();
        keys.Set(1, true);
        EXPECT_EQ(keys, g_States[i].keys);
        auto modifiers = ::nn::hid::KeyboardModifierSet();
        modifiers.Set(2, true);
        EXPECT_EQ(modifiers, g_States[i].modifiers);
        auto attributes = ::nn::hid::KeyboardAttributeSet();
        attributes.Set(3, true);
        EXPECT_EQ(attributes, g_States[i].attributes);
    }

    ::nn::hid::debug::FinalizeKeyboard();
}

//!< 高優先度のスレッドから入力状態の読み出しが可能か
TEST(KeyboardReadingStateSuite, ThreadingTest1)
{
    ThreadingTest(::nn::os::HighestThreadPriority);
}

//!< 低優先度のスレッドから入力状態の読み出しが可能か
TEST(KeyboardReadingStateSuite, ThreadingTest2)
{
    ThreadingTest(::nn::os::LowestThreadPriority);
}

namespace
{

void ThreadingTest(int priority)
{
    ::nn::os::ThreadType* pThread = ::nn::os::GetCurrentThread();

    // 指定の優先度に設定する。
    const int original = ::nn::os::ChangeThreadPriority(pThread, priority);

    ::nn::hid::InitializeKeyboard();

    int lastCount = 1;
    int64_t lastSamplingNumber = 0;

    for (int i = 0; i < 60; ++i)
    {
        int count = ::nn::hid::GetKeyboardStates(
            g_States,
            ::nn::hid::KeyboardStateCountMax);
        EXPECT_LE(lastCount, count);
        EXPECT_LE(lastSamplingNumber, g_States[0].samplingNumber);
        for (int j = 0; j < count - 1; ++j)
        {
            const ::nn::hid::KeyboardState& lhs = g_States[j];
            const ::nn::hid::KeyboardState& rhs = g_States[j + 1];
            EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
        }

        lastCount = count;
        lastSamplingNumber = g_States[0].samplingNumber;

        ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(16));
    }

    EXPECT_EQ(::nn::hid::KeyboardStateCountMax, lastCount);

    ::nn::hid::debug::FinalizeKeyboard();

    // 優先度を元に戻す。
    ::nn::os::ChangeThreadPriority(pThread, original);
}

} // namespace
