﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_KeyCode.h>
#include <nnt/nntest.h>

//!< Usage Id の境界値が正しく処理されるか
TEST(KeyCodeMapSuite, UsageIdTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedStates;

    keyCode = uint16_t();
    usageId = static_cast<int>(::std::numeric_limits<uint16_t>::min());
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index - 1;
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);

    keyCode = uint16_t();
    usageId = static_cast<int>(::std::numeric_limits<uint16_t>::max());
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< Alt Gr キーの押下が正しく判定されるか
TEST(KeyCodeMapSuite, AltGrTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayoutType();

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    layout = ::nn::hid::KeyboardLayout::EnglishUnitedStatesInternational;
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::LeftAlt>();
    layout = ::nn::hid::KeyboardLayout::EnglishUnitedStatesInternational;
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    modifiers.Set<::nn::hid::KeyboardModifier::LeftAlt>();
    layout = ::nn::hid::KeyboardLayout::EnglishUnitedStatesInternational;
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00E1, keyCode);
}

//!< Shift キーの押下が正しく判定されるか
TEST(KeyCodeMapSuite, ShiftTest1)
{
    auto keyCode = uint16_t();
    auto usageId = ::nn::hid::KeyboardKey::A::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedStates;

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0041, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::LeftAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::Gui>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< 英語 (イギリス) のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedKingdomTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedKingdom;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Backslash::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D3::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00A3, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0023, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< 英語 (イギリス) のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedKingdomTest2)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedKingdom;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::U::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00FA, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::U::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00DA, keyCode);
}

//!< 英語 (イギリス) のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedKingdomTest3)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedKingdom;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Hiragana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< 英語 (イギリス) のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedKingdomTest4)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedKingdom;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Katakana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< 英語 (イギリス) のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedKingdomTest5)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedKingdom;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0041, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00C1, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D9::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0039, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D9::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0028, keyCode);
}

//!< 英語 (イギリス) のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedKingdomTest6)
{
    auto keyCode = uint16_t();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedKingdom;

    keyCode = uint16_t();
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);
}

//!< 英語 (アメリカ) のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedStatesTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedStates;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Backquote::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0060, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D2::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0040, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Tilde::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< 英語 (アメリカ) のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedStatesTest2)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedStates;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    modifiers.Set<::nn::hid::KeyboardModifier::LeftAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< 英語 (アメリカ) のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedStatesTest3)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedStates;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Hiragana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< 英語 (アメリカ) のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedStatesTest4)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedStates;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Katakana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< 英語 (アメリカ) のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedStatesTest5)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedStates;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::C::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0043, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Backquote::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0060, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Backquote::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x007E, keyCode);
}

//!< 英語 (アメリカ) のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedStatesTest6)
{
    auto keyCode = uint16_t();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedStates;

    keyCode = uint16_t();
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);
}

//!< 英語 (US-International) のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedStatesInternationalTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedStatesInternational;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Z::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x007A, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Backquote::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0303, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Tilde::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< 英語 (US-International) のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedStatesInternationalTest2)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedStatesInternational;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::T::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00FE, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::T::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00DE, keyCode);
}

//!< 英語 (US-International) のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedStatesInternationalTest3)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedStatesInternational;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Hiragana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< 英語 (US-International) のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedStatesInternationalTest4)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedStatesInternational;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Katakana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< 英語 (US-International) のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedStatesInternationalTest5)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedStatesInternational;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Comma::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x002C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Comma::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00C7, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00AB, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Comma::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003C, keyCode);
}

//!< 英語 (US-International) のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, EnglishUnitedStatesInternationalTest6)
{
    auto keyCode = uint16_t();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::EnglishUnitedStatesInternational;

    keyCode = uint16_t();
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);
}

//!< フランス語のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, FrenchTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::French;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Semicolon::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x006D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D1::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0031, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x002A, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< フランス語のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeMapSuite, FrenchTest2)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::French;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::CloseBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00A4, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::CloseBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< フランス語のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeMapSuite, FrenchTest3)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::French;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Hiragana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0071, keyCode);
}

//!< フランス語のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeMapSuite, FrenchTest4)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::French;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Katakana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0071, keyCode);
}

//!< フランス語のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, FrenchTest5)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::French;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0051, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x20AC, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Backslash::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Backslash::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003E, keyCode);
}

//!< フランス語のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, FrenchTest6)
{
    auto keyCode = uint16_t();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::French;

    keyCode = uint16_t();
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);
}

//!< フランス語 (カナダ) のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, FrenchCanadaTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::FrenchCanada;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Slash::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00E9, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Slash::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00C9, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< フランス語 (カナダ) のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeMapSuite, FrenchCanadaTest2)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::FrenchCanada;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::P::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00B6, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::P::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< フランス語 (カナダ) のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeMapSuite, FrenchCanadaTest3)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::FrenchCanada;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Hiragana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< フランス語 (カナダ) のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeMapSuite, FrenchCanadaTest4)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::FrenchCanada;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Katakana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< フランス語 (カナダ) のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, FrenchCanadaTest5)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::FrenchCanada;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::O::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x004F, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::O::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00A7, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D1::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0031, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D1::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0021, keyCode);
}

//!< フランス語 (カナダ) のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, FrenchCanadaTest6)
{
    auto keyCode = uint16_t();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::FrenchCanada;

    keyCode = uint16_t();
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);
}

//!< ドイツ語のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, GermanTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::German;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Z::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0079, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Y::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005A, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0023, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< ドイツ語のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeMapSuite, GermanTest2)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::German;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Q::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0040, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Q::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< ドイツ語のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeMapSuite, GermanTest3)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::German;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Hiragana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< ドイツ語のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeMapSuite, GermanTest4)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::German;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Katakana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< ドイツ語のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, GermanTest5)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::German;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::M::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x004D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::M::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00B5, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Plus::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0301, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Plus::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0300, keyCode);
}

//!< ドイツ語のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, GermanTest6)
{
    auto keyCode = uint16_t();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::German;

    keyCode = uint16_t();
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);
}

//!< イタリア語のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, ItalianTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Italian;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00E8, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00E9, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00F9, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< イタリア語のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeMapSuite, ItalianTest2)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Italian;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005B, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x007B, keyCode);
}

//!< イタリア語のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeMapSuite, ItalianTest3)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Italian;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Hiragana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< イタリア語のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeMapSuite, ItalianTest4)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Italian;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Katakana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< イタリア語のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, ItalianTest5)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Italian;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Z::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005A, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D5::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x20AC, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D0::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0030, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D0::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);
}

//!< イタリア語のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, ItalianTest6)
{
    auto keyCode = uint16_t();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Italian;

    keyCode = uint16_t();
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);
}

//!< 日本語のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, JapaneseTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Japanese;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0041, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::ZenkakuHankaku::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Yen::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Muhenkan::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< 日本語のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeMapSuite, JapaneseTest2)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Japanese;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    modifiers.Set<::nn::hid::KeyboardModifier::LeftAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< 日本語のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeMapSuite, JapaneseTest3)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Japanese;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D3::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Hiragana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x3042, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D3::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::Hiragana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x3041, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D3::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    modifiers.Set<::nn::hid::KeyboardModifier::Hiragana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x3042, keyCode);
}

//!< 日本語のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeMapSuite, JapaneseTest4)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Japanese;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Katakana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x30A4, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::Katakana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x30A3, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    modifiers.Set<::nn::hid::KeyboardModifier::Katakana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x30A4, keyCode);
}

//!< 日本語のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, JapaneseTest5)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Japanese;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::C::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0043, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Quote::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003A, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Quote::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x002A, keyCode);
}

//!< 日本語のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, JapaneseTest6)
{
    auto keyCode = uint16_t();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Japanese;

    keyCode = uint16_t();
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Hiragana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    modifiers.Set<::nn::hid::KeyboardModifier::Hiragana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Katakana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    modifiers.Set<::nn::hid::KeyboardModifier::Katakana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);
}

//!< ポルトガル語のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, PortugueseTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Portuguese;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Semicolon::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00E7, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Semicolon::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00C7, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0303, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< ポルトガル語のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeMapSuite, PortugueseTest2)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Portuguese;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0308, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< ポルトガル語のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeMapSuite, PortugueseTest3)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Portuguese;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Hiragana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< ポルトガル語のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeMapSuite, PortugueseTest4)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Portuguese;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Katakana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< ポルトガル語のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, PortugueseTest5)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Portuguese;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0045, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x20AC, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D8::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0038, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D8::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0028, keyCode);
}

//!< ポルトガル語のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, PortugueseTest6)
{
    auto keyCode = uint16_t();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Portuguese;

    keyCode = uint16_t();
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);
}

//!< ロシア語 (キリル) のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, RussianCyrillicTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::RussianCyrillic;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::CloseBracket::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x044A, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::CloseBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x042A, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Tilde::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< ロシア語 (キリル) のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeMapSuite, RussianCyrillicTest2)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::RussianCyrillic;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);
}

//!< ロシア語 (キリル) のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeMapSuite, RussianCyrillicTest3)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::RussianCyrillic;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Hiragana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0444, keyCode);
}

//!< ロシア語 (キリル) のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeMapSuite, RussianCyrillicTest4)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::RussianCyrillic;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Katakana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0444, keyCode);
}

//!< ロシア語 (キリル) のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, RussianCyrillicTest5)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::RussianCyrillic;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0424, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Slash::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x002E, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Slash::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x002C, keyCode);
}

//!< ロシア語 (キリル) のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, RussianCyrillicTest6)
{
    auto keyCode = uint16_t();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::RussianCyrillic;

    keyCode = uint16_t();
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);
}

//!< ロシア語 (ラテン) のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, RussianLatinTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::RussianLatin;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Backslash::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D3::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0023, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< ロシア語 (ラテン) のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeMapSuite, RussianLatinTest2)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::RussianLatin;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);
}

//!< ロシア語 (ラテン) のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeMapSuite, RussianLatinTest3)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::RussianLatin;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Hiragana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< ロシア語 (ラテン) のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeMapSuite, RussianLatinTest4)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::RussianLatin;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Katakana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< ロシア語 (ラテン) のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, RussianLatinTest5)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::RussianLatin;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0041, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Slash::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x002F, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Slash::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003F, keyCode);
}

//!< ロシア語 (ラテン) のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, RussianLatinTest6)
{
    auto keyCode = uint16_t();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::RussianLatin;

    keyCode = uint16_t();
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);
}

//!< スペイン語のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, SpanishTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Spanish;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Semicolon::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00F1, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Semicolon::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00D1, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00E7, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< スペイン語のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeMapSuite, SpanishTest2)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Spanish;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005B, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< スペイン語のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeMapSuite, SpanishTest3)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Spanish;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Hiragana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< スペイン語のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeMapSuite, SpanishTest4)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Spanish;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Katakana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< スペイン語のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, SpanishTest5)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Spanish;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0045, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x20AC, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Plus::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00A1, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Plus::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x00BF, keyCode);
}

//!< スペイン語のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, SpanishTest6)
{
    auto keyCode = uint16_t();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::Spanish;

    keyCode = uint16_t();
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);
}

//!< スペイン語 (ラテンアメリカ) のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, SpanishLatinAmericaTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::SpanishLatinAmerica;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Q::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0071, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Q::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0051, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x007D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< スペイン語 (ラテンアメリカ) のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeMapSuite, SpanishLatinAmericaTest2)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::SpanishLatinAmerica;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Minus::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Minus::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< スペイン語 (ラテンアメリカ) のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeMapSuite, SpanishLatinAmericaTest3)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::SpanishLatinAmerica;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Hiragana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< スペイン語 (ラテンアメリカ) のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeMapSuite, SpanishLatinAmericaTest4)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::SpanishLatinAmerica;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Katakana>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);
}

//!< スペイン語 (ラテンアメリカ) のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, SpanishLatinAmericaTest5)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::SpanishLatinAmerica;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Q::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0051, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Q::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0040, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::CloseBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x002B, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::CloseBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x002A, keyCode);
}

//!< スペイン語 (ラテンアメリカ) のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeMapSuite, SpanishLatinAmericaTest6)
{
    auto keyCode = uint16_t();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::SpanishLatinAmerica;

    keyCode = uint16_t();
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0035, keyCode);

    keyCode = uint16_t();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0020, keyCode);
}

//!< 韓国語 (ハングル) のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, KoreanHangulTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::KoreanHangul;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::O::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x3150, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::O::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x3152, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Tilde::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< 韓国語 (ラテン) のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, KoreanLatinTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::KoreanLatin;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::O::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x006F, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::O::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x004F, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Tilde::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< 簡体字中国語のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, ChineseSimplifiedTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::ChineseSimplified;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Backquote::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0060, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D2::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0040, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Tilde::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< 繁体字中国語 (注音) のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, ChineseTraditionalZhuyinTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::ChineseTraditionalZhuyin;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D3::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x02C7, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::D3::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0023, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Tilde::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< 繁体字中国語 (倉頡) のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, ChineseTraditionalCangjieTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::ChineseTraditionalCangjie;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x6C34, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0045, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Tilde::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}

//!< 繁体字中国語 (ラテン) のレイアウトが正しく処理されるか
TEST(KeyCodeMapSuite, ChineseTraditionalLatinTest1)
{
    auto keyCode = uint16_t();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();
    auto layout = ::nn::hid::KeyboardLayout::ChineseTraditionalLatin;

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0061, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0041, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::Tilde::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x005C, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x003D, keyCode);

    keyCode = uint16_t();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::hid::GetKeyCode(&keyCode, usageId, modifiers, layout);
    EXPECT_EQ(0x0000, keyCode);
}
