﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/hid/hid_KeyCode.h>
#include <nnt/nntest.h>

//!< キーコードの組み立てモードは初期化直後に全て off になっているか
TEST(KeyCodeComposerSuite, ModeTest1)
{
    ::nn::hid::KeyCodeComposer composer;
    EXPECT_FALSE(composer.GetDeadKeyMode());
    EXPECT_FALSE(composer.GetRomajiHiraganaMode());
    EXPECT_FALSE(composer.GetRomajiKatakanaMode());
}

//!< キーコードの組み立てモードを off にできるか
TEST(KeyCodeComposerSuite, ModeTest2)
{
    ::nn::hid::KeyCodeComposer composer;

    composer.SetDeadKeyMode(true);
    EXPECT_TRUE(composer.GetDeadKeyMode());
    composer.SetDeadKeyMode(false);
    EXPECT_FALSE(composer.GetDeadKeyMode());

    composer.SetRomajiHiraganaMode(true);
    EXPECT_TRUE(composer.GetRomajiHiraganaMode());
    composer.SetRomajiHiraganaMode(false);
    EXPECT_FALSE(composer.GetRomajiHiraganaMode());

    composer.SetRomajiKatakanaMode(true);
    EXPECT_TRUE(composer.GetRomajiKatakanaMode());
    composer.SetRomajiKatakanaMode(false);
    EXPECT_FALSE(composer.GetRomajiKatakanaMode());
}

//!< キーコードの組み立てモードの切り替えは排他状態を維持するか
TEST(KeyCodeComposerSuite, ModeTest3)
{
    ::nn::hid::KeyCodeComposer composer;

    composer.SetDeadKeyMode(true);
    EXPECT_TRUE(composer.GetDeadKeyMode());
    EXPECT_FALSE(composer.GetRomajiHiraganaMode());
    EXPECT_FALSE(composer.GetRomajiKatakanaMode());

    composer.SetRomajiHiraganaMode(true);
    EXPECT_FALSE(composer.GetDeadKeyMode());
    EXPECT_TRUE(composer.GetRomajiHiraganaMode());
    EXPECT_FALSE(composer.GetRomajiKatakanaMode());

    composer.SetRomajiKatakanaMode(true);
    EXPECT_FALSE(composer.GetDeadKeyMode());
    EXPECT_FALSE(composer.GetRomajiHiraganaMode());
    EXPECT_TRUE(composer.GetRomajiKatakanaMode());

    composer.SetDeadKeyMode(true);
    EXPECT_TRUE(composer.GetDeadKeyMode());
    EXPECT_FALSE(composer.GetRomajiHiraganaMode());
    EXPECT_FALSE(composer.GetRomajiKatakanaMode());
}

//!< キーコードの組み立てモードの切り替えは組み立て中キーコード列を破棄するか
TEST(KeyCodeComposerSuite, ModeTest4)
{
    const char* modeNames[] = { "DeadKey", "RomajiHiragana", "RomajiKatakana" };

    const auto modeCount = sizeof(modeNames) / sizeof(modeNames[0]);

    const auto SetMode = [](::nn::hid::KeyCodeComposer* pComposer,
                            int mode, bool enables) NN_NOEXCEPT
    {
        switch (mode)
        {
        case 0:
            pComposer->SetDeadKeyMode(enables);
            break;
        case 1:
            pComposer->SetRomajiHiraganaMode(enables);
            break;
        case 2:
            pComposer->SetRomajiKatakanaMode(enables);
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    };

    ::nn::hid::KeyCodeComposer composer;

    auto keyCode = uint16_t();

    for (const char* (&name) : modeNames)
    {
        SCOPED_TRACE(::testing::Message() << "Mode: " << name);

        const auto mode = static_cast<int>(&name - modeNames);

        keyCode = static_cast<uint16_t>('a');
        composer.EnqueueKeyCodes(&keyCode, 1);
        EXPECT_EQ(1, composer.GetCompositionCount());

        composer.SetDeadKeyMode(false);
        composer.SetRomajiHiraganaMode(false);
        composer.SetRomajiKatakanaMode(false);
        EXPECT_EQ(1, composer.GetCompositionCount());

        SetMode(&composer, mode, true);
        EXPECT_EQ(0, composer.GetCompositionCount());

        keyCode = static_cast<uint16_t>('a');
        composer.EnqueueKeyCodes(&keyCode, 1);
        EXPECT_EQ(1, composer.GetCompositionCount());

        SetMode(&composer, mode, true);
        EXPECT_EQ(1, composer.GetCompositionCount());

        SetMode(&composer, (mode + 1) % modeCount, true);
        EXPECT_EQ(0, composer.GetCompositionCount());

        keyCode = static_cast<uint16_t>('a');
        composer.EnqueueKeyCodes(&keyCode, 1);
        EXPECT_EQ(1, composer.GetCompositionCount());

        SetMode(&composer, (mode + 1) % modeCount, false);
        EXPECT_EQ(0, composer.GetCompositionCount());
    }
}

//!< 組み立て中キーコード列は初期化直後に空になっているか
TEST(KeyCodeComposerSuite, CompositionTest1)
{
    ::nn::hid::KeyCodeComposer composer;
    EXPECT_EQ(0, composer.GetCompositionCount());
    EXPECT_EQ(0, composer.GetOutputCount());
}

//!< 組み立て中キーコード列を正しく取得できるか
TEST(KeyCodeComposerSuite, CompositionTest2)
{
    ::nn::hid::KeyCodeComposer composer;
    composer.SetRomajiHiraganaMode(true);

    auto keyCode = uint16_t();

    const auto bufferSize = 8;
    uint16_t buffer[bufferSize] = {};

    keyCode = static_cast<uint16_t>('n');
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));
    EXPECT_EQ(0, composer.GetOutputCount());
    EXPECT_EQ(1, composer.GetCompositionCount());
    EXPECT_EQ(1, composer.GetComposition(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>('n'), buffer[0]);

    keyCode = static_cast<uint16_t>('k');
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));
    EXPECT_EQ(1, composer.GetOutputCount());
    EXPECT_EQ(2, composer.GetCompositionCount());
    EXPECT_EQ(2, composer.GetComposition(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>(0x3093), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>('k'), buffer[1]);

    EXPECT_EQ(1, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>(0x3093), buffer[0]);
    EXPECT_EQ(0, composer.GetOutputCount());
    EXPECT_EQ(1, composer.GetCompositionCount());
    EXPECT_EQ(1, composer.GetComposition(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>('k'), buffer[0]);

    keyCode = static_cast<uint16_t>('a');
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));
    EXPECT_EQ(1, composer.GetOutputCount());
    EXPECT_EQ(1, composer.GetCompositionCount());
    EXPECT_EQ(1, composer.GetComposition(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>(0x304B), buffer[0]);

    EXPECT_EQ(1, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>(0x304B), buffer[0]);
    EXPECT_EQ(0, composer.GetOutputCount());
    EXPECT_EQ(0, composer.GetCompositionCount());

    keyCode = static_cast<uint16_t>('k');
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));
    EXPECT_EQ(0, composer.GetOutputCount());
    EXPECT_EQ(1, composer.GetCompositionCount());
    EXPECT_EQ(1, composer.GetComposition(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>('k'), buffer[0]);
}

//!< 組み立て中キーコード列が最大長付近にある際に正しく取得できるか
TEST(KeyCodeComposerSuite, CompositionTest3)
{
    ::nn::hid::KeyCodeComposer composer;
    composer.SetRomajiHiraganaMode(true);

    const uint16_t keyCodes[] = { 'l', 'l', 't' };
    EXPECT_EQ(3, composer.EnqueueKeyCodes(keyCodes, 3));

    const int altCodes[] = { 4, 8 };
    EXPECT_EQ(2, composer.EnqueueAltCodes(altCodes, 2));

    const auto keyCode = static_cast<uint16_t>('s');
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));

    const auto bufferSize = 8;
    uint16_t buffer[bufferSize] = {};
    EXPECT_EQ(2, composer.GetOutputCount());
    EXPECT_EQ(5, composer.GetCompositionCount());
    EXPECT_EQ(5, composer.GetComposition(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>('0'), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>('l'), buffer[1]);
    EXPECT_EQ(static_cast<uint16_t>('l'), buffer[2]);
    EXPECT_EQ(static_cast<uint16_t>('t'), buffer[3]);
    EXPECT_EQ(static_cast<uint16_t>('s'), buffer[4]);

    EXPECT_EQ(2, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>('0'), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>('l'), buffer[1]);
    EXPECT_EQ(0, composer.GetOutputCount());
    EXPECT_EQ(3, composer.GetCompositionCount());
    EXPECT_EQ(3, composer.GetComposition(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>('l'), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>('t'), buffer[1]);
    EXPECT_EQ(static_cast<uint16_t>('s'), buffer[2]);
}

//!< キーコード列の追加が正しく処理されるか
TEST(KeyCodeComposerSuite, EnqueueKeyCodesTest1)
{
    ::nn::hid::KeyCodeComposer composer;
    composer.SetRomajiKatakanaMode(true);

    const uint16_t keyCodes[] = { 'n', 'd', 'a' };
    EXPECT_EQ(2, composer.EnqueueKeyCodes(keyCodes, 3));
    EXPECT_EQ(1, composer.GetOutputCount());

    EXPECT_EQ(0, composer.EnqueueKeyCodes(&keyCodes[2], 1));

    auto keyCode = uint16_t();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x30F3), keyCode);
    EXPECT_EQ(0, composer.GetOutputCount());

    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCodes[2], 1));

    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x30C0), keyCode);
    EXPECT_EQ(0, composer.GetOutputCount());
}

//!< 組み立てモードが未設定の場合にキーコードが正しく処理されるか
TEST(KeyCodeComposerSuite, DefaultModeTest1)
{
    ::nn::hid::KeyCodeComposer composer;

    auto keyCode = uint16_t();

    keyCode = static_cast<uint16_t>(0x0300);
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));
    EXPECT_EQ(1, composer.GetOutputCount());
    keyCode = uint16_t();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x0300), keyCode);

    keyCode = static_cast<uint16_t>('k');
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));
    EXPECT_EQ(1, composer.GetOutputCount());
    keyCode = uint16_t();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>('k'), keyCode);
}

//!< デッドキーが正しく処理されるか
TEST(KeyCodeComposerSuite, DeadKeyTest1)
{
    ::nn::hid::KeyCodeComposer composer;
    composer.SetDeadKeyMode(true);

    auto keyCode = uint16_t();

    keyCode = static_cast<uint16_t>(0x0300);
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));
    EXPECT_EQ(0, composer.GetOutputCount());

    const auto bufferSize = 8;
    uint16_t buffer[bufferSize] = {};

    keyCode = static_cast<uint16_t>(0x0301);
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));
    EXPECT_EQ(2, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>(0x0060), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>(0x00B4), buffer[1]);

    keyCode = static_cast<uint16_t>(0x0302);
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));
    EXPECT_EQ(0, composer.GetOutputCount());

    keyCode = static_cast<uint16_t>('k');
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));
    EXPECT_EQ(2, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>(0x005E), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>('k'), buffer[1]);

    keyCode = static_cast<uint16_t>(0x0303);
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));
    EXPECT_EQ(0, composer.GetOutputCount());

    keyCode = static_cast<uint16_t>('n');
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));
    EXPECT_EQ(1, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>(0x00F1), buffer[0]);

    keyCode = static_cast<uint16_t>('0');
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));
    EXPECT_EQ(1, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>('0'), buffer[0]);
}

//!< ローマ字変換（ひらがな）が正しく処理されるか
TEST(KeyCodeComposerSuite, RomajiHiraganaTest1)
{
    ::nn::hid::KeyCodeComposer composer;
    composer.SetRomajiHiraganaMode(true);

    const auto bufferSize = 8;
    uint16_t buffer[bufferSize] = {};

    const uint16_t keyCodes1[] = { 'n', '0' };
    EXPECT_EQ(2, composer.EnqueueKeyCodes(keyCodes1, 2));
    EXPECT_EQ(2, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>(0x3093), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>('0'), buffer[1]);

    const uint16_t keyCodes2[] = { 'n', 'l' };
    EXPECT_EQ(2, composer.EnqueueKeyCodes(keyCodes2, 2));
    EXPECT_EQ(1, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>(0x3093), buffer[0]);

    const uint16_t keyCodes3[] = { 'l', 't', 's' };
    EXPECT_EQ(3, composer.EnqueueKeyCodes(keyCodes3, 3));
    EXPECT_EQ(1, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>('l'), buffer[0]);

    const uint16_t keyCodes4[] = { 'a' };
    EXPECT_EQ(1, composer.EnqueueKeyCodes(keyCodes4, 1));
    EXPECT_EQ(2, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>(0x3063), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>(0x3041), buffer[1]);

    const uint16_t keyCodes5[] = { 'w', 'w', 'h', 'a' };
    EXPECT_EQ(4, composer.EnqueueKeyCodes(keyCodes5, 4));
    EXPECT_EQ(3, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>(0x3063), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>(0x3046), buffer[1]);
    EXPECT_EQ(static_cast<uint16_t>(0x3041), buffer[2]);
}

//!< ローマ字変換（カタカナ）が正しく処理されるか
TEST(KeyCodeComposerSuite, RomajiKatakanaTest1)
{
    ::nn::hid::KeyCodeComposer composer;
    composer.SetRomajiKatakanaMode(true);

    const auto bufferSize = 8;
    uint16_t buffer[bufferSize] = {};

    const uint16_t keyCodes1[] = { 'n', '!' };
    EXPECT_EQ(2, composer.EnqueueKeyCodes(keyCodes1, 2));
    EXPECT_EQ(2, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>(0x30F3), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>('!'), buffer[1]);

    const uint16_t keyCodes2[] = { 'n', 'x' };
    EXPECT_EQ(2, composer.EnqueueKeyCodes(keyCodes2, 2));
    EXPECT_EQ(1, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>(0x30F3), buffer[0]);

    const uint16_t keyCodes3[] = { 'x', 'w', 'h' };
    EXPECT_EQ(3, composer.EnqueueKeyCodes(keyCodes3, 3));
    EXPECT_EQ(1, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>('x'), buffer[0]);

    const uint16_t keyCodes4[] = { 'a' };
    EXPECT_EQ(1, composer.EnqueueKeyCodes(keyCodes4, 1));
    EXPECT_EQ(2, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>(0x30A5), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>(0x30A1), buffer[1]);

    const uint16_t keyCodes5[] = { 'c', 'c', 'y', 'a' };
    EXPECT_EQ(4, composer.EnqueueKeyCodes(keyCodes5, 4));
    EXPECT_EQ(3, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>(0x30C3), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>(0x30C1), buffer[1]);
    EXPECT_EQ(static_cast<uint16_t>(0x30E3), buffer[2]);
}

//!< キーコード列の削除が正しく処理されるか
TEST(KeyCodeComposerSuite, RemoveKeyCodesTest1)
{
    ::nn::hid::KeyCodeComposer composer;
    composer.SetRomajiKatakanaMode(true);

    const auto bufferSize = 8;
    uint16_t buffer[bufferSize] = {};

    EXPECT_EQ(0, composer.RemoveKeyCodes(buffer, bufferSize));

    const uint16_t keyCodes1[] = { 'n', 'y' };
    EXPECT_EQ(2, composer.EnqueueKeyCodes(keyCodes1, 2));
    EXPECT_EQ(2, composer.RemoveKeyCodes(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>('n'), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>('y'), buffer[1]);
    EXPECT_EQ(0, composer.GetCompositionCount());

    const uint16_t keyCodes2[] = { 'l', 'l', 'w', 'h' };
    EXPECT_EQ(4, composer.EnqueueKeyCodes(keyCodes2, 4));
    EXPECT_EQ(1, composer.RemoveKeyCodes(buffer, 1));
    EXPECT_EQ(static_cast<uint16_t>('h'), buffer[0]);
    EXPECT_EQ(1, composer.GetOutputCount());
    EXPECT_EQ(3, composer.GetCompositionCount());

    EXPECT_EQ(2, composer.RemoveKeyCodes(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>('l'), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>('w'), buffer[1]);
    EXPECT_EQ(1, composer.GetOutputCount());
    EXPECT_EQ(1, composer.GetCompositionCount());

    EXPECT_EQ(1, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>('l'), buffer[0]);
}

//!< キーコード列の明示的な確定が正しく処理されるか
TEST(KeyCodeComposerSuite, FlushKeyCodesTest1)
{
    ::nn::hid::KeyCodeComposer composer;

    auto keyCode = uint16_t();

    composer.SetDeadKeyMode(true);
    composer.FlushKeyCodes();
    keyCode = static_cast<uint16_t>(0x0308);
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));
    composer.FlushKeyCodes();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x00A8), keyCode);

    composer.SetRomajiHiraganaMode(true);
    composer.FlushKeyCodes();
    keyCode = static_cast<uint16_t>('n');
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));
    composer.FlushKeyCodes();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x3093), keyCode);

    composer.SetRomajiKatakanaMode(true);
    composer.FlushKeyCodes();
    keyCode = static_cast<uint16_t>('n');
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));
    composer.FlushKeyCodes();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x30F3), keyCode);

    const uint16_t keyCodes1[] = { 'x', 'x', 't', 's' };
    EXPECT_EQ(4, composer.EnqueueKeyCodes(keyCodes1, 4));
    composer.FlushKeyCodes();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>('x'), keyCode);
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>('x'), keyCode);
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>('t'), keyCode);
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>('s'), keyCode);
}

//!< Alt コードの追加が正しく処理されるか
TEST(KeyCodeComposerSuite, EnqueueAltCodesTest1)
{
    ::nn::hid::KeyCodeComposer composer;
    composer.SetRomajiKatakanaMode(true);

    auto keyCode = uint16_t();

    keyCode = static_cast<uint16_t>('e');
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));

    const int altCodes[] = { 1, 2, 1 };
    EXPECT_EQ(0, composer.EnqueueAltCodes(altCodes, 3));

    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x30A8), keyCode);

    keyCode = static_cast<uint16_t>('k');
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));

    EXPECT_EQ(3, composer.EnqueueAltCodes(altCodes, 3));

    EXPECT_EQ(1, composer.GetCompositionCount());

    keyCode = static_cast<uint16_t>('u');
    EXPECT_EQ(1, composer.EnqueueKeyCodes(&keyCode, 1));

    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>('y'), keyCode);

    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x30AF), keyCode);
}

//!< Alt コードが正しく処理されるか
TEST(KeyCodeComposerSuite, AltCodeTest1)
{
    ::nn::hid::KeyCodeComposer composer;

    auto keyCode = uint16_t();

    const int altCodes1[] = { 1, 2, 6 };
    EXPECT_EQ(3, composer.EnqueueAltCodes(altCodes1, 3));
    composer.FlushAltCodes();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>('~'), keyCode);

    const int altCodes2[] = { 2, 5, 6 };
    EXPECT_EQ(3, composer.EnqueueAltCodes(altCodes2, 3));
    composer.FlushKeyCodes();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x0000), keyCode);

    const int altCodes3[] = { 2, 8, 9 };
    EXPECT_EQ(3, composer.EnqueueAltCodes(altCodes3, 3));
    composer.FlushAltCodes();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>('!'), keyCode);

    const int altCodes4[] = { 1, 0, 7, 3, 7, 4, 1, 8, 3 };
    EXPECT_EQ(9, composer.EnqueueAltCodes(altCodes4, 9));
    composer.FlushKeyCodes();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(' '), keyCode);

    const int altCodes5[] = { 1, 2, 8 };
    EXPECT_EQ(3, composer.EnqueueAltCodes(altCodes5, 3));
    composer.FlushAltCodes();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x00C7), keyCode);

    const int altCodes6[] = { 0, 1, 2, 8 };
    EXPECT_EQ(4, composer.EnqueueAltCodes(altCodes6, 4));
    composer.FlushKeyCodes();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x20AC), keyCode);

    const int altCodes7[] = { 0, 1, 5, 9 };
    EXPECT_EQ(4, composer.EnqueueAltCodes(altCodes7, 4));
    composer.FlushAltCodes();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x0178), keyCode);

    const int altCodes8[] = { 1, 5, 9 };
    EXPECT_EQ(3, composer.EnqueueAltCodes(altCodes8, 3));
    composer.FlushKeyCodes();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x0192), keyCode);

    const int altCodes9[] = { 0, 1, 6, 0 };
    EXPECT_EQ(4, composer.EnqueueAltCodes(altCodes9, 4));
    composer.FlushAltCodes();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x00A0), keyCode);

    const int altCodes10[] = { 2, 5, 5 };
    EXPECT_EQ(3, composer.EnqueueAltCodes(altCodes10, 3));
    composer.FlushKeyCodes();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x00A0), keyCode);

    const int altCodes11[] = { 0, 2, 5, 5 };
    EXPECT_EQ(4, composer.EnqueueAltCodes(altCodes11, 4));
    composer.FlushAltCodes();
    EXPECT_EQ(1, composer.DequeueOutputs(&keyCode, 1));
    EXPECT_EQ(static_cast<uint16_t>(0x00FF), keyCode);
}

//!< キーコード列の破棄が正しく処理されるか
TEST(KeyCodeComposerSuite, ClearTest1)
{
    ::nn::hid::KeyCodeComposer composer;
    composer.SetRomajiHiraganaMode(true);

    const uint16_t keyCodes1[] = { 'g' };
    EXPECT_EQ(1, composer.EnqueueKeyCodes(keyCodes1, 1));

    const int altCodes1[] = { 0, 1, 3, 0 };
    EXPECT_EQ(4, composer.EnqueueAltCodes(altCodes1, 4));

    composer.Clear();
    EXPECT_EQ(0, composer.GetCompositionCount());
    EXPECT_EQ(0, composer.GetOutputCount());

    const uint16_t keyCodes2[] = { 'm' };
    EXPECT_EQ(1, composer.EnqueueKeyCodes(keyCodes2, 1));
    EXPECT_EQ(1, composer.GetCompositionCount());
    EXPECT_EQ(0, composer.GetOutputCount());

    const int altCodes2[] = { 1, 3, 0 };
    EXPECT_EQ(3, composer.EnqueueAltCodes(altCodes2, 3));

    const uint16_t keyCodes3[] = { 'i' };
    EXPECT_EQ(1, composer.EnqueueKeyCodes(keyCodes3, 1));
    EXPECT_EQ(2, composer.GetCompositionCount());
    EXPECT_EQ(2, composer.GetOutputCount());

    const auto bufferSize = 8;
    uint16_t buffer[bufferSize] = {};
    EXPECT_EQ(2, composer.DequeueOutputs(buffer, bufferSize));
    EXPECT_EQ(static_cast<uint16_t>(0x00E9), buffer[0]);
    EXPECT_EQ(static_cast<uint16_t>(0x307F), buffer[1]);
}
