﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/debug/hid_HomeButton.h>
#include <nn/hid/system/hid_HomeButton.h>
#include <nn/os.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nntest.h>

#include "../Common/testHid_HomeButton.h"

namespace
{

::nn::hid::system::HomeButtonState
    g_States1[::nn::hid::system::HomeButtonStateCountMax];
::nn::hid::system::HomeButtonState
    g_States2[::nn::hid::system::HomeButtonStateCountMax];

//!< イベントのシグナル状況を確認するか否かを表す値です。
const bool EnablesSignalAsserion =
#if defined(NN_BUILD_CONFIG_OS_WIN)
    true;
#else
    false;
#endif

//!< ホームボタンの自動操作状態をファジングします。
void FuzzHomeButtonAutoPilotState(
    ::nn::hid::debug::HomeButtonAutoPilotState* pState, int seed) NN_NOEXCEPT;

//!< ホームボタンの入力状態が入力無し状態であることを期待します。
void ExpectDefaultHomeButtonState(
    const ::nn::hid::system::HomeButtonState& state) NN_NOEXCEPT;

//!< ホームボタンの入力状態がファジングされた入力状態であることを期待します。
void ExpectFuzzedHomeButtonState(
    const ::nn::hid::system::HomeButtonState& state) NN_NOEXCEPT;

//!< ホームボタンの入力状態が同じ自動操作の状態から決定されたことを期待します。
void ExpectSameOriginHomeButtonStates(
    const ::nn::hid::system::HomeButtonState& lhs,
    const ::nn::hid::system::HomeButtonState& rhs) NN_NOEXCEPT;

//!< ホームボタンの入力状態が隣り合っていることを期待します。
void ExpectAdjacentHomeButtonStates(
    const ::nn::hid::system::HomeButtonState& lhs,
    const ::nn::hid::system::HomeButtonState& rhs) NN_NOEXCEPT;

} // namespace

//!< HomeButton の初期化処理は自動操作状態を正しく管理するか
TEST(HomeButtonAutoPilotSuite, InitializationTest1)
{
    ::nn::os::SystemEventType event = {};

    ::nn::hid::system::BindHomeButtonEvent(
        &event, ::nn::os::EventClearMode_AutoClear);

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&event));
    }

    ::nn::hid::debug::HomeButtonAutoPilotState autoPilotState;
    FuzzHomeButtonAutoPilotState(&autoPilotState, 1);

    ::nn::hid::system::InitializeHomeButton();

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_FALSE(::nn::os::TryWaitSystemEvent(&event));
    }

    ::nn::hid::debug::SetHomeButtonAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::HomeButtonStateCountMax / 2 *
            ::nnt::hid::GetHomeButtonSamplingInterval().GetMilliSeconds()));

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&event));
    }

    // 自動操作が反映された入力状態が取得される。
    ::nn::hid::system::GetHomeButtonStates(&g_States1[0], 1);
    {
        SCOPED_TRACE("");
        ExpectFuzzedHomeButtonState(g_States1[0]);
    }

    ::nn::hid::system::InitializeHomeButton();

    ::nn::hid::debug::FinalizeHomeButton();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::HomeButtonStateCountMax / 2 *
            ::nnt::hid::GetHomeButtonSamplingInterval().GetMilliSeconds()));

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_FALSE(::nn::os::TryWaitSystemEvent(&event));
    }

    // まだユーザが存在するので自動操作設定は解除されない。
    ::nn::hid::system::GetHomeButtonStates(&g_States1[0], 1);
    {
        SCOPED_TRACE("");
        ExpectFuzzedHomeButtonState(g_States1[0]);
    }

    ::nnt::hid::FinalizeHomeButtonForAutoPilot();

    ::nn::hid::system::InitializeHomeButton();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::HomeButtonStateCountMax / 2 *
            ::nnt::hid::GetHomeButtonSamplingInterval().GetMilliSeconds()));

    // ホームボタンが完全に開放されたことで自動操作設定が解除される。
    ::nn::hid::system::GetHomeButtonStates(&g_States1[0], 1);
    {
        SCOPED_TRACE("");
        ExpectDefaultHomeButtonState(g_States1[0]);
    }

    ::nnt::hid::FinalizeHomeButtonForAutoPilot();

    ::nn::os::DestroySystemEvent(&event);
}

//!< ホームボタンの入力状態は不正な自動操作状態が設定されても仕様を保証するか
TEST(HomeButtonAutoPilotSuite, InvalidArgumentTest1)
{
    ::nn::hid::debug::HomeButtonAutoPilotState autoPilotState = {};

    // 無効なビットフラグも設定
    autoPilotState.buttons.Set( 0, true);
    autoPilotState.buttons.Set( 8, true);
    autoPilotState.buttons.Set(16, true);
    autoPilotState.buttons.Set(24, true);

    ::nn::hid::system::InitializeHomeButton();

    ::nn::hid::debug::SetHomeButtonAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::HomeButtonStateCountMax / 2 *
            ::nnt::hid::GetHomeButtonSamplingInterval().GetMilliSeconds()));

    ::nn::hid::system::HomeButtonState& state = g_States1[0];
    ::nn::hid::system::GetHomeButtonStates(&state, 1);

    ::nnt::hid::FinalizeHomeButtonForAutoPilot();

    // 有効なビットフラグのみ結果に反映
    EXPECT_TRUE (state.buttons.Test( 0));
    EXPECT_FALSE(state.buttons.Test( 8));
    EXPECT_FALSE(state.buttons.Test(16));
    EXPECT_FALSE(state.buttons.Test(24));
}

//!< ホームボタンの入力状態は自動操作状態の設定を正しく反映するか
TEST(HomeButtonAutoPilotSuite, StateReadingTest1)
{
    ::nn::os::SystemEventType event = {};

    ::nn::hid::system::BindHomeButtonEvent(&event,
                                           ::nn::os::EventClearMode_AutoClear);

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&event));
    }

    ::nn::hid::system::InitializeHomeButton();

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_FALSE(::nn::os::TryWaitSystemEvent(&event));
    }

    ::nn::hid::system::GetHomeButtonStates(&g_States1[0], 1);
    {
        SCOPED_TRACE("");
        ExpectDefaultHomeButtonState(g_States1[0]);
    }

    ::nn::hid::debug::HomeButtonAutoPilotState autoPilotState;

    for (int i = 0; i <= ::nn::hid::system::HomeButtonStateCountMax / 3; ++i)
    {
        FuzzHomeButtonAutoPilotState(&autoPilotState, i + 1);
        ::nn::hid::debug::SetHomeButtonAutoPilotState(autoPilotState);
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                2 * ::nnt::hid::GetHomeButtonSamplingInterval()
                        .GetMilliSeconds()));
        if (NN_STATIC_CONDITION(EnablesSignalAsserion))
        {
            EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&event));
        }
    }

    int count = ::nn::hid::system::GetHomeButtonStates(
        g_States1, ::nn::hid::system::HomeButtonStateCountMax);
    EXPECT_LE(1, count);

    // 自動操作状態に応じてホームボタンの入力状態は変化する。
    for (int i = 0; i < count; ++i)
    {
        const ::nn::hid::system::HomeButtonState& lhs = g_States1[i];

        EXPECT_LE(0, lhs.samplingNumber);

        if (i + 1 < count)
        {
            SCOPED_TRACE("");
            ExpectAdjacentHomeButtonStates(lhs, g_States1[i + 1]);
        }
    }

    ::nnt::hid::FinalizeHomeButtonForAutoPilot();

    ::nn::os::DestroySystemEvent(&event);
}

//!< ホームボタンの入力状態は自動操作状態の解除を正しく反映するか
TEST(HomeButtonAutoPilotSuite, StateReadingTest2)
{
    ::nn::os::SystemEventType event = {};

    ::nn::hid::system::BindHomeButtonEvent(&event,
                                           ::nn::os::EventClearMode_AutoClear);

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&event));
    }

    ::nn::hid::system::InitializeHomeButton();

    ::nn::hid::debug::HomeButtonAutoPilotState autoPilotState;
    FuzzHomeButtonAutoPilotState(&autoPilotState, 1);

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_FALSE(::nn::os::TryWaitSystemEvent(&event));
    }

    ::nn::hid::debug::SetHomeButtonAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::HomeButtonStateCountMax *
            ::nnt::hid::GetHomeButtonSamplingInterval().GetMilliSeconds()));

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&event));
    }

    // 自動操作状態を解除
    ::nn::hid::debug::UnsetHomeButtonAutoPilotState();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::HomeButtonStateCountMax / 2 *
            ::nnt::hid::GetHomeButtonSamplingInterval().GetMilliSeconds()));

    if (NN_STATIC_CONDITION(EnablesSignalAsserion))
    {
        EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&event));
    }

    int count = ::nn::hid::system::GetHomeButtonStates(
        g_States1, ::nn::hid::system::HomeButtonStateCountMax);
    EXPECT_EQ(::nn::hid::system::HomeButtonStateCountMax, count);

    bool hasChanged = false;

    ::nn::hid::system::HomeButtonState state = {};

    // 入力状態列状で自動操作状態は連続的に切り替わる。
    for (int i = 0; i < count; ++i)
    {
        if (i + 1 < count)
        {
            const ::nn::hid::system::HomeButtonState& lhs = g_States1[i];
            const ::nn::hid::system::HomeButtonState& rhs = g_States1[i + 1];
            SCOPED_TRACE("");
            EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
        }

        if (g_States1[i].buttons != state.buttons)
        {
            EXPECT_FALSE(hasChanged);
            hasChanged = true;
            state.buttons = autoPilotState.buttons;
        }

        {
            SCOPED_TRACE("");
            ExpectSameOriginHomeButtonStates(state, g_States1[i]);
        }
    }

    EXPECT_TRUE(hasChanged);

    ::nnt::hid::FinalizeHomeButtonForAutoPilot();

    ::nn::os::DestroySystemEvent(&event);
}

//!< LIFO の境界を跨いで自動操作状態の設定を正しく反映するか
TEST(HomeButtonAutoPilotSuite, StateReadingTest3)
{
    ::nn::hid::system::InitializeHomeButton();

    ::nn::hid::debug::HomeButtonAutoPilotState autoPilotState;

    int lastCount = 0;

    // LIFO の境界を 2 回跨ぐまでサンプリングを継続
    for (int i = 0; i < ::nn::hid::system::HomeButtonStateCountMax * 3; ++i)
    {
        FuzzHomeButtonAutoPilotState(&autoPilotState, i);
        ::nn::hid::debug::SetHomeButtonAutoPilotState(autoPilotState);
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                ::nnt::hid::GetHomeButtonSamplingInterval().GetMilliSeconds()));

        // 今回のサンプリング結果を格納するバッファ
        ::nn::hid::system::HomeButtonState *ptr1 = (i % 2 == 1) ? &g_States1[0]
                                                                : &g_States2[0];

        // 前回のサンプリング結果が格納されているバッファ
        ::nn::hid::system::HomeButtonState *ptr2 = (i % 2 == 1) ? &g_States2[0]
                                                                : &g_States1[0];

        // 取得される入力状態の数は単調増加する。
        int count = ::nn::hid::system::GetHomeButtonStates(
            ptr1, ::nn::hid::system::HomeButtonStateCountMax);
        EXPECT_LE(lastCount, count);

        for (int j = 0; j < count; ++j)
        {
            if (j + 1 < count)
            {
                SCOPED_TRACE("");
                ExpectAdjacentHomeButtonStates(ptr1[j], ptr1[j + 1]);
            }

            // 前回のサンプリング結果と重複する箇所には同じ値が入る。
            if (lastCount > 0 &&
                ptr1[j].samplingNumber == ptr2[0].samplingNumber)
            {
                for (int k = 0; k < lastCount - j; ++k)
                {
                    SCOPED_TRACE("");
                    EXPECT_EQ(ptr2[k].samplingNumber,
                              ptr1[j + k].samplingNumber);
                    ExpectSameOriginHomeButtonStates(ptr2[k], ptr1[j + k]);
                }

                break;
            }
        }

        lastCount = count;
    }

    EXPECT_EQ(::nn::hid::system::HomeButtonStateCountMax, lastCount);

    ::nnt::hid::FinalizeHomeButtonForAutoPilot();
}

namespace
{

void FuzzHomeButtonAutoPilotState(
    ::nn::hid::debug::HomeButtonAutoPilotState* pState, int seed) NN_NOEXCEPT
{
    pState->buttons.Reset();
    pState->buttons.Set<::nn::hid::system::HomeButton::Active>(seed % 2 == 1);
}

void ExpectDefaultHomeButtonState(
    const ::nn::hid::system::HomeButtonState& state) NN_NOEXCEPT
{
    EXPECT_LE(0, state.samplingNumber);
    EXPECT_TRUE(state.buttons.IsAllOff());
}

void ExpectFuzzedHomeButtonState(
    const ::nn::hid::system::HomeButtonState& state) NN_NOEXCEPT
{
    EXPECT_LE(0, state.samplingNumber);
    ::nn::hid::system::HomeButtonSet buttons = {};
    buttons.Set<::nn::hid::system::HomeButton::Active>(true);
    EXPECT_EQ(buttons, state.buttons);
}

void ExpectSameOriginHomeButtonStates(
    const ::nn::hid::system::HomeButtonState& lhs,
    const ::nn::hid::system::HomeButtonState& rhs) NN_NOEXCEPT
{
    EXPECT_EQ(lhs.buttons, rhs.buttons);
}

void ExpectAdjacentHomeButtonStates(
    const ::nn::hid::system::HomeButtonState& lhs,
    const ::nn::hid::system::HomeButtonState& rhs) NN_NOEXCEPT
{
    EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
}

} // namespace
