﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/debug/hid_ConsoleSixAxisSensor.h>
#include <nn/os.h>
#include <nn/util/util_Vector.h>
#include <nnt/nntest.h>

#include "../Common/testHid_ConsoleSixAxisSensor.h"

#define VERBOSE

namespace
{

const nn::hid::NpadIdType NpadIds[] = { nn::hid::NpadId::Handheld };

const int NpadIdCountMax = NN_ARRAY_SIZE(NpadIds);

const auto WaitIntervalMilliSeconds = 100;

::nn::hid::ConsoleSixAxisSensorHandle g_ConsoleHandle;
::nn::hid::SixAxisSensorHandle        g_HandheldHandle;

int64_t ConvertToIntervalMilliSeconds(int freq) NN_NOEXCEPT
{
    if (freq == 0)
    {
        return 0;
    }

    return 1000 / freq;
}

void Initialize()
{
    ::nn::hid::InitializeNpad();
    ::nn::hid::InitializeConsoleSixAxisSensor();
    ::nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);
    ::nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleHandheld::Mask);

    // handle の取得
    ::nn::hid::GetSixAxisSensorHandle(&g_ConsoleHandle);
    int count = nn::hid::GetSixAxisSensorHandles(&g_HandheldHandle,
        1,
        nn::hid::NpadId::Handheld,
        nn::hid::NpadStyleHandheld::Mask);

    EXPECT_EQ(1, count);
}

void Finalize()
{
    ::nn::hid::StopSixAxisSensor(g_HandheldHandle);
    ::nn::hid::StopSixAxisSensor(g_ConsoleHandle);
    ::nn::hid::SetNpadHandheldActivationMode(nn::hid::NpadHandheldActivationMode_Dual);
    ::nn::hid::debug::FinalizeConsoleSixAxisSensor();
}

void SamplingFrequencyTest(::nn::hid::NpadHandheldActivationMode mode,
                           bool enableConsole,
                           bool enableHandheld,
                           int expectedSamplingFrequency) NN_NOEXCEPT
{
    ::nn::hid::SetNpadHandheldActivationMode(mode);

    if (enableConsole)
    {
        ::nn::hid::StartSixAxisSensor(g_ConsoleHandle);
    }
    else
    {
        ::nn::hid::StopSixAxisSensor(g_ConsoleHandle);
    }

    if (enableHandheld)
    {
        ::nn::hid::StartSixAxisSensor(g_HandheldHandle);
    }
    else
    {
        ::nn::hid::StopSixAxisSensor(g_HandheldHandle);
    }

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            WaitIntervalMilliSeconds
        )
    );

    EXPECT_EQ(expectedSamplingFrequency,
              ::nn::hid::debug::GetConsoleSixAxisSensorSamplingFrequency());
}

} // namespace

TEST(ConsoleSixAxisSensorNpadHandheldSuite, ReadingStateTest1)
{
    Initialize();

    // Npad Handheld のサンプリング開始
    ::nn::hid::SetNpadHandheldActivationMode(nn::hid::NpadHandheldActivationMode_None);
    ::nn::hid::StartSixAxisSensor(g_HandheldHandle);
    ::nn::hid::StartSixAxisSensor(g_ConsoleHandle);

    ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(WaitIntervalMilliSeconds));

    ::nn::hid::SixAxisSensorState state;
    ::nn::hid::GetSixAxisSensorState(&state, g_HandheldHandle);

    const auto SamplingNumber = state.samplingNumber;
    const auto ExpectedSamplingNumberDelta = 1000.f;

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ConvertToIntervalMilliSeconds(
                ::nnt::hid::NpadHandheldSamplingFrequency
            ) * ExpectedSamplingNumberDelta
        )
    );

    ::nn::hid::GetSixAxisSensorState(&state, g_HandheldHandle);

    // 値の妥当性チェック
    ::nnt::hid::AssertIfNotAtRest<::nn::hid::SixAxisSensorState>(state);

    const auto ErrorRate = 1.f - (state.samplingNumber - SamplingNumber) / ExpectedSamplingNumberDelta;
    NN_LOG("ErrorRate = %f, Delta = %lld, ExpectedDelta = %lld\n"
        , ErrorRate
        , state.samplingNumber - SamplingNumber
        , static_cast<int64_t>(ExpectedSamplingNumberDelta)
    );
    EXPECT_LT(ErrorRate, 0.1f);

    Finalize();
}

TEST(ConsoleSixAxisSensorNpadHandheldSuite, SamplingFrequencyTest1)
{
    Initialize();

    struct TestCase
    {
        ::nn::hid::NpadHandheldActivationMode mode;
        bool enableConsole;
        bool enableHandheld;
        int expectedSamplingFrequency;
    };

    const TestCase TestCaseArray[] = {
        {
            ::nn::hid::NpadHandheldActivationMode_Dual,
            false,
            false,
            0,
        },
        {
            ::nn::hid::NpadHandheldActivationMode_None,
            false,
            false,
            ::nnt::hid::NpadHandheldSamplingFrequency,
        },
        {
            ::nn::hid::NpadHandheldActivationMode_None,
            false,
            true,
            ::nnt::hid::NpadHandheldSamplingFrequency,
        },
        {
            ::nn::hid::NpadHandheldActivationMode_None,
            true,
            true,
            ::nnt::hid::VrSamplingFrequency,
        },
        {
            ::nn::hid::NpadHandheldActivationMode_None,
            false,
            true,
            ::nnt::hid::NpadHandheldSamplingFrequency,
        },
        {
            ::nn::hid::NpadHandheldActivationMode_Dual,
            false,
            false,
            0,
        },
    };

    for (const auto& testCase : TestCaseArray)
    {
        SamplingFrequencyTest(
            testCase.mode,
            testCase.enableConsole,
            testCase.enableHandheld,
            testCase.expectedSamplingFrequency
        );
    }

    Finalize();
}
