﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/util/util_Vector.h>

#include <nn/TargetConfigs/build_Base.h>

namespace nnt { namespace hid {

//!< Npad Handheld のサンプリング周波数です。
const auto NpadHandheldSamplingFrequency = 200;

//!< Vr 利用時のサンプリング周波数です。
const auto VrSamplingFrequency = 1000;

//!< ConsoleSixAxisSensor のドライバ層でのサンプリング間隔を返します
inline ::nn::TimeSpanType GetConsoleSixAxisSensorDriverSamplingInterval() NN_NOEXCEPT
{
    // sasbus でのセンサーのサンプリング周期を返します
    return ::nn::TimeSpanType::FromMilliSeconds(1);
}

//!< ConsoleSixAxisSensor のサンプリング間隔を返します
inline ::nn::TimeSpanType GetConsoleSixAxisSensorSamplingInterval() NN_NOEXCEPT
{
    // hid でのサンプリングタイマー周期を返します
    return ::nn::TimeSpanType::FromMilliSeconds(5);
}

//!< count 数分のサンプルがバッファに溜まるまで待つ
inline void SleepThreadForSampleCount(int count)
{
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            (count / ::nnt::hid::GetConsoleSixAxisSensorDriverSamplingInterval().GetMilliSeconds())
            + ::nnt::hid::GetConsoleSixAxisSensorSamplingInterval().GetMilliSeconds()
        )
    );
}

//!< サンプリング開始直後に一定時間待つ
inline void SleepThreadOnSamplingStarted()
{
    // TORIAEZU: 1000 ms
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(1000)
    );
}

//!< 静止時のセンサーの値の妥当性を検証
template<typename T>
void AssertIfNotAtRest(const T& state) NN_NOEXCEPT
{
    // 加速度値が 1G 付近か
    ::nn::util::Vector3f accelerometer;
    ::nn::util::VectorLoad(&accelerometer, state.acceleration);

    ASSERT_NEAR(accelerometer.Length(), 1.0f, 0.1);

    // 角速度の各成分が 0.0f 付近か
    for (const auto& angularVelocity : state.angularVelocity.v)
    {
        ASSERT_NEAR(angularVelocity, 0.0f, 0.1f);
    }
}

}} // namaspace nnt::hid
