﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <algorithm>

#include <hos/nvhdcp_up.h>

#include "../Shared/testHdcp_Config.h"
#include "../Shared/testHdcp_Helper.h"
#include "../Shared/testHdcp_Util-hardware.nx.h"
#include "../Shared/HdcpRawTestInitializer.h"

/**
 * @brief   Check if APIs are working correctly
 *          using boundary arguments of general value and defined value.
 */
TEST(RawCommon, BoundaryArgs)
{
    nnt::hdcp::HdcpRawTestInitializer initializer;

    HDCP_CLIENT_HANDLE handle = nullptr;
    HDCP_RET_ERROR error;

    for (int i1 = 0; i1 < 16; ++i1) {
        ///////////////////////////////////////////////////////////////////////////////////////
        // QUESTION:
        // When setting a value excepts "1" to fbIndex, HDCP_RET_SUCCESS is returned from hdcp_open().
        // Is it a correct behavior (i.e., specification)?
        ///////////////////////////////////////////////////////////////////////////////////////
        error = hdcp_open(&handle, i1);
        ASSERT_EQ(HDCP_RET_SUCCESS, error);
        error = hdcp_close(handle);
        ASSERT_EQ(HDCP_RET_SUCCESS, error);
    }
}

/**
 * @brief   Check if error processings of APIs are working correctly
 *          using irregular arguments of general value and defined value.
 */
TEST(RawCommon, IrregularArgs)
{
    nnt::hdcp::HdcpRawTestInitializer initializer;

    HDCP_CLIENT_HANDLE handle = nullptr;
    HDCP_CLIENT_HANDLE* nullHandle = nullptr;
    HDCP_RET_ERROR error;

    // Null pointer check.
    error = hdcp_open(nullHandle, nnt::hdcp::HdcpDeviceId);
    EXPECT_EQ(HDCP_RET_INVALID_PARAMETER, error);
    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_INVALID_PARAMETER, error);
    error = hdcp_enable(handle, true);
    EXPECT_EQ(HDCP_RET_INVALID_PARAMETER, error);
    error = hdcp_enable(handle, false);
    EXPECT_EQ(HDCP_RET_INVALID_PARAMETER, error);
    error = hdcp_close(handle);
    EXPECT_EQ(HDCP_RET_INVALID_PARAMETER, error);

    ///////////////////////////////////////////////////////////////////////////////////////
    // FIX (small thing):
    // For hdcp_open, some error codes should be returned.
    ///////////////////////////////////////////////////////////////////////////////////////
    error = hdcp_open(&handle, -1);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    error = hdcp_close(handle);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    error = hdcp_open(&handle, std::numeric_limits<int>::max());
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    error = hdcp_close(handle);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    ///////////////////////////////////////////////////////////////////////////////////////
    // QUESTION:
    // How can I test the following error codes?
    // - HDCP_RET_NO_MEMORY
    // - HDCP_RET_READM_FAILED
    // - HDCP_RET_LINK_PENDING
    ///////////////////////////////////////////////////////////////////////////////////////
}

/**
 * @brief   Check if the specification of APIs is correct without initialization.
 */
TEST(RawCommon, NotInitialzed)
{
    HDCP_CLIENT_HANDLE handle;

    // Unfortunately, death test is unsupported on NX environment.
    // Moreover, there is an issue that nv::FinalizeGraphics cannot finalize nv::Graphics completely.
    // Please see "FIX" message in testHdcp_Util-hardware.nx.cpp.
    EXPECT_DEATH_IF_SUPPORTED(hdcp_open(&handle, nnt::hdcp::HdcpDeviceId), "");
    EXPECT_DEATH_IF_SUPPORTED(hdcp_enable(handle, true), "");
    EXPECT_DEATH_IF_SUPPORTED(hdcp_enable(handle, false), "");
    EXPECT_DEATH_IF_SUPPORTED(hdcp_status(handle), "");
    EXPECT_DEATH_IF_SUPPORTED(hdcp_close(handle), "");

    nnt::hdcp::HdcpRawTestInitializer initializer;

    HDCP_RET_ERROR error;

    // Generate invalid handle.
    error = hdcp_open(&handle, nnt::hdcp::HdcpDeviceId);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);
    error = hdcp_close(handle);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);

    ///////////////////////////////////////////////////////////////////////////////////////
    // FIX (small thing):
    // Preferable behaviors:
    //  - Abort within API and show error message.
    //  - Returning "NOT INITIALIZED".
    ///////////////////////////////////////////////////////////////////////////////////////
    error = hdcp_enable(handle, true);
    EXPECT_EQ(HDCP_RET_UNSUCCESSFUL, error);
    error = hdcp_enable(handle, false);
    EXPECT_EQ(HDCP_RET_UNSUCCESSFUL, error);
    ///////////////////////////////////////////////////////////////////////////////////////
    // FIX:
    // HDCP_RET_READS_FAILED should be returned just as an error
    //  which indicates that the API tried to read but failed.
    // Here, no reading should not occur because an invalid handle is used.
    // HDCP_RET_UNSUCCESSFUL is bit good but "NOT INITIALIZED" is better.
    ///////////////////////////////////////////////////////////////////////////////////////
    error = hdcp_status(handle);
    EXPECT_EQ(HDCP_RET_READS_FAILED, error);
}

/**
 * @brief   Check if "open" and "close" are working well.
 */
TEST(RawCommon, OpenAndClose)
{
    nnt::hdcp::HdcpRawTestInitializer initializer;

    HDCP_CLIENT_HANDLE handle[nnt::hdcp::MaxHandleCount + 1];
    HDCP_RET_ERROR error;

    error = hdcp_open(&handle[nnt::hdcp::MaxHandleCount], nnt::hdcp::HdcpDeviceId);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    for (int i1 = 0; i1 < nnt::hdcp::MaxHandleCount; ++i1)
    {
        error = hdcp_open(&handle[i1], nnt::hdcp::HdcpDeviceId);
        EXPECT_EQ(HDCP_RET_UNSUCCESSFUL, error);
    }

    error = hdcp_close(handle[nnt::hdcp::MaxHandleCount]);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
}
