﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <algorithm>

#include "../Shared/testHdcp_Config.h"
#include "../Shared/testHdcp_Helper.h"
#include "../Shared/testHdcp_Util-hardware.nx.h"
#include "../Shared/HdcpTestInitializer.h"

/**
 * @brief   一般的な変数の境界値で正常動作するか
 *          定義された変数の境界値で正常動作するか
 */
TEST(Common, BoundaryArgs)
{
    nnt::hdcp::HdcpTestInitializer initializer;
    nn::os::SystemEventType eventType;
    hdcpc::GetHdcpStateTransitionEvent(&eventType);
    hdcpc::SetCurrentHdcpMode(nn::hdcp::HdcpMode_Enabled);
    EXPECT_EQ(nn::hdcp::HdcpMode_Enabled, hdcpc::GetCurrentHdcpMode());
    hdcpc::SetCurrentHdcpMode(nn::hdcp::HdcpMode_Disabled);
    EXPECT_EQ(nn::hdcp::HdcpMode_Disabled, hdcpc::GetCurrentHdcpMode());
    nn::hdcp::HdcpAuthenticationState state;
    hdcpc::GetHdcpAuthenticationState(&state);
    nn::os::TimedWaitSystemEvent(&eventType, nn::TimeSpan::FromMilliSeconds(500));
    nn::os::ClearSystemEvent(&eventType);
}

/**
 * @brief   一般的な変数の異常値で正常にエラー処理されるか
 *          定義された変数の異常値で正常にエラー処理されるか
 */
TEST(Common, IrregularArgs)
{
    nnt::hdcp::HdcpTestInitializer initializer;

    // 今のところ、境界値チェックできる引数(e.g., int, float etc.)を持ったAPIはない。

    // nullptr チェック
    EXPECT_DEATH_IF_SUPPORTED(hdcpc::GetHdcpStateTransitionEvent(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(hdcpc::GetHdcpAuthenticationState(nullptr), "");
}

/**
 * @brief   初期化せずに実行したときにアボートするか
 */
TEST(Common, NotInitialzed)
{
    nn::os::SystemEventType eventType;
    EXPECT_DEATH_IF_SUPPORTED(hdcpc::GetHdcpStateTransitionEvent(&eventType), "");
    EXPECT_DEATH_IF_SUPPORTED(hdcpc::SetCurrentHdcpMode(nn::hdcp::HdcpMode_Disabled), "");
    nn::hdcp::HdcpAuthenticationState state;
    EXPECT_DEATH_IF_SUPPORTED(hdcpc::GetHdcpAuthenticationState(&state), "");
    EXPECT_DEATH_IF_SUPPORTED(hdcpc::GetCurrentHdcpMode(), "");
}

/**
 * @brief   繰り返しの初期化と終了処理で異常なことが起きないか。
 */
TEST(Common, InitializeAndFinalize)
{
    nnt::hdcp::HdcpTestInitializer initializer;
    nnt::hdcp::HdcpService service = initializer.GetHdcpServiceObject();
    for (int i1 = 0; i1 < nnt::hdcp::MaxHandleCount; ++i1)
    {
        nn::hdcp::Initialize(service);
    }
    for (int i1 = 0; i1 < nnt::hdcp::MaxHandleCount; ++i1)
    {
        nn::hdcp::Finalize();
    }
    for (int i1 = 0; i1 < nnt::hdcp::MaxHandleCount; ++i1)
    {
        nn::hdcp::Initialize(service);
        nn::hdcp::Finalize();
    }
}
