﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Assert.h>
#include <nn/mem.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nn/ui2d.h>
#include <nn/font.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nn/util/util_Vector.h>
#include <nn/util/util_Matrix.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#endif

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/init.h>
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nv/nv_MemoryManagement.h>
#endif

#include <nnt.h>
#include <nnt/graphics/testGraphics_Path.h>
#include <nnt/graphics/testGraphics_GetHostExecutableFilepath.h>
#include <nnt/graphics/testGraphics_CreateDirectories.h>
#include <nnt/graphics/testGraphics_InitializeAllocatorFunctionForStandardAllocator.h>
#include <nnt/graphics/testGraphics_Png.h>
#include "testUi2d_Dataset.h"

//------------------------------------------------------------------------------
//  変数の宣言
//------------------------------------------------------------------------------
static int g_AllocationCount = 0;

// アプリケーションヒープとして確保されるメモリ領域の大きさ(byte)
static const size_t ApplicationHeapSize = 256 * 1024 * 1024;

// アプリケーションヒープ
static nn::util::BytePtr g_pMemoryHeap(NULL);
static nn::mem::StandardAllocator g_ApplicationHeapHandle;

// ui2d レイアウト
static nn::ui2d::Layout* g_pLayout = NULL;
// ui2d アーカイブリソースアクセッサ
static nn::ui2d::ArcResourceAccessor* g_pArcResourceAccessor = NULL;
// レイアウトアーカイブバイナリデータ
static void* g_pLayoutArchiveBinary;
// ui2d が描画に使用するグラフィックスリソース
static nn::ui2d::GraphicsResource g_GraphicsResource;

// ui2d 描画の際に使用するコンスタントバッファ
static nn::font::GpuBuffer  g_Ui2dConstantBuffer;
// フォントの描画に使用するコンスタントバッファ
static nn::font::GpuBuffer  g_Ui2dFontConstantBuffer;
// ui2d 描画情報
static nn::ui2d::DrawInfo g_DrawInfo;
// アニメーションコントロール用の ui2d アニメータ
static nn::ui2d::GroupAnimator* g_pAnimator;

static const int CaptureImageWidth = 1280;
static const int CaptureImageHeight = 720;

// グラフィックスフレームワーク
nns::gfx::GraphicsFramework                 g_GfxFramework;
ptrdiff_t                                   g_offsetToUi2dDrawMemoryPool;

static const char* LayoutArchivePath = "Contents:/Resources/testUi2d_RuntimeUnitTest.arc";
static const char* FontDirectoryPath = "Contents:/Resources";

static nnt::ui2d::Dataset g_Dataset;

static const int MaxNumberOfFonts = 256;
int g_NumberOfFonts = 0;
nn::font::ResFont* g_Fonts[MaxNumberOfFonts];

//------------------------------------------------------------------------------
//  アプリケーションヒープからメモリを確保する
//------------------------------------------------------------------------------
static void* AllocateFromApplicationHeap(size_t size, size_t alignment)
{
    void* ptr = g_ApplicationHeapHandle.Allocate(size, alignment);
    if (ptr != NULL)
    {
        g_AllocationCount++;
    }
    return ptr;
}

static void* AllocateFromApplicationHeap(size_t size)
{
    return AllocateFromApplicationHeap(size, 4);
}

//------------------------------------------------------------------------------
//  アプリケーションヒープから確保したメモリを開放する
//------------------------------------------------------------------------------
static void DeallocateApplicationHeap(void* ptr)
{
    if (ptr == NULL)
    {
        return;
    }
    g_AllocationCount--;
    g_ApplicationHeapHandle.Free(ptr);
}

template <typename TObject>
TObject* AllocAndConstruct()
{
    return new(AllocateFromApplicationHeap(sizeof(TObject))) TObject();
}

template <typename TObject>
void DestructAndFree(TObject* ptr)
{
    ptr->TObject::~TObject();
    DeallocateApplicationHeap(ptr);
}

//------------------------------------------------------------------------------
// ui2d 用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* Ui2dAllocateFunction(size_t size, size_t alignment, void* pUserData)
{
    NN_UNUSED(pUserData);
    return AllocateFromApplicationHeap(size, alignment);
}

static void Ui2dDeallocateFunction(void* ptr, void* pUserData)
{
    NN_UNUSED(pUserData);
    DeallocateApplicationHeap(ptr);
}

//------------------------------------------------------------------------------
// nn::fs用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* FsAllocateFunction( size_t size )
{
    return malloc(size);
}

static void FsDeallocateFunction( void* ptr, size_t size )
{
    NN_UNUSED( size );
    free(ptr);
}

//------------------------------------------------------------------------------

bool RegisterSlotForTexture_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);

        const int slotIndex = graphicsFramework->AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1 );
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);

        pPool->BeginUpdate();
        {
            pPool->SetTextureView(slotIndex, &textureView);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(pDstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------

bool RegisterSlotForSampler_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);

        const int slotIndex = graphicsFramework->AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, 1 );
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);

        pPool->BeginUpdate();
        {
            pPool->SetSampler(slotIndex, &sampler);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(pDstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------

void UnregisterSlotForSampler_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    NN_UNUSED(sampler);
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);
        const int slotIndex = pPool->GetDescriptorSlotIndex(*pDstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, slotIndex);
    }

}

//------------------------------------------------------------------------------

void UnregisterSlotForTexture_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    NN_UNUSED(textureView);
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);
        int slotIndex = pPool->GetDescriptorSlotIndex(*pDstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, slotIndex);
    }
}

//------------------------------------------------------------------------------
// メモリを確保してファイルを読み込みます
//------------------------------------------------------------------------------
void* ReadFileWithAllocate(const char* pFileName, size_t alignment)
{
    nn::Result result;
    nn::fs::FileHandle fileHandle;
    int64_t fileSize;

    NN_LOG("Loading file: %s ...", pFileName);

    result = nn::fs::OpenFile(&fileHandle, pFileName, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::fs::GetFileSize(&fileSize, fileHandle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    void* binary = AllocateFromApplicationHeap(static_cast<size_t>(fileSize), alignment);
    size_t readSize;
    result = nn::fs::ReadFile(&readSize, fileHandle, 0, binary, static_cast<size_t>(fileSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::fs::CloseFile(fileHandle);

    NN_LOG("ok\n", pFileName);

    return binary;
}

//-----------------------------------------------------------------------------
// ファイルシステムの初期化
//-----------------------------------------------------------------------------
void InitializeFileSystem()
{
    nn::fs::SetAllocator( FsAllocateFunction, FsDeallocateFunction );

    nn::Result result;

    nnt::graphics::Path contentsPath(nnt::graphics::GetHostExecutableFilepath());
    contentsPath.MakeParent();
    contentsPath.MakeParent();
    contentsPath.CombineAssign("Contents");

#if defined(NN_BUILD_APISET_NX)
    contentsPath.CombineAssign("NX");
#elif defined(NN_BUILD_CONFIG_SPEC_GENERIC)
    contentsPath.CombineAssign("Generic");
#else
    NN_ABORT("Unexpected spec.");
#endif
    NN_LOG("Mount Contents: %s\n", contentsPath.GetString());
    result = nn::fs::MountHost("Contents", contentsPath.GetString());
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nnt::graphics::Path outputsPath(nnt::graphics::GetHostExecutableFilepath());
    outputsPath.MakeParent();
    NN_LOG("Mount Outputs: %s\n", outputsPath.GetString());
    result = nn::fs::MountHost("Outputs", outputsPath.GetString());
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nnt::graphics::CreateDirectories("Outputs:/Outputs");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    NN_UNUSED(result);
}

//------------------------------------------------------------------------------
// ui2d 関連の初期化処理(SampleMode_Simple)
//------------------------------------------------------------------------------
static void RegisterDatasetFonts(const char* fontname, void* param)
{
    NN_UNUSED(param);
    NN_ASSERT(g_NumberOfFonts + 1 < MaxNumberOfFonts);

    nn::gfx::Device*    pDevice = g_GfxFramework.GetDevice();

    // フォントの初期化
    nn::font::ResFont* pFont = AllocAndConstruct<nn::font::ResFont>();

    // フォントの読み込み
    nnt::graphics::Path fontPath(FontDirectoryPath);
    fontPath.CombineAssign(fontname);

    void* pFontBinary = ReadFileWithAllocate(fontPath.GetString(), nn::font::ResourceAlignment);
    bool    result = pFont->SetResource(pDevice, pFontBinary);
    NN_ASSERT(result);
    pFont->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture_, &g_GfxFramework);

    g_pArcResourceAccessor->RegisterFont(fontname, pFont);

    g_Fonts[g_NumberOfFonts] = pFont;
    g_NumberOfFonts++;
}

//------------------------------------------------------------------------------
// ui2d 描画のためのバッファ関連初期化処理
//------------------------------------------------------------------------------
void InitializeUi2dBuffers(nn::ui2d::DrawInfo& drawInfo, const nn::ui2d::BuildResultInformation& buildResultInformation)
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // バッファの再セットアップ
    // ui2d 描画に使用する各種 gfx バッファのためのメモリプールを作成
    nn::gfx::MemoryPoolInfo memPoolInfo;
    memPoolInfo.SetDefault();
    memPoolInfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );

    // BuildWithName でレイアウトデータをセットアップした際に BuildResultInformation に収集した各種バッファのサイズを満たすメモリプールを作成する。
    const size_t constantBufferTotalSize = buildResultInformation.requiredUi2dConstantBufferSize + buildResultInformation.requiredFontConstantBufferSize;
    const size_t memoryPoolGranularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pDevice, memPoolInfo);
    const size_t memoryPoolSize = nn::util::align_up(constantBufferTotalSize, memoryPoolGranularity);
    const size_t MemoryPoolAlignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, memPoolInfo);

    if (memoryPoolSize > 0)
    {
        const nns::gfx::GraphicsFramework::MemoryPoolType constantBufferType = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;
        const size_t totalMemoryPoolSize = memoryPoolSize * nn::ui2d::ConstantBufferCount;

        nn::gfx::MemoryPool* pMemoryPool = g_GfxFramework.GetMemoryPool(constantBufferType);
        g_offsetToUi2dDrawMemoryPool = g_GfxFramework.AllocatePoolMemory(constantBufferType, totalMemoryPoolSize, MemoryPoolAlignment);

        size_t  bufferOffset = g_offsetToUi2dDrawMemoryPool;

        // ui2d 用のコンスタントバッファを作成する。
        if (buildResultInformation.requiredUi2dConstantBufferSize > 0)
        {
            nn::font::GpuBuffer::InitializeArg  arg;

                arg.SetBufferSize(buildResultInformation.requiredUi2dConstantBufferSize);
            arg.SetBufferCount(nn::ui2d::ConstantBufferCount);
            arg.SetGpuAccessFlag(nn::gfx::GpuAccess_ConstantBuffer);
                arg.SetMemoryPool(pMemoryPool);
            arg.SetMemoryPoolOffset(bufferOffset);
            arg.SetAllocator(Ui2dAllocateFunction, NULL);
            arg.SetAllocateSyncFlag(false);

            // フォント描画のためのコンスタントバッファを作成
                g_Ui2dConstantBuffer.Initialize(pDevice, arg);
            drawInfo.SetUi2dConstantBuffer(&g_Ui2dConstantBuffer);

                bufferOffset += buildResultInformation.requiredUi2dConstantBufferSize * nn::ui2d::ConstantBufferCount;

        }
        else
        {
            drawInfo.SetUi2dConstantBuffer(NULL);
        }

        // ui2d 内で使用される font 用のコンスタントバッファを作成する。
        if (buildResultInformation.requiredFontConstantBufferSize > 0)
        {
            nn::font::GpuBuffer::InitializeArg  arg;

                arg.SetBufferSize(buildResultInformation.requiredFontConstantBufferSize);
            arg.SetBufferCount(nn::ui2d::ConstantBufferCount);
            arg.SetGpuAccessFlag(nn::gfx::GpuAccess_ConstantBuffer);
                arg.SetMemoryPool(pMemoryPool);
            arg.SetMemoryPoolOffset(bufferOffset);
            arg.SetAllocator(Ui2dAllocateFunction, NULL);
            arg.SetAllocateSyncFlag(false);

                g_Ui2dFontConstantBuffer.Initialize(pDevice, arg);
            drawInfo.SetFontConstantBuffer(&g_Ui2dFontConstantBuffer);

                bufferOffset += buildResultInformation.requiredFontConstantBufferSize * nn::ui2d::ConstantBufferCount;
        }
        else
        {
            drawInfo.SetFontConstantBuffer(NULL);
        }
    }
}

//------------------------------------------------------------------------------
// ui2d 関連の初期化処理
//------------------------------------------------------------------------------
void InitializeUi2dRuntime(const char* pLayoutName, bool isAnimationData)
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // レイアウトライブラリの初期化
    nn::ui2d::Initialize(Ui2dAllocateFunction, Ui2dDeallocateFunction, NULL);

    // リソースアクセサの初期化
    g_pArcResourceAccessor = AllocAndConstruct<nn::ui2d::ArcResourceAccessor>();

    // レイアウトアーカイブの読み込み
    g_pLayoutArchiveBinary = ReadFileWithAllocate(LayoutArchivePath, nn::ui2d::ArchiveResourceAlignment);
    {
        bool    result = g_pArcResourceAccessor->Attach(g_pLayoutArchiveBinary, ".");
        NN_ASSERT(result);
    }

    // フォントの読み込み
    g_Dataset.ForEachFont(RegisterDatasetFonts, NULL);

    // レイアウトの初期化
    g_pLayout = AllocAndConstruct<nn::ui2d::Layout>();

    nn::ui2d::Layout::BuildOption   opt;
    opt.SetDefault();
    nn::ui2d::BuildResultInformation   buildResultInformation;
    buildResultInformation.SetDefault();
    if (!g_pLayout->BuildWithName(&buildResultInformation, pDevice, g_pArcResourceAccessor, NULL, NULL, opt, pLayoutName))
    {
        NN_LOG("Failed to build the layout data[%s].", pLayoutName);
    }

    // グラフィックスリソースの設定
    g_GraphicsResource.Setup(pDevice, 512);
    g_GraphicsResource.RegisterCommonSamplerSlot(RegisterSlotForSampler_, &g_GfxFramework);

    // Ui2d の描画に使用される各種バッファを初期化して DrawInfo へ設定する
    InitializeUi2dBuffers(g_DrawInfo, buildResultInformation);

    // 描画に使用する情報の設定
    {
        nn::util::MatrixT4x4fType   projection;
        nn::font::Rectangle rect = g_pLayout->GetLayoutRect();
        nn::util::MatrixOrthographicOffCenterRightHanded(
            &projection,
            rect.left,
            rect.right,
            rect.bottom,
            rect.top,
            0.0f,
            300.0f);
        nn::util::MatrixT4x3fType   view;
        nn::util::Vector3fType  pos;
        nn::util::Vector3fType  up;
        nn::util::Vector3fType  target;
        nn::util::VectorSet(&pos, 0.0f, 0.0f, 1.0f);
        nn::util::VectorSet(&up, 0.0f, 1.0f, 0.0f);
        nn::util::VectorSet(&target, 0.0f, 0.0f, 0.0f);
        nn::util::MatrixLookAtRightHanded(
            &view,
            pos,
            target,
            up);

        g_DrawInfo.SetGraphicsResource(&g_GraphicsResource);
        g_DrawInfo.SetProjectionMtx(projection);
        g_DrawInfo.SetViewMtx(view);
    }

    // アニメーションの作成と再生
    if(isAnimationData)
    {
        g_pAnimator = g_pLayout->CreateGroupAnimator(g_GfxFramework.GetDevice(), g_Dataset.GetAnimatorTag());
        g_pAnimator->PlayAuto(1.0f);
    }

    // テクスチャパターンアニメーションに利用されるテクスチャが正しく登録されるように、アニメーションの初期化後に実行します。
    g_pArcResourceAccessor->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture_, &g_GfxFramework);
}

//------------------------------------------------------------------------------
// ui2d 関連の終了処理
//------------------------------------------------------------------------------
void FinalizeUi2dRuntime()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    g_pLayout->Finalize(pDevice);
    DestructAndFree<nn::ui2d::Layout>(g_pLayout);

    g_Ui2dConstantBuffer.Finalize(pDevice, Ui2dDeallocateFunction, NULL);
    g_Ui2dFontConstantBuffer.Finalize(pDevice, Ui2dDeallocateFunction, NULL);

    g_GraphicsResource.UnregisterCommonSamplerSlot(UnregisterSlotForSampler_, &g_GfxFramework);
    g_GraphicsResource.Finalize(pDevice);

    for(int i = 0; i < g_NumberOfFonts; i++)
    {
        auto& pFont = g_Fonts[i];
        pFont->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, &g_GfxFramework);
        void* pFontResource = pFont->RemoveResource(pDevice);
        pFont->Finalize(pDevice);
        DeallocateApplicationHeap(pFontResource);

        DestructAndFree<nn::font::ResFont>(pFont);
    }
    g_NumberOfFonts = 0;

    g_pArcResourceAccessor->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, &g_GfxFramework);
    g_pArcResourceAccessor->Detach();
    g_pArcResourceAccessor->Finalize(pDevice);
    DestructAndFree<nn::ui2d::ArcResourceAccessor>(g_pArcResourceAccessor);

    // ArcResourceAccessor の Finalize でアクセスされるため最後に開放する。
    DeallocateApplicationHeap(g_pLayoutArchiveBinary);
}

//------------------------------------------------------------------------------
// メインループ を中断するかどうかを判定
//------------------------------------------------------------------------------
bool CheckToBreakMainLoop()
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    // WindowsMessege の処理
    {
        MSG  msg;
        if ( PeekMessage( &msg, NULL, 0, 0, PM_REMOVE) )
        {
            TranslateMessage(&msg);
            if ( msg.message == WM_QUIT)
            {
                return true;
            }
            DispatchMessage(&msg);
        }
    }
#elif !defined(NN_BUILD_CONFIG_OS_HORIZON)
    if(g_Pad.IsHold(g_Pad.MASK_L|g_Pad.MASK_R))
    {
        return true;
    }
#endif

    return false;
};


//------------------------------------------------------------------------------
// アプリケーションの初期化処理
//------------------------------------------------------------------------------
void InitializeApp_()
{
    // アプリケーションで使用するヒープ領域を初期化
    g_pMemoryHeap.Reset(new uint8_t[ApplicationHeapSize]);
    g_ApplicationHeapHandle.Initialize(g_pMemoryHeap.Get(), ApplicationHeapSize);

    // ファイルシステムの初期化(gfx の初期化より前に実行します)
    InitializeFileSystem();

    // グラフィックスフレームワークの初期化
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(1024 * 1024 * 32);
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayWidth(CaptureImageWidth);
    fwInfo.SetDisplayHeight(CaptureImageHeight);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, 32 * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 32 * 1024 * 1024);
    fwInfo.SetColorBufferFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
    g_GfxFramework.Initialize(fwInfo);
}

//------------------------------------------------------------------------------
// アプリケーションの終了処理
//------------------------------------------------------------------------------
void FinalizeApp_()
{
    nn::fs::Unmount("Outputs");
    // ファイルシステム終了処理。
    nn::fs::Unmount("Contents");

    // グラフィックスフレームワーク 終了処理
    g_GfxFramework.FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, g_offsetToUi2dDrawMemoryPool);
    g_GfxFramework.Finalize();

    // ヒープメモリーの終了処理
    g_ApplicationHeapHandle.Finalize();
    delete[] reinterpret_cast<uint8_t*>(g_pMemoryHeap.Get());
}

//------------------------------------------------------------------------------
//  コピーコンストラクタが正常に動作するかどうかのテスト
//------------------------------------------------------------------------------
TEST(RuntimeUnitTest, CopyConstructorTest)
{
    nn::gfx::Device*    pDevice = g_GfxFramework.GetDevice();

    // フォントを読み込むためにデータセットのデータを使用しています。
    g_Dataset.Initialize("Contents:/Resources/dataset.txt");
    g_Dataset.ReadConfig();

    //-----------------------
    // テスト用のループ
    //-----------------------
    while (g_Dataset.NextData())
    {
        if (strcmp(g_Dataset.GetDataName(), "Simple") == 0)
        {
            NN_LOG("Start the copy constructor test on runtime.\n");

            // ui2d ランタイムの初期化処理
            InitializeUi2dRuntime(g_Dataset.GetLayoutName(), false);

            nn::ui2d::Layout* pCopiedLayout = new(AllocateFromApplicationHeap(sizeof(nn::ui2d::Layout))) nn::ui2d::Layout(pDevice, *g_pLayout, NULL, NULL);

            bool result = pCopiedLayout->CompareCopiedInstanceTest(*g_pLayout);

            pCopiedLayout->Finalize(pDevice);
            DestructAndFree<nn::ui2d::Layout>(pCopiedLayout);

            FinalizeUi2dRuntime();

            ASSERT_TRUE(result);

            break;
        }
    }

    NN_LOG("Complete copy constructor test\n");

    g_Dataset.Finalize();

    NN_ASSERT(g_AllocationCount == 0);
    SUCCEED();
}

//------------------------------------------------------------------------------
//  特別な初期化処理が正常に動作するかどうかのテスト
//------------------------------------------------------------------------------
TEST(RuntimeUnitTest, SpecialInitializationTest)
{
    nn::gfx::Device*    pDevice = g_GfxFramework.GetDevice();

    // フォントを読み込むためにデータセットのデータを使用しています。
    g_Dataset.Initialize("Contents:/Resources/dataset.txt");
    g_Dataset.ReadConfig();

    NN_LOG("Start special initialization test on runtime.\n");

    //-----------------------
    // テスト用のループ
    //-----------------------
    while (g_Dataset.NextData())
    {
        //-------------------------------------------------------------------------------------------
        //  コンスタントバッファを別に管理していて計測結果が不要なケースのテスト。
        //  Build 時に NULL の BuildResultInformation を渡した際に問題なく動作するかチェック。
        //  全てのペインの種類をテストするため Simple と PictureShape を使用する。
        //-------------------------------------------------------------------------------------------
        if (strcmp(g_Dataset.GetDataName(), "Simple") == 0 ||
            strcmp(g_Dataset.GetDataName(), "PictureShape") == 0)
        {
            // レイアウトライブラリの初期化
            nn::ui2d::Initialize(Ui2dAllocateFunction, Ui2dDeallocateFunction, NULL);

            // リソースアクセサの初期化
            g_pArcResourceAccessor = AllocAndConstruct<nn::ui2d::ArcResourceAccessor>();

            // レイアウトアーカイブの読み込み
            g_pLayoutArchiveBinary = ReadFileWithAllocate(LayoutArchivePath, nn::ui2d::ArchiveResourceAlignment);
            {
                bool    attachResult = g_pArcResourceAccessor->Attach(g_pLayoutArchiveBinary, ".");
                NN_ASSERT(attachResult);
            }

            // レイアウトの初期化
            g_pLayout = AllocAndConstruct<nn::ui2d::Layout>();

            nn::ui2d::Layout::BuildOption   opt;
            opt.SetDefault();
            bool buildResult = g_pLayout->BuildWithName(NULL, pDevice, g_pArcResourceAccessor, NULL, NULL, opt, g_Dataset.GetLayoutName());

            ASSERT_TRUE(buildResult);

            g_pLayout->Finalize(pDevice);
            DestructAndFree<nn::ui2d::Layout>(g_pLayout);

            g_pArcResourceAccessor->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, &g_GfxFramework);
            g_pArcResourceAccessor->Detach();
            g_pArcResourceAccessor->Finalize(pDevice);
            DestructAndFree<nn::ui2d::ArcResourceAccessor>(g_pArcResourceAccessor);

            // ArcResourceAccessor の Finalize でアクセスされるため最後に開放する。
            DeallocateApplicationHeap(g_pLayoutArchiveBinary);
        }
    }

    NN_LOG("Complete copy constructor test\n");

    g_Dataset.Finalize();

    NN_ASSERT(g_AllocationCount == 0);
    SUCCEED();
}

//------------------------------------------------------------------------------
//  AcquireAnimTagNameByIndexs のテスト
//------------------------------------------------------------------------------
TEST(RuntimeUnitTest, LayoutTagNameTest)
{
    g_Dataset.Initialize("Contents:/Resources/dataset.txt");
    g_Dataset.ReadConfig();

    while (g_Dataset.NextData())
    {
        if (strcmp(g_Dataset.GetDataName(), "Simple") == 0)
        {
            NN_LOG("Start the LayoutTagNameTest test on runtime.\n");

            // ui2d ランタイムの初期化処理
            InitializeUi2dRuntime(g_Dataset.GetLayoutName(), false);
            {
                if (g_pLayout != nullptr)
                {
                    const int animTagCount = g_pLayout->AcquireAnimTagNameCount();
                    ASSERT_EQ(animTagCount, 1);

                    for (int i = 0; i < animTagCount; i++)
                    {
                        const char* pTagName = g_pLayout->AcquireAnimTagNameByIndex(i);
                        ASSERT_STREQ(pTagName, "loop");
                    }
                }
            }
            FinalizeUi2dRuntime();
            break;
        }
    }

    NN_LOG("Complete LayoutTagNameTest test\n");

    g_Dataset.Finalize();

    NN_ASSERT(g_AllocationCount == 0);
    SUCCEED();
}

//------------------------------------------------------------------------------
//  RectDrawerWorkBuffer のテスト
//------------------------------------------------------------------------------
TEST(RuntimeUnitTest, RectDrawerWorkBufferTest)
{
    nn::font::RectDrawer rectDrawer;

    nn::gfx::Device*    pDevice = g_GfxFramework.GetDevice();

    const uint32_t CharCount = 1000;
    const size_t workBufSize = nn::font::RectDrawer::GetWorkBufferSize(pDevice, CharCount);
    const size_t workBufAlign = nn::font::RectDrawer::GetWorkBufferAlignment();

    const ptrdiff_t MerginForMutex = 128;
    void* pWorkBuf = AllocateFromApplicationHeap(workBufSize + MerginForMutex, workBufAlign);
    rectDrawer.Initialize(pDevice, pWorkBuf, CharCount);

    // ワークバッファの直後の領域で mutex を初期化。
    // NX 実機においては、終端で確保する MemoryPool がはみ出していれば、
    //「非キャッシュ領域に同期オブジェクトを置こうとしています。」というメッセージとともにアサートするはず。
    // NINTENDOSDK-3726 で報告された不具合ケースの再現です。
    nn::os::Mutex* pMutex = new(static_cast<char*>(pWorkBuf) + workBufSize)nn::os::Mutex(true, 16);
    NN_UNUSED(pMutex);

    rectDrawer.Finalize(pDevice);

    DeallocateApplicationHeap(pWorkBuf);
}

//------------------------------------------------------------------------------
//  リソース情報検索機能のテスト
//------------------------------------------------------------------------------
TEST(RuntimeUnitTest, FindResourceTest)
{
    //nn::gfx::Device*    pDevice = g_GfxFramework.GetDevice();

    // フォントを読み込むためにデータセットのデータを使用しています。
    g_Dataset.Initialize("Contents:/Resources/dataset.txt");
    g_Dataset.ReadConfig();

    //-----------------------
    // テスト用のループ
    //-----------------------
    while (g_Dataset.NextData())
    {
        if (strcmp(g_Dataset.GetDataName(), "Simple") == 0)
        {
            NN_LOG("Start the FindResource test on runtime.\n");

            // ui2d ランタイムの初期化処理
            InitializeUi2dRuntime(g_Dataset.GetLayoutName(), false);

            // 上書きしているパラメーターをとる場合
            {
                const nn::ui2d::ResPane* pResPane = NULL;
                const nn::ui2d::ResExtUserDataList* pOutExtUserDataList = NULL;

                // まず、部品ペインのリソースを取る
                const void* pResourceHead = g_pLayout->GetLayoutResource();
                nn::ui2d::Layout::FindResPaneByName(&pResPane, &pOutExtUserDataList, pResourceHead, "vi_parts", NULL);
                const nn::ui2d::ResParts* pResParts = static_cast<const nn::ui2d::ResParts*>(pResPane);
                {
                    // 部品アクセサークラスを初期化
                    nn::ui2d::Layout::PartsBuildDataAccessor partsBuildDataAccessor(pResParts);

                    // 上書きパラメーターの取得
                    const nn::ui2d::ResPartsProperty* pResPartsProperty = partsBuildDataAccessor.FindPartsPropertyFromName("textBox_vi");
                    const void* pOverwriteRes = partsBuildDataAccessor.GetPropertyResBlock(pResPartsProperty);
                    const nn::ui2d::ResTextBox* pOverwriteResTexture = static_cast<const nn::ui2d::ResTextBox*>(pOverwriteRes);

                    // 上書きしている文字列を取得して、チェックします。
                    auto pText = nn::util::ConstBytePtr(pOverwriteResTexture, pOverwriteResTexture->textStrOffset).Get<uint16_t>();

                    const uint16_t* pRefText = reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("Property Override"));
                    const int len = nn::font::CalculateWideCharString16Length(pRefText);
                    for (int i = 0; i < len + 1/* 終端文字を含む */; i++)
                    {
                        ASSERT_TRUE(pText[i] == pRefText[i]);
                    }
                }
            }

            // 部品配下のペインについて、リソースを取得する（上書き前の値や、上書きをしていない場合はこのように取得）
            {
                const nn::ui2d::ResPane* pResPane = NULL;
                const nn::ui2d::ResExtUserDataList* pOutExtUserDataList = NULL;

                // 部品ペインの持っているレイアウトから、元リソースへのポインタを得る。
                const nn::ui2d::Parts* pParts = g_pLayout->FindPartsPaneByName("vi_parts");
                const void* pPartsResource = pParts->GetLayout()->GetLayoutResource();

                nn::ui2d::Layout::FindResPaneByName(&pResPane, &pOutExtUserDataList, pPartsResource, "textBox_vi", NULL);
                const nn::ui2d::ResTextBox* pTextBox = static_cast<const nn::ui2d::ResTextBox*>(pResPane);
                auto pText = nn::util::ConstBytePtr(pTextBox, pTextBox->textStrOffset).Get<uint16_t>();

                const uint16_t* pRefText = reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL("TextBox Pane"));
                const int len = nn::font::CalculateWideCharString16Length(pRefText);
                for (int i = 0; i < len + 1/* 終端文字を含む */; i++)
                {
                    ASSERT_TRUE(pText[i] == pRefText[i]);
                }
            }

            // 検索スタートペインの指定
            {
                const nn::ui2d::ResPane* pSearchStartPane = NULL;
                const nn::ui2d::ResExtUserDataList* pOutExtUserDataList = NULL;

                // 見つかる
                const void* pResource = g_pLayout->GetLayoutResource();
                nn::ui2d::Layout::FindResPaneByName(&pSearchStartPane, &pOutExtUserDataList, pResource, "picture_tp", NULL);
                ASSERT_TRUE(pSearchStartPane != NULL);

                // 見つかったペインを起点に検索をする。今度は見つからない。
                const nn::ui2d::ResPane* pResPane2 = NULL;
                nn::ui2d::Layout::FindResPaneByName(&pResPane2, &pOutExtUserDataList, pResource, "picture_tp", pSearchStartPane);
                ASSERT_TRUE(pResPane2 == NULL);

                nn::ui2d::Layout::FindResPaneByName(&pResPane2, &pOutExtUserDataList, pResource, "picture_ts", pSearchStartPane);
                ASSERT_TRUE(pResPane2 != NULL);
            }

            FinalizeUi2dRuntime();
            break;
        }
    }

    NN_LOG("Complete FindResource test\n");

    g_Dataset.Finalize();

    NN_ASSERT(g_AllocationCount == 0);
    SUCCEED();
}

//------------------------------------------------------------------------------
//  Material クラスのテスト
//------------------------------------------------------------------------------
TEST(RuntimeUnitTest, MaterialTest)
{
    nn::gfx::Device*    pDevice = g_GfxFramework.GetDevice();

    // フォントを読み込むためにデータセットのデータを使用しています。
    g_Dataset.Initialize("Contents:/Resources/dataset.txt");
    g_Dataset.ReadConfig();

    //-----------------------
    // テスト用のループ
    //-----------------------
    while (g_Dataset.NextData())
    {
        if (strcmp(g_Dataset.GetDataName(), "Fragment") == 0)
        {
            NN_LOG("Start the material test on runtime.\n");

            // ui2d ランタイムの初期化処理
            InitializeUi2dRuntime(g_Dataset.GetLayoutName(), false);

            // ブレンドを変更して再初期化する
            {
                //「デフォルト設定を利用する」を無効にし、
                // かつプリセットの設定に無い設定を行っている場合
                {
                    const bool isRecursive = true;
                    nn::ui2d::Material* pMaterial = g_pLayout->GetRootPane()->FindMaterialByName("Picture_30", isRecursive);
                    NN_SDK_ASSERT_NOT_NULL(pMaterial);

                    NN_ASSERT(pMaterial->IsBlendModeCap());

                    nn::ui2d::ResBlendMode color_blend_mode(nn::ui2d::BlendOp_Disable, nn::ui2d::BlendFactor_0, nn::ui2d::BlendFactor_1, nn::ui2d::LogicOp_Disable);
                    pMaterial->SetBlendMode(color_blend_mode);

                    const int prevAllocationCount = g_AllocationCount;

                    pMaterial->FinalizeBlendInformation(pDevice);
                    pMaterial->InitializeBlendInformation(pDevice);

                    // メモリリークが起こっていないことを確認します。
                    NN_ASSERT(g_AllocationCount == prevAllocationCount);
                }

                //「デフォルト設定を利用する」を無効にし、
                // プリセットの設定と同じ設定を行っている場合
                {
                    const bool isRecursive = true;
                    nn::ui2d::Material* pMaterial = g_pLayout->GetRootPane()->FindMaterialByName("Picture_05", isRecursive);
                    NN_SDK_ASSERT_NOT_NULL(pMaterial);

                    // プリセット設定を使う場合、本来リソースは不要なはずなので、別途修正の必要があります。
                    NN_ASSERT(pMaterial->IsBlendModeCap());

                    nn::ui2d::ResBlendMode color_blend_mode(nn::ui2d::BlendOp_Disable, nn::ui2d::BlendFactor_0, nn::ui2d::BlendFactor_1, nn::ui2d::LogicOp_Disable);
                    pMaterial->SetBlendMode(color_blend_mode);

                    const int prevAllocationCount = g_AllocationCount;

                    // 新たにアロケーションが起こる。この場合、FinalizeBlendInformation は不要。
                    pMaterial->InitializeBlendInformation(pDevice);
                    NN_ASSERT(g_AllocationCount == prevAllocationCount + 2);
                    // Finalize して解放される
                    pMaterial->FinalizeBlendInformation(pDevice);
                    NN_ASSERT(g_AllocationCount == prevAllocationCount);
                }
            }

            FinalizeUi2dRuntime();
            break;
        }
    }

    NN_LOG("Complete material test\n");

    g_Dataset.Finalize();

    NN_ASSERT(g_AllocationCount == 0);
    SUCCEED();
}

//------------------------------------------------------------------------------
//  テキストボックスのテスト
//------------------------------------------------------------------------------
TEST(RuntimeUnitTest, SetStringToTextBoxTest)
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // フォントを読み込むためにデータセットのデータを使用しています。
    g_Dataset.Initialize("Contents:/Resources/dataset.txt");
    g_Dataset.ReadConfig();

    //-----------------------
    // テスト用のループ
    //-----------------------
    while (g_Dataset.NextData())
    {
        if (strcmp(g_Dataset.GetDataName(), "Simple") == 0)
        {
            NN_LOG("Start text box test on runtime.\n");

            // ui2d ランタイムの初期化処理
            InitializeUi2dRuntime(g_Dataset.GetLayoutName(), false);

            nn::ui2d::TextBox* pTextBox = static_cast<nn::ui2d::TextBox*>(g_pLayout->GetRootPane()->FindPaneByName("textBox_4"));
            // テキストボックスに動的に文字列を設定するテスト
            {
                ASSERT_TRUE(pTextBox != NULL);
                const uint16_t* pText = reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL(
                    "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"
                ));
                int len = nn::font::CalculateWideCharString16Length(pText);
                pTextBox->AllocateStringBuffer(pDevice, static_cast<uint16_t>(len));
                pTextBox->SetString(pText);
                const uint16_t* pString = pTextBox->GetString();
                for (int i = 0; i < len + 1/* 終端文字を含む */; i++)
                {
                    ASSERT_TRUE(pText[i] == pString[i]);
                }
            }

            // UTF-8 モードで動的にテキストボックスを作成するテスト
            {
                nn::ui2d::TextBox *pNewTextBox = nn::ui2d::Layout::AllocateAndConstruct<nn::ui2d::TextBox>(true/*isUtf8*/);
                pNewTextBox->SetFont(pTextBox->GetFont());
                pTextBox->AppendChild(pNewTextBox);
                {
                    const char* pText = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
                    size_t len = strlen(pText);
                    pNewTextBox->AllocateStringBuffer(pDevice, static_cast<uint16_t>(len));
                    pNewTextBox->SetStringUtf8(pText);
                    const char* pString = pNewTextBox->GetStringUtf8();
                    for (size_t i = 0; i < len + 1/* 終端文字を含む */; i++)
                    {
                        ASSERT_TRUE(pText[i] == pString[i]);
                    }
                }

                // サロゲートペア領域の文字を設定するテスト
                {
                    char utf8Text[8];
                    uint32_t utf32Text[2] = { 0x1f600/*サロゲートペア領域の絵文字「😀」*/, 0 };
                    nn::util::CharacterEncodingResult result = nn::util::ConvertStringUtf32ToUtf8(utf8Text, sizeof(utf8Text) / sizeof(utf8Text[0]), utf32Text);
                    ASSERT_TRUE(result == nn::util::CharacterEncodingResult_Success);
                    pNewTextBox->SetStringUtf8(utf8Text);
                    const char* pString = pNewTextBox->GetStringUtf8();
                    uint32_t utf32Text2[2];
                    result = nn::util::ConvertStringUtf8ToUtf32(utf32Text2, sizeof(utf32Text2) / sizeof(utf32Text2[0]), pString);
                    ASSERT_TRUE(result == nn::util::CharacterEncodingResult_Success);
                    for (int i = 0; i < sizeof(utf32Text) / sizeof(utf32Text[0])/* 終端文字を含む */; i++)
                    {
                        ASSERT_TRUE(utf32Text[i] == utf32Text2[i]);
                    }
                }
            }

            // ui2d ランタイムの終了処理
            FinalizeUi2dRuntime();
            break;
        }
    }

    NN_LOG("Complete text box test\n");

    g_Dataset.Finalize();

    NN_ASSERT(g_AllocationCount == 0);
    SUCCEED();
}

//------------------------------------------------------------------------------
//  DrawInfo のコピーコンストラクタのテスト
//------------------------------------------------------------------------------
TEST(RuntimeUnitTest, DrawInfoCopyConstructorTest)
{
    const int garbageData = 0xcd; // ゴミデータ
    nn::ui2d::DrawInfo drawInfo1;
    nn::ui2d::DrawInfo drawInfo2;
    bool paddingArea[sizeof(nn::ui2d::DrawInfo)] = { 0 };

    // DrawInfo のうち実行時型情報を除いた部分のバイナリ
    const size_t sizeOfRuntimeTypeInfo = sizeof(nn::font::detail::RuntimeTypeInfo*) + sizeof(nn::font::detail::RuntimeTypeInfo*);
    uint8_t* pDrawInfoBinary1 = reinterpret_cast<uint8_t*>(&drawInfo1) + sizeOfRuntimeTypeInfo;
    uint8_t* pDrawInfoBinary2 = reinterpret_cast<uint8_t*>(&drawInfo2) + sizeOfRuntimeTypeInfo;
    const size_t sizeOfDrawInfoWithoutRuntimeTypeInfo = sizeof(nn::ui2d::DrawInfo) - sizeOfRuntimeTypeInfo;

    // DrawInfo をゴミデータで埋めて、
    // 通常のコンストラクタでゴミデータのままになっている部分を暗黙パディング領域として記録する。
    memset(pDrawInfoBinary1, garbageData, sizeOfDrawInfoWithoutRuntimeTypeInfo);
    new(&drawInfo1)nn::ui2d::DrawInfo();
    for (size_t i = 0; i < sizeOfDrawInfoWithoutRuntimeTypeInfo; i++)
    {
        paddingArea[i] = (pDrawInfoBinary1[i] == garbageData);
    }

    // コピーコンストラクタで複製する。
    memset(pDrawInfoBinary2, garbageData, sizeOfDrawInfoWithoutRuntimeTypeInfo);
    new(&drawInfo2)nn::ui2d::DrawInfo(drawInfo1);

    // 暗黙パディング領域以外で差分がないかをチェックする。
    for (size_t i = 0; i < sizeOfDrawInfoWithoutRuntimeTypeInfo; i++)
    {
        if (paddingArea[i])
        {
            continue;
        }
        ASSERT_TRUE(pDrawInfoBinary1[i] == pDrawInfoBinary2[i]);
    }

    NN_LOG("Complete DrawInfoCopyConstructorTest\n");

    SUCCEED();
}

//------------------------------------------------------------------------------
//  WordWrapping の改行テスト
//------------------------------------------------------------------------------
class MyWordWrappingCallback : public nn::font::WordWrapCallback
{
public:
    virtual const uint16_t* GetLineBreakLimit(const uint16_t* str, const uint16_t* end)
    {
        const int pos = 3; // 3 文字分しか画面に収まらないことにする。
        if (static_cast<int>(end - str) > pos)
            return str + pos;
        return end;
    }
};

TEST(RuntimeUnitTest, WordWrappingTest)
{
    // http://unicode.org/reports/tr14/ の仕様で改行されることを確認するテスト。
    uint32_t writerSize;
    uint16_t outTextBuffer[256];
    const uint16_t* pInputText = reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL(
        "abcdefg hijk l m-n-o!(pq0-1-23) $45"
    ));
    const uint16_t* pCorrectText = reinterpret_cast<const uint16_t*>(
        NN_CHAR16LITERAL("abc\n")
        NN_CHAR16LITERAL("def\n")
        NN_CHAR16LITERAL("g\n") // SP " ": Enable indirect line breaks
        NN_CHAR16LITERAL("hij\n")
        NN_CHAR16LITERAL("k l\n") // SP " ": Enable indirect line breaks
        NN_CHAR16LITERAL("m-\n") // HY "-": Provide a line break opportunity after the character, except in numeric context
        NN_CHAR16LITERAL("n-\n")
        NN_CHAR16LITERAL("o!\n") // EX "!": Prohibit line breaks before
        NN_CHAR16LITERAL("(pq\n") // OP "(": Prohibit line breaks after
        NN_CHAR16LITERAL("0-1\n") // HY "-": Provide a line break opportunity after the character, except in numeric context
        NN_CHAR16LITERAL("-23\n")
        NN_CHAR16LITERAL(")\n")
        NN_CHAR16LITERAL("$45") // PR "$": Do not break in front of a numeric expression
    );
    MyWordWrappingCallback callbackFunc;
    bool result = nn::font::WordWrapping::CalculateWordWrapping(
        &writerSize,
        outTextBuffer,
        sizeof(outTextBuffer) / sizeof(outTextBuffer[0]),
        pInputText,
        callbackFunc
    );
    ASSERT_TRUE(result);
    int len = nn::font::CalculateWideCharString16Length(outTextBuffer);
    for (int i = 0; i < len + 1/* 終端文字を含む */; i++)
    {
        ASSERT_TRUE(outTextBuffer[i] == pCorrectText[i]);
    }
}

// -------------------------------------------------------------------------- -
//  Main 関数
//---------------------------------------------------------------------------
extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    // 初期化
    InitializeApp_();

    // すべてのテストを実行する。
    int result = RUN_ALL_TESTS();

    // 終了処理
    FinalizeApp_();

    ::nnt::Exit(result);
}


