﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <cstring>

#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Log.h>
#include <nn/gfx.h>
#include <nn/util/util_Color.h>

#define NN_PERF_PROFILE_ENABLED
#include <nn/perf.h>

#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>

#include <nnt.h>
namespace{

struct TlsSlotValue
{
    int                 coreNumber;
    nn::perf::GpuMeter* meter;
    const char*         name;
    nn::util::Color4u8  color;
};

}

// 0フィルされた名前を取得
const char* GetNullName() NN_NOEXCEPT;

// フレームワークを初期化
void InitializeGraphicsFramework() NN_NOEXCEPT;

// フレームワークを破棄
void FinalizeGraphicsFramework() NN_NOEXCEPT;

// フレームワークを取得
nns::gfx::GraphicsFramework* GetGfw() NN_NOEXCEPT;

// サブコマンドバッファを初期化(並列コマンド発行に使用)
void InitializeSubCommandBuffer() NN_NOEXCEPT;

// サブコマンドバッファを破棄
void FinalizeSubCommandBuffer() NN_NOEXCEPT;

// サブコマンドバッファを取得
nns::gfx::GraphicsFramework::CommandBuffer* GetSubCommandBuffer(int index) NN_NOEXCEPT;

// CPU スリープ
void Sleep() NN_NOEXCEPT;

// GPU スリープ
void SleepGpu() NN_NOEXCEPT;

// サブコマンドバッファ GPU スリープ
void SleepGpu(int index) NN_NOEXCEPT;

// CpuMeter 管理クラス
class CpuMeterManager
{
public:
    void Initialize(int coreCount, int bufferCount, int sectionCountMax) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    nn::perf::CpuMeter* GetMeter() NN_NOEXCEPT;

private:
    nn::perf::CpuMeter m_CpuMeter;
    void* m_Memory;
};

// GpuMeter 管理クラス
class GpuMeterManager
{
public:
    void Initialize(int coreCount, int bufferCount, int sectionCountMax) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    nn::perf::GpuMeter* GetMeter() NN_NOEXCEPT;

private:
    nn::perf::GpuMeter m_GpuMeter;
    void* m_Memory;
    ptrdiff_t m_MemoryPoolOffset;
};

// LoadMeterCenter 管理クラス
class LoadMeterCenterManager
{
    NN_DISALLOW_COPY(LoadMeterCenterManager);
public:
    static void Initialize(bool isGpuEnabled, int coreCount, int userMeterCount, int cpuBufferCount, int gpuBufferCount, int sectionCountMax) NN_NOEXCEPT;
    static void Finalize() NN_NOEXCEPT;

private:
    static LoadMeterCenterManager& GetInstance() NN_NOEXCEPT
    {
        static LoadMeterCenterManager m_Instance;
        return m_Instance;
    }

    LoadMeterCenterManager() NN_NOEXCEPT
    {}

    void* m_Memory;
    ptrdiff_t m_MemoryPoolOffset;
    bool m_IsGpuEnabled;
};
