﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// DescriptorPoolAllocator の確認テスト

#include <nn/nn_Assert.h>
#include <nn/gfx/util/gfx_DescriptorPoolAllocator.h>
#include <nnt.h>

extern bool g_bUserAllocatorInvalid;
extern int g_UserAllocatorAllocCount;
extern void* g_pExpectedAllocateUserData;
extern void* g_pExpectedFreeUserData;

void* UserAllocatorMalloc(size_t size, void* pUserData);
void UserAllocatorFree(void* ptr, void* pUserData);

TEST(DescriptorPoolAllocator, SingleThread)
{
    const int baseSlotIndex = 100;
    const int slotCount = 100;
    const int InvalidIndex = nn::gfx::util::DescriptorPoolAllocator::InvalidIndex;
    nn::gfx::DescriptorPool descriptorPool;
    nn::gfx::util::DescriptorPoolAllocator allocator;
    int allocateFunctionUserData = 0;
    int freeFunctionUserData = 0;

    g_bUserAllocatorInvalid = false;
    g_UserAllocatorAllocCount = 0;
    g_pExpectedAllocateUserData = reinterpret_cast<void*>(&allocateFunctionUserData);
    g_pExpectedFreeUserData = reinterpret_cast<void*>(&freeFunctionUserData);

    // Initialize() 前は未初期化状態か
    EXPECT_FALSE(allocator.IsInitialized());

    // 初期化する
    allocator.Initialize(
        UserAllocatorMalloc,
        g_pExpectedAllocateUserData,
        UserAllocatorFree,
        g_pExpectedFreeUserData,
        &descriptorPool,
        baseSlotIndex,
        slotCount,
        false
        );

    // Initialize() 後は初期化済み状態か
    EXPECT_TRUE(allocator.IsInitialized());

    // Initialize() で渡したパラメータを取得できているか
    EXPECT_EQ(allocator.GetDescriptorPool(), &descriptorPool);
    EXPECT_EQ(allocator.GetBaseSlotIndex(), baseSlotIndex);
    EXPECT_EQ(allocator.GetSlotCount(), slotCount);

    // slotCount 分の範囲を１０回アロケートする。
    // １回ごとに Free() するので、すべて成功するはず。
    for(int i=0; i<10; i++)
    {
        int index = allocator.Allocate( slotCount );

        // 戻ってきたインデックスは範囲内か
        EXPECT_GE(index, baseSlotIndex);
        EXPECT_LT(index, baseSlotIndex + slotCount);

        allocator.Free( index );
    }

    // slotCount を超える値でアロケートする。
    // 失敗して InvalidIndex が返るはず。
    {
        int index = allocator.Allocate( slotCount + 1 );

        // InvalidIndex が返っているか
        EXPECT_EQ(index, InvalidIndex);

        // Free(InvalidIndex) は有効
        allocator.Free( index );
    }

    // １スロット分を slotCount 回アロケートする。
    // すべて成功するはず。
    for(int i=0; i<slotCount; i++)
    {
        int index = allocator.Allocate( 1 );

        // 戻ってきたインデックスは範囲内か
        EXPECT_GE(index, baseSlotIndex);
        EXPECT_LT(index, baseSlotIndex + slotCount);
    }

    // Free() せずにさらにもう一回アロケートする。スロットは尽きているので
    // 失敗して InvalidIndex が返るはず。
    {
        int index = allocator.Allocate( 1 );

        // InvalidIndex が返っているか
        EXPECT_EQ(index, InvalidIndex);
    }

    // 終了する
    allocator.Finalize();

    // Finalize() 後は未初期化状態か
    EXPECT_FALSE(allocator.IsInitialized());

    // ユーザーアロケータの Malloc(), Free() の回数が一致しているか
    EXPECT_EQ(g_UserAllocatorAllocCount, 0);


    // ユーザーアロケータの Malloc() が nullptr を返す場合のテスト。
    // Allocate() 時に確実にユーザーアロケータが呼ばれるように、
    // 再初期化して初回の Allocate() でテストする。(ついでに２回目の Initialize, Finalize のテストも行う)
    // 失敗して InvalidIndex が返るはず。
    {
        // 初期化する
        allocator.Initialize(
            UserAllocatorMalloc,
            g_pExpectedAllocateUserData,
            UserAllocatorFree,
            g_pExpectedFreeUserData,
            &descriptorPool,
            baseSlotIndex,
            slotCount,
            false
            );

        // Initialize() 後は初期化済み状態か
        EXPECT_TRUE(allocator.IsInitialized());

        g_bUserAllocatorInvalid = true;

        int index = allocator.Allocate( 1 );

        // InvalidIndex が返っているか
        EXPECT_EQ(index, InvalidIndex);

        g_bUserAllocatorInvalid = false;

        // 終了する
        allocator.Finalize();

        // Finalize() 後は未初期化状態か
        EXPECT_FALSE(allocator.IsInitialized());
    }
}

TEST(DescriptorPoolAllocatorDeathTest, DeathTest)
{
    const int baseSlotIndex = 100;
    const int slotCount = 100;
    const int SlotCountMax = nn::gfx::util::DescriptorPoolAllocator::SlotCountMax;
    nn::gfx::DescriptorPool descriptorPool;
    nn::gfx::util::DescriptorPoolAllocator allocator;

    g_bUserAllocatorInvalid = false;
    g_UserAllocatorAllocCount = 0;
    g_pExpectedAllocateUserData = nullptr;
    g_pExpectedFreeUserData = nullptr;

    // 初期化前に呼ぶと死ぬことの確認
    EXPECT_DEATH_IF_SUPPORTED(allocator.Finalize(), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Allocate(1), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Free(baseSlotIndex), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.GetDescriptorPool(), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.GetBaseSlotIndex(), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.GetSlotCount(), "");

    // Initialize のパラメータで死ぬことの確認
    EXPECT_DEATH_IF_SUPPORTED(allocator.Initialize(nullptr, nullptr, UserAllocatorFree, nullptr, &descriptorPool, baseSlotIndex, slotCount, false), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Initialize(UserAllocatorMalloc, nullptr, nullptr, nullptr, &descriptorPool, baseSlotIndex, slotCount, false), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Initialize(UserAllocatorMalloc, nullptr, UserAllocatorFree, nullptr, nullptr, baseSlotIndex, slotCount, false), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Initialize(UserAllocatorMalloc, nullptr, UserAllocatorFree, nullptr, &descriptorPool, -1, slotCount, false), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Initialize(UserAllocatorMalloc, nullptr, UserAllocatorFree, nullptr, &descriptorPool, baseSlotIndex, 0, false), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Initialize(UserAllocatorMalloc, nullptr, UserAllocatorFree, nullptr, &descriptorPool, baseSlotIndex, SlotCountMax + 1, false), "");

    // 初期化する
    allocator.Initialize(
        UserAllocatorMalloc,
        nullptr,
        UserAllocatorFree,
        nullptr,
        &descriptorPool,
        baseSlotIndex,
        slotCount,
        false
        );

    // 初期化後に呼ぶと死ぬことの確認
    EXPECT_DEATH_IF_SUPPORTED(allocator.Initialize(UserAllocatorMalloc, nullptr, UserAllocatorFree, nullptr, &descriptorPool, baseSlotIndex, slotCount, false), "");

    // 不正なパラメータで呼ぶと死ぬことの確認
    EXPECT_DEATH_IF_SUPPORTED(allocator.Allocate(-1), "");
    int index = allocator.Allocate(10);
    EXPECT_DEATH_IF_SUPPORTED(allocator.Free(index - 1), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Free(index + 1), "");
    // 二重解放で死ぬことの確認
    allocator.Free(index);
    EXPECT_DEATH_IF_SUPPORTED(allocator.Free(index), "");

    // 終了する
    allocator.Finalize();
}
